use rocksdb::{DB, Writable, SeekKey, DBIterator, Kv};
use tempdir::TempDir;

fn prev_collect<'a>(mut iter: DBIterator<'a>) -> Vec<Kv> {
    let mut buf = vec![];
    while iter.valid() {
        buf.push(iter.kv().unwrap());
        iter.prev();
    }
    buf
}

#[test]
pub fn test_iterator() {
    let path = TempDir::new("_rust_rocksdb_iteratortest").expect("");

    let k1 = b"k1";
    let k2 = b"k2";
    let k3 = b"k3";
    let k4 = b"k4";
    let v1 = b"v1111";
    let v2 = b"v2222";
    let v3 = b"v3333";
    let v4 = b"v4444";
    let db = DB::open_default(path.path().to_str().unwrap()).unwrap();
    let p = db.put(k1, v1);
    assert!(p.is_ok());
    let p = db.put(k2, v2);
    assert!(p.is_ok());
    let p = db.put(k3, v3);
    assert!(p.is_ok());
    let expected = vec![(k1.to_vec(), v1.to_vec()),
                        (k2.to_vec(), v2.to_vec()),
                        (k3.to_vec(), v3.to_vec())];

    let mut iter = db.iter(SeekKey::Start);
    assert_eq!(iter.collect::<Vec<_>>(), expected);

    // Test that it's idempotent
    iter = db.iter(SeekKey::Start);
    assert_eq!(iter.collect::<Vec<_>>(), expected);

    // Test it in reverse a few times
    iter = db.iter(SeekKey::End);
    let mut tmp_vec = prev_collect(iter);
    tmp_vec.reverse();
    assert_eq!(tmp_vec, expected);

    iter = db.iter(SeekKey::End);
    let mut tmp_vec = prev_collect(iter);
    tmp_vec.reverse();
    assert_eq!(tmp_vec, expected);

    // Try it forward again
    iter = db.iter(SeekKey::Start);
    assert_eq!(iter.collect::<Vec<_>>(), expected);

    iter = db.iter(SeekKey::Start);
    assert_eq!(iter.collect::<Vec<_>>(), expected);

    let old_iterator = db.iter(SeekKey::Start);
    let p = db.put(&*k4, &*v4);
    assert!(p.is_ok());
    let expected2 = vec![(k1.to_vec(), v1.to_vec()),
                         (k2.to_vec(), v2.to_vec()),
                         (k3.to_vec(), v3.to_vec()),
                         (k4.to_vec(), v4.to_vec())];
    assert_eq!(old_iterator.collect::<Vec<_>>(), expected);

    iter = db.iter(SeekKey::Start);
    assert_eq!(iter.collect::<Vec<_>>(), expected2);

    iter = db.iter(SeekKey::Key(k2));
    let expected = vec![(k2.to_vec(), v2.to_vec()),
                        (k3.to_vec(), v3.to_vec()),
                        (k4.to_vec(), v4.to_vec())];
    assert_eq!(iter.collect::<Vec<_>>(), expected);

    iter = db.iter(SeekKey::Key(k2));
    let expected = vec![(k2.to_vec(), v2.to_vec()), (k1.to_vec(), v1.to_vec())];
    assert_eq!(prev_collect(iter), expected);

    iter = db.iter(SeekKey::Key(b"k0"));
    assert!(iter.valid());
    iter.seek(SeekKey::Key(b"k1"));
    assert!(iter.valid());
    iter.seek(SeekKey::Key(b"k11"));
    assert!(iter.valid());
    iter.seek(SeekKey::Key(b"k5"));
    assert!(!iter.valid());
    iter.seek(SeekKey::Key(b"k0"));
    assert!(iter.valid());
    iter.seek(SeekKey::Key(b"k1"));
    assert!(iter.valid());
    iter.seek(SeekKey::Key(b"k11"));
    assert!(iter.valid());
    iter.seek(SeekKey::Key(b"k5"));
    assert!(!iter.valid());

    iter.seek(SeekKey::Key(b"k4"));
    assert!(iter.valid());
    iter.prev();
    assert!(iter.valid());
    iter.next();
    assert!(iter.valid());
    iter.next();
    assert!(!iter.valid());
    // Once iterator is invalid, it can't be reverted.
    iter.prev();
    assert!(!iter.valid());
}
