// Copyright 2017 PingCAP, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// See the License for the specific language governing permissions and
// limitations under the License.

use librocksdb_sys;
use rocksdb_options::{Options, EnvOptions};

use std::ffi::CString;

/// SstFileWriter is used to create sst files that can be added to database later
/// All keys in files generated by SstFileWriter will have sequence number = 0
pub struct SstFileWriter {
    inner: *mut librocksdb_sys::SstFileWriter,
}

unsafe impl Send for SstFileWriter {}

impl SstFileWriter {
    pub fn new(env_opt: &EnvOptions, opt: &Options) -> SstFileWriter {
        unsafe {
            SstFileWriter {
                inner: librocksdb_sys::crocksdb_sstfilewriter_create(env_opt.inner, opt.inner),
            }
        }
    }

    /// Prepare SstFileWriter to write into file located at "file_path".
    pub fn open(&mut self, name: &str) -> Result<(), String> {
        let path = match CString::new(name.to_owned()) {
            Err(e) => return Err(format!("invalid path {}: {:?}", name, e)),
            Ok(p) => p,
        };
        unsafe { Ok(ffi_try!(crocksdb_sstfilewriter_open(self.inner, path.as_ptr()))) }
    }

    /// Add key, value to currently opened file
    /// REQUIRES: key is after any previously added key according to comparator.
    pub fn add(&mut self, key: &[u8], val: &[u8]) -> Result<(), String> {
        unsafe {
            ffi_try!(crocksdb_sstfilewriter_add(self.inner,
                                                key.as_ptr(),
                                                key.len(),
                                                val.as_ptr(),
                                                val.len()));
            Ok(())
        }
    }

    /// Finalize writing to sst file and close file.
    pub fn finish(&mut self) -> Result<(), String> {
        unsafe {
            ffi_try!(crocksdb_sstfilewriter_finish(self.inner));
            Ok(())
        }
    }
}

impl Drop for SstFileWriter {
    fn drop(&mut self) {
        unsafe { librocksdb_sys::crocksdb_sstfilewriter_destroy(self.inner) }
    }
}
