#pragma once

#include "blob_file_builder.h"
#include "blob_file_manager.h"
#include "blob_file_set.h"
#include "rocksdb/types.h"
#include "table/table_builder.h"
#include "titan/options.h"
#include "titan_stats.h"

namespace rocksdb {
namespace titandb {

class TitanTableBuilder : public TableBuilder {
 public:
  TitanTableBuilder(uint32_t cf_id, const TitanDBOptions& db_options,
                    const TitanCFOptions& cf_options,
                    std::unique_ptr<TableBuilder> base_builder,
                    std::shared_ptr<BlobFileManager> blob_manager,
                    std::weak_ptr<BlobStorage> blob_storage, TitanStats* stats,
                    int merge_level, int target_level)
      : cf_id_(cf_id),
        db_options_(db_options),
        cf_options_(cf_options),
        base_builder_(std::move(base_builder)),
        blob_manager_(blob_manager),
        blob_storage_(blob_storage),
        stats_(stats),
        target_level_(target_level),
        merge_level_(merge_level) {}

  void Add(const Slice& key, const Slice& value) override;

  Status status() const override;

  Status Finish() override;

  void Abandon() override;

  uint64_t NumEntries() const override;

  uint64_t FileSize() const override;

  bool NeedCompact() const override;

  TableProperties GetTableProperties() const override;

 private:
  friend class TableBuilderTest;

  bool ok() const { return status().ok(); }

  bool builder_unbuffered() const {
    return !blob_builder_ || blob_builder_->GetBuilderState() ==
                                 BlobFileBuilder::BuilderState::kUnbuffered;
  }

  std::unique_ptr<BlobFileBuilder::BlobRecordContext> NewCachedRecordContext(
      const ParsedInternalKey& ikey, const Slice& value);

  void AddBlob(const ParsedInternalKey& ikey, const Slice& value);

  void AddToBaseTable(const BlobFileBuilder::OutContexts& contexts);

  bool ShouldMerge(const std::shared_ptr<BlobFileMeta>& file);

  void FinishBlobFile();

  void UpdateInternalOpStats();

  Status GetBlobRecord(const BlobIndex& index, BlobRecord* record,
                       PinnableSlice* buffer);

  Status status_;
  uint32_t cf_id_;
  TitanDBOptions db_options_;
  TitanCFOptions cf_options_;
  std::unique_ptr<TableBuilder> base_builder_;
  std::unique_ptr<BlobFileHandle> blob_handle_;
  std::shared_ptr<BlobFileManager> blob_manager_;
  std::unique_ptr<BlobFileBuilder> blob_builder_;
  std::weak_ptr<BlobStorage> blob_storage_;
  std::vector<
      std::pair<std::shared_ptr<BlobFileMeta>, std::unique_ptr<BlobFileHandle>>>
      finished_blobs_;
  std::unordered_map<uint64_t, std::unique_ptr<BlobFilePrefetcher>>
      input_file_prefetchers_;
  TitanStats* stats_;

  // target level in LSM-Tree for generated SSTs and blob files
  int target_level_;
  // with cf_options_.level_merge == true, if target_level_ is higher than or
  // equals to merge_level_, values belong to blob files which have lower level
  // than target_level_ will be merged to new blob file
  int merge_level_;

  // counters
  uint64_t bytes_read_ = 0;
  uint64_t bytes_written_ = 0;
  uint64_t io_bytes_read_ = 0;
  uint64_t io_bytes_written_ = 0;
  uint64_t error_read_cnt_ = 0;
};

}  // namespace titandb
}  // namespace rocksdb
