package com.cloudam.cromwell.core;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.cloudam.cromwell.CloudamRequest;
import com.cloudam.cromwell.constant.CloudamURL;
import com.cloudam.cromwell.domain.JobDescription;
import com.cloudam.cromwell.domain.Os;
import okhttp3.MediaType;
import okhttp3.Request;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Base64;
import java.util.function.BiConsumer;

public class CreateJobRequest extends CloudamRequest<CreateJobResponse> {

    private final String urlSuffix = CloudamURL.CREATE_JOB_URL_SUFFIX;

    private final Logger logger = LoggerFactory.getLogger(CreateJobRequest.class);

    private JobDescription jobDescription;

    public void setJobDescription(JobDescription jobDescription) {
        this.jobDescription = jobDescription;
    }

    public JobDescription getJobDescription() {
        return jobDescription;
    }

    @Override
    public Class<CreateJobResponse> getResponseClass() {
        return CreateJobResponse.class;
    }

    @Override
    public Request.Builder getBuilder(Request.Builder builder, String url, CloudamRequest<CreateJobResponse> cloudamRequest) {
        CreateJobRequest createJobRequest = (CreateJobRequest) cloudamRequest;
        return builder
                .url(url + createJobRequest.urlSuffix)
                .post(okhttp3.RequestBody.create(MediaType.parse("charset=utf-8"), JSON.toJSONBytes(buildJson(createJobRequest.getJobDescription()))));
    }

    private JSONObject buildJson(JobDescription jobDescription) {
        JSONObject jsonObject = new JSONObject();

        logger.info("createJobRequest JobDescription:{}", JSON.toJSONString(jobDescription));

        jsonObject.put("envs", jobDescription.getEnvs());
        jsonObject.put("jobName", jobDescription.getName());
        jsonObject.put("projectId", jobDescription.getProjectId());
        jsonObject.put("zoneId", jobDescription.getZoneId());
        jsonObject.put("os", jobDescription.getAutoCluster().getOs());
        jsonObject.put("slurmCommand", Base64.getEncoder().encodeToString(buildSbatchSubmit(jobDescription).getBytes()));
        jsonObject.put("workflowJobId", jobDescription.getWorkflowJobId());
        jsonObject.put("username", jobDescription.getUsername());
        return jsonObject;
    }

    private String buildSbatchSubmit(JobDescription jobDescription) {
        // sbatch -n 1 -c 8 --comment disk:CLOUD_SSD:2700G:s-NV0tQ0PL run_local.sh
        // sbatch -n 1 -c 8 -p c-52-2 -q eco cpu.sh
        StringBuffer commandBuffer = new StringBuffer("#!/bin/sh\n");
        BiConsumer<String, String> stringValidAppend = (key, args) -> {
            if (StringUtils.isNotBlank(key)) {
                commandBuffer.append("#SBATCH ").append(args).append(" ").append(key).append("\n");
            }
        };
        BiConsumer<Integer, String> intValidAppend = (key, args) -> {
            if (key != null && key != 0) {
                commandBuffer.append("#SBATCH ").append(args).append(" ").append(key).append("\n");
            }
        };
        stringValidAppend.accept(jobDescription.getName(), "-J");
        stringValidAppend.accept(jobDescription.getAutoCluster().getPartition(), "-p");

        intValidAppend.accept(jobDescription.getCpus(), "-c");
        intValidAppend.accept(jobDescription.getGpus(), "-g");
        intValidAppend.accept(jobDescription.getNtasks(), "-n");
        intValidAppend.accept(jobDescription.getTimeout(), "-t");

        commandBuffer.append(jobDescription.getWorkScript()).append("\n");
        return commandBuffer.toString();
    }
}
