/* eslint global-require: off, no-console: off, promise/always-return: off */

/**
 * This module executes inside of electron's main process. You can start
 * electron renderer process from here and communicate with the other processes
 * through IPC.
 *
 * When running `npm run build` or `npm run build:main`, this file is compiled to
 * `./src/main.js` using webpack. This gives us some performance wins.
 */
import path from 'path';
import { app, BrowserWindow, shell, ipcMain, Tray, Menu } from 'electron';
import { autoUpdater } from 'electron-updater';
import log from 'electron-log';
import MenuBuilder from './menu';
import { resolveHtmlPath } from './util';
const nativeImage = require('electron').nativeImage;
const positioner = require('electron-traywindow-positioner');

class AppUpdater {
    constructor() {
        log.transports.file.level = 'info';
        autoUpdater.logger = log;
        autoUpdater.checkForUpdatesAndNotify();
    }
}
let tray: Tray | null = null;
let mainWindow: BrowserWindow | null = null;
let siteWindown: BrowserWindow | null = null;
let homeWindown: BrowserWindow | null = null;

const isDev = process.env.NODE_ENV === 'development';
const alignment = { x: 'left', y: 'up' };

const RESOURCES_PATH = app.isPackaged
    ? path.join(process.resourcesPath, 'assets')
    : path.join(__dirname, '../../assets');

const getAssetPath = (...paths: string[]): string => {
    return path.join(RESOURCES_PATH, ...paths);
};

let image = nativeImage.createFromPath(getAssetPath('/img/iconTemplate.png'));

Menu.setApplicationMenu(null);

ipcMain.on('createNewWindow-site', (event, arg) => {
    if (siteWindown) {
        siteWindown.focus();
        return;
    }
    siteWindown = new BrowserWindow({
        width: 620,
        height: 482,
        minWidth: 100,
        minHeight: 100,
        frame: true,
        resizable: false,
        webPreferences: {
            sandbox: false,
            nodeIntegration: true,
            contextIsolation: false,
            webSecurity: false,
        },
        // parent: siteWindown, //win是主窗口
    });
    siteWindown.loadURL(resolveHtmlPath(`index.html#/site`));
    isDev && siteWindown.webContents.openDevTools();
    siteWindown.on('closed', () => {
        siteWindown = null;
    });
});

ipcMain.on('resize-home', (event) => {
    if (!tray) return;
    mainWindow?.close();
    if (homeWindown) {
        homeWindown.focus();
        return;
    }
    homeWindown = new BrowserWindow({
        show: false,
        width: 350,
        height: 425,
        minWidth: 100,
        minHeight: 100,
        frame: false,
        resizable: false,
        minimizable: false,
        maximizable: false,
        webPreferences: {
            sandbox: false,
            nodeIntegration: true,
            contextIsolation: false,
            webSecurity: false,
        },
    });
    homeWindown.loadURL(resolveHtmlPath(`index.html#/home`));
    positioner.position(homeWindown, tray.getBounds(), alignment);
    isDev && homeWindown.webContents.openDevTools();
    // homeWindown.webContents.openDevTools();
    homeWindown.once('ready-to-show', () => {
        homeWindown?.show();
    });

    homeWindown.on('blur', () => {
        !isDev && homeWindown?.hide();
    });

    homeWindown.on('closed', () => {
        homeWindown = null;
    });
});

ipcMain.on('window-close', (event) => {
    app.quit();
});

if (process.env.NODE_ENV === 'production') {
    const sourceMapSupport = require('source-map-support');
    sourceMapSupport.install();
}

const isDebug =
    process.env.NODE_ENV === 'development' || process.env.DEBUG_PROD === 'true';

if (isDebug) {
    require('electron-debug')();
}

const installExtensions = async () => {
    const installer = require('electron-devtools-installer');
    const forceDownload = !!process.env.UPGRADE_EXTENSIONS;
    const extensions = ['REACT_DEVELOPER_TOOLS'];

    return installer
        .default(
            extensions.map((name) => installer[name]),
            forceDownload
        )
        .catch(console.log);
};

const createWindow = async () => {
    if (isDebug) {
        await installExtensions();
    }
    if (homeWindown) return;
    mainWindow = new BrowserWindow({
        show: false,
        width: 480,
        height: 486,
        minWidth: 100,
        minHeight: 100,
        icon: getAssetPath('icon.ico'),
        frame: true,
        resizable: false,
        titleBarStyle: 'hidden',
        titleBarOverlay: true,
        // autoHideMenuBar: true,
        webPreferences: {
            sandbox: false,
            nodeIntegration: true,
            contextIsolation: false,
            webSecurity: false,
        },
    });

    tray = new Tray(await image);
    // tray.setTitle('test');

    mainWindow.loadURL(resolveHtmlPath('index.html'));
    // mainWindow.webContents.openDevTools();
    mainWindow.once('ready-to-show', () => {
        if (!mainWindow) {
            throw new Error('"mainWindow" is not defined');
        }
        if (process.env.START_MINIMIZED) {
            mainWindow.minimize();
        } else {
            mainWindow.show();
            // mainWindow.webContents.openDevTools();
        }
    });

    mainWindow.on('closed', () => {
        mainWindow = null;
    });

    // const menuBuilder = new MenuBuilder(mainWindow);
    // menuBuilder.buildMenu();

    // Open urls in the user's browser
    mainWindow.webContents.setWindowOpenHandler((edata) => {
        shell.openExternal(edata.url);
        return { action: 'deny' };
    });

    // Remove this if your app does not use auto updates
    // eslint-disable-next-line
    new AppUpdater();

    tray.on('click', function () {
        if (mainWindow) {
            mainWindow.show();
            return;
        }
        if (!homeWindown || !tray) return;
        homeWindown.show();
        positioner.position(homeWindown, tray.getBounds(), alignment);
    });
};

/**
 * Add event listeners...
 */

app.on('window-all-closed', () => {
    // Respect the OSX convention of having the application in memory even
    // after all windows have been closed
    if (process.platform !== 'darwin') {
        app.quit();
    }
});

app.whenReady()
    .then(() => {
        createWindow();
        app.on('activate', () => {
            // On macOS it's common to re-create a window in the app when the
            // dock icon is clicked and there are no other windows open.
            if (mainWindow === null) createWindow();
        });
    })
    .catch(console.log);
