import { useState, useEffect } from "react";

import MyDialog from "@/components/mui/MyDialog";
import { checkIsNumberLetterChinese } from "@/utils/util";
import { useMessage } from "@/components/MySnackbar";
import CloudEController from "@/api/fileserver/CloudEController";
import usePass from "@/hooks/usePass";
import InputAdornment from "@mui/material/InputAdornment";
import MyInput from "@/components/mui/MyInput";
import MyButton from "@/components/mui/MyButton";

type IAddFolderProps = {
	list: Array<any>;
	path: string;
	projectId: string;
	fileToken: string;
	refresh: any;
	selectIds: Array<any>;
};

const AddFolder = (props: IAddFolderProps) => {
	const { list, path, projectId, fileToken, refresh, selectIds } = props;
	const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
	const isPass = usePass();
	const Message = useMessage();
	const [fileName, setFileName] = useState("");
	const [fileNameCheck, setFileNameCheck] = useState({
		error: false,
		help: "",
	});

	useEffect(() => {
		if (deleteDialogOpen) {
			setFileName("");
			setFileNameCheck({
				error: false,
				help: "",
			});
		}
	}, [deleteDialogOpen]);

	const handleAddSubmit = () => {
		if (fileName && !fileNameCheck.error) {
			const newFolderPath =
				path === "/" ? `${path}${fileName}` : `${path}/${fileName}`;
			// 请求接口
			CloudEController.JobFileNewFolder(
				newFolderPath,
				fileToken,
				projectId
			)?.then((res) => {
				Message.success("新建成功");
				setDeleteDialogOpen(false);
				refresh();
			});
		} else {
			Message.error(fileNameCheck.help || "请输入文件夹名称");
		}
	};

	const handleFileNameChange = (e: any) => {
		const fileName = e.target.value;
		if (fileName.length <= 30) {
			setFileName(fileName);
		}
		if (!fileName) {
			setFileNameCheck({
				error: true,
				help: "文件夹名称不能为空",
			});
		} else if (!checkIsNumberLetterChinese(fileName) || fileName.length > 30) {
			setFileNameCheck({
				error: true,
				help: "格式不正确，必须在30字符以内，仅限大小写字母、数字、中文",
			});
		} else if (isRepeat(fileName)) {
			setFileNameCheck({
				error: true,
				help: "已存在同名文件夹",
			});
		} else {
			setFileNameCheck({
				error: false,
				help: "",
			});
		}
	};

	const isRepeat = (fileName: string) => {
		return list.some((item: any) => {
			return item.name === fileName;
		});
	};

	return (
		<>
			<MyButton
				text="新建文件夹"
				variant="outlined"
				size="medium"
				onClick={() => setDeleteDialogOpen(true)}
				disabled={
					selectIds.length !== 0 || !isPass("PROJECT_DATA_ADDDIR", "USER")
				}
			/>

			<MyDialog
				open={deleteDialogOpen}
				onClose={() => setDeleteDialogOpen(false)}
				onConfirm={handleAddSubmit}
				title="新建文件夹"
			>
				<div>
					<MyInput
						sx={{
							width: "388px",
						}}
						required
						error={fileNameCheck.error}
						id="fileName"
						label="名称"
						variant="outlined"
						value={fileName}
						onChange={handleFileNameChange}
						helperText={fileNameCheck.help}
						InputProps={{
							endAdornment: (
								<InputAdornment
									position="end"
									style={{
										color: fileName.length >= 30 ? "#d32f2f" : "#C2C6CC",
									}}
								>
									{fileName.length}/30
								</InputAdornment>
							),
						}}
						size="medium"
					/>
				</div>
			</MyDialog>
		</>
	);
};

export default AddFolder;
