import React, { useCallback } from "react";
import { makeStyles } from "tss-react/mui";
import {
	Typography,
	Menu,
	MenuItem,
	Button,
	ThemeProvider,
	createTheme,
	ExtendButtonBase,
	ButtonTypeMap,
} from "@mui/material";
import ArrowDropDownIcon from "@mui/icons-material/ArrowDropDown";

interface ButtonTagProps
	extends Omit<ExtendButtonBase<ButtonTypeMap<{}, "button">>, "text"> {
	startIcon?: React.ReactNode /** 开始Icon */;
	endIcon?: React.ReactNode /** 结束Icon */;
	text: string; //文本内容
	variant?: "text" | "contained" | "outlined"; //按钮样式
	onClick?: any; //点击事件
	select?: any[]; //选择按钮的下拉列表
	fontSize?: string; //按钮文字大小
	dropValue?: boolean; //选择的值
	drop?: boolean; //是否开启选择
	color?:
		| "inherit"
		| "primary"
		| "secondary"
		| "success"
		| "error"
		| "info"
		| "warning"; //按钮颜色风格
	size?: "large" | "medium" | "small"; //按钮尺寸
	disabled?: boolean; //是否禁用
	style?: any; //按钮自定义样式
	img?: JSX.Element; //图标按钮中的图标
	selectCallBack?: (item: any, key: number) => void; //选择按钮的回调
}

const theme = createTheme({
	components: {
		MuiButton: {
			styleOverrides: {
				root: {
					minWidth: "48px",
					"&.MuiButton-textError": {
						color: "rgba(255, 78, 78, 1)",
					},
					"&.MuiButton-outlinedError": {
						color: "rgba(255, 78, 78, 1)",
						border: "1px solid rgba(255, 78, 78, 1)",
					},
				},
				contained: {
					backgroundColor: "#1370FF",
					"&:hover": {
						backgroundColor: "#0459D9",
					},
				},
				outlined: {
					backgroundColor: "#FFFFFF",
					border: "1px solid #1370FF",
					boxShadow: "none !important",
					color: "#1370FF",
					"&:hover": { backgroundColor: "#ECF4FF " },
					"&.MuiButton-textError": {
						color: "rgba(255, 78, 78, 1)",
					},
				},
				text: {
					backgroundColor: "transparent",
					boxShadow: "none !important",
					color: "#1370FF",
					"&:hover": { backgroundColor: "#ECF4FF " },
				},

				containedSecondary: {
					backgroundColor: "#FF4E4E",
					"&:hover": {
						backgroundColor: "#FF4E4E",
					},
				},
				outlinedSecondary: {
					border: "1px solid #FF4E4E",
					boxShadow: "none !important",
					color: "#FF4E4E",
					"&:hover": {
						backgroundColor: "#FFEDED ",
						border: "1px solid #FF4E4E",
					},
				},
				textSecondary: {
					backgroundColor: "transparent",
					boxShadow: "none !important",
					color: "#FF4E4E",
					"&:hover": { backgroundColor: "#FFEDED " },
				},
				sizeSmall: {
					"& p": { fontSize: "12px" },
					height: "28px",
					padding: "0 12px",
				},
				sizeMedium: {
					"& p": { fontSize: "14px" },
					height: "32px",
					padding: "0 16px",
				},
				sizeLarge: {
					"& p": { fontSize: "16px" },
					height: "36px",
					padding: "0 16px",
				},
			},
		},
		MuiPaper: {
			styleOverrides: {
				elevation: {},
			},
		},

		MuiMenuItem: {
			styleOverrides: {
				root: {
					"&:hover": { backgroundColor: "#ECF4FF " },
				},
			},
		},
	},
});

const ButtonComponent = (props: ButtonTagProps) => {
	const {
		size,
		disabled,
		variant,
		color,
		img,
		select,
		selectCallBack,
		...other
	} = props;
	const { classes, cx } = useStyles({});
	const [anchorEl, setAnchorEl] = React.useState(null);

	const handleClick = useCallback(
		(event: { currentTarget: React.SetStateAction<null> }) => {
			setAnchorEl(event.currentTarget);
		},
		[]
	);

	const defaultClick = useCallback((event: { stoppropagation: () => any }) => {
		event && event.stoppropagation && event.stoppropagation();
	}, []);

	const handleCloseOption = useCallback(
		(item: any, key: number) => {
			setAnchorEl(null);
			selectCallBack && selectCallBack(item, key);
		},
		[selectCallBack]
	);

	const handleClose = () => setAnchorEl(null);

	return (
		<ThemeProvider theme={theme}>
			<Button
				size={size || "medium"}
				variant={variant || "contained"}
				color={color || "primary"}
				disableRipple={true}
				disableFocusRipple={true}
				disabled={disabled || false}
				style={{ ...props.style }}
				onClick={props.select ? handleClick : props.onClick || defaultClick}
				{...other}
			>
				{img || ""}
				<Typography style={{ fontSize: props.fontSize }}>
					{props.text}
				</Typography>
				{((props.select && props.select.length > 0) || props.drop) && (
					<ArrowDropDownIcon
						classes={{
							root: cx({
								[classes.ArrowDropDownIconRoot]: true,
								[classes.ArrowDropDownIconRootOpen]: Boolean(
									props.dropValue || anchorEl
								),
							}),
						}}
					/>
				)}
			</Button>
			<Menu
				id="simple-menu"
				anchorEl={anchorEl}
				keepMounted
				open={Boolean(anchorEl)}
				onClose={handleClose}
			>
				{select &&
					select.length > 0 &&
					select.map((item, key) => {
						return (
							<MenuItem
								key={key}
								// classes={{ root: classes.menuItemRoot }}
								onClick={() => handleCloseOption(item, key)}
							>
								{item.name || ""}
							</MenuItem>
						);
					})}
			</Menu>
		</ThemeProvider>
	);
};

const useStyles = makeStyles<{}>()((theme) => ({
	ArrowDropDownIconRoot: {
		color: "#8A9099",
		transition: "all 0.2s !important",
		transform: "rotate(0)",
	},
	ArrowDropDownIconRootOpen: { color: "#8A9099", transform: "rotate(180deg)" },
}));

export default ButtonComponent;
