import { toJS } from "mobx";
import { useNavigate, useLocation } from "react-router-dom";
import { observer } from "mobx-react-lite";
import { useMemo } from "react";

import { formatTime, storageUnitFromB } from "@/utils/util";
import useGlobalStore from "@/hooks/useGlobalStore";
import { IUploadInfo } from "@/store/modules/fileList";
import { useMessage } from "@/components/MySnackbar";
import bkunyunFile from "@/assets/img/bkunyunFile.svg";
import MyProgress from "@/components/mui/MyProgress";

import styles from "./index.module.css";

/*
 * @Author: 吴永生#A02208 yongsheng.wu@wholion.com
 * @Date: 2022-06-11 15:46:42
 * @LastEditors: 吴永生#A02208 yongsheng.wu@wholion.com
 * @LastEditTime: 2022-06-16 02:07:56
 * @FilePath: /bkunyun/src/views/ConsoleLayout/components/FileItem/index.tsx
 * @Description: 这是默认设置,请设置`customMade`, 打开koroFileHeader查看配置 进行设置: https://github.com/OBKoro1/koro1FileHeader/wiki/%E9%85%8D%E7%BD%AE
 */

interface IProps {
	fileItemInfo: IUploadInfo;
}
const FileItem = observer((props: IProps) => {
	const { fileItemInfo } = props;
	const itemInfo = toJS(fileItemInfo)?.info;
	const { statusMsg = "" } = itemInfo || {};
	const uploadInfoStore = toJS(useGlobalStore("fileListStore"));
	const currentProjectStore = toJS(useGlobalStore("currentProjectStore"));
	const Message = useMessage();
	const navigate = useNavigate();
	const location: any = useLocation();

	/** 时间 */
	const TimeText = useMemo(() => {
		const val = itemInfo?.endTime - itemInfo?.startTime;
		return formatTime(val) || "";
	}, [itemInfo?.endTime, itemInfo?.startTime]);

	const text = useMemo(() => {
		let result = "";
		if (statusMsg === "上传失败") {
			result = "重新上传";
		}
		if (statusMsg === "正在上传") {
			if (itemInfo?.isSuspend) {
				result = "重新上传";
			} else {
				result = "暂停";
			}
		}
		if (statusMsg === "上传成功") {
			result = "查看文件";
		}
		return result;
	}, [itemInfo?.isSuspend, statusMsg]);

	/** 操作合集 */
	const onOperation = () => {
		if (text === "暂停") {
			itemInfo?.upload?.abort(true).then(() => {
				Message.info("暂停成功！");
				uploadInfoStore.setUploadInfoList(fileItemInfo?.id, {
					isSuspend: true,
				});
			});
		}
		if (text === "重新上传") {
			itemInfo?.upload?.start();
			uploadInfoStore.setUploadInfoList(fileItemInfo?.id, {
				isSuspend: false,
			});
		}
		if (text === "查看文件") {
			if (
				location?.state?.pathName !== fileItemInfo?.path ||
				location?.pathname !== "/product/cadd/projectData"
			) {
				navigate(`/product/cadd/projectData`, {
					state: { pathName: fileItemInfo?.path },
				});
			}
		}
	};

	const speed = useMemo(() => {
		let val = 0;
		const time = Math.floor((itemInfo?.endTime - itemInfo.startTime) / 1000);
		if (time > 0) {
			val = Math.floor(itemInfo?.bytesUploaded / time);
		}
		return val;
	}, [itemInfo?.bytesUploaded, itemInfo?.endTime, itemInfo.startTime]);
	return (
		<div className={styles.itemBox}>
			<div className={styles.leftBox}>
				<img src={bkunyunFile} alt="" style={{ marginRight: 16 }} />
				<div>
					<div>
						<b className={styles.fileNameBox} title={itemInfo?.name || ""}>
							{itemInfo?.name || ""}
						</b>
						{statusMsg === "上传失败" ? (
							<span
								className={styles.span}
								style={{ marginLeft: 16, color: "#FF4E4E" }}
							>{`(${statusMsg})`}</span>
						) : (
							<span className={styles.span} style={{ marginLeft: 16 }}>
								{TimeText}
							</span>
						)}
					</div>
					{statusMsg !== "上传成功" ? (
						<MyProgress
							sx={{
								width: 300,
								margin: "6px 0",
							}}
							color="error"
							value={itemInfo?.percentage}
						/>
					) : null}
					<div style={{ fontSize: 12 }}>
						{statusMsg === "上传成功" ? (
							<>
								<span style={{ color: "#8A9099" }}>已上传至</span>
								<span style={{ color: "#565C66", marginLeft: 12 }}>
									{`CADD - ${
										currentProjectStore?.currentProjectInfo?.name || ""
									}`}
								</span>
							</>
						) : (
							<div className={styles.speedBox}>
								<span className={styles.span}>{`${storageUnitFromB(
									itemInfo?.bytesUploaded || 0
								)}/${storageUnitFromB(itemInfo?.bytesTotal || 0)}`}</span>
								{statusMsg !== "上传失败" ? (
									<span className={styles.span}>{`${storageUnitFromB(
										speed
									)}/s`}</span>
								) : null}
							</div>
						)}
					</div>
				</div>
			</div>
			<div className={styles.rightBox} onClick={onOperation}>
				{text}
			</div>
		</div>
	);
});

export default FileItem;
