// 多选下拉框, 是input还是text还是其他由children决定

import * as React from "react";
import Menu from "@mui/material/Menu";
import MenuItem from "@mui/material/MenuItem";
import Checkbox from "@mui/material/Checkbox";
import { ThemeProvider, createTheme } from "@mui/material/styles";
import ArrowDropDownIcon from "@mui/icons-material/ArrowDropDown";
import ArrowDropUpIcon from "@mui/icons-material/ArrowDropUp";
import _ from "lodash";

const theme = createTheme({
	components: {
		MuiMenu: {
			styleOverrides: {
				root: {
					overflow: "overlay",
				},
			},
		},
		MuiMenuItem: {
			styleOverrides: {
				root: {
					fontSize: "14px",
					lineHeight: "36px",
					display: "flex",
					justifyContent: "flex-start",
					paddingTop: 0,
					paddingBottom: 0,
					paddingRight: "32px",
					paddingLeft: "4px",
					":hover": {
						color: "rgba(30, 38, 51, 1)",
						background: "rgba(240, 242, 245, 1)",
					},
				},
			},
		},
		MuiList: {
			styleOverrides: {
				root: {
					maxHeight: "268px",
				},
			},
		},
		MuiSvgIcon: {
			styleOverrides: {
				root: {
					width: "22px",
					height: "22px",
				},
			},
		},
		MuiButtonBase: {
			styleOverrides: {
				root: {},
			},
		},
		MuiCheckbox: {
			styleOverrides: {
				root: {
					color: "rgba(209, 214, 222, 1)",
					"&.Mui-checked": {
						color: "rgba(19, 112, 255, 1)",
					},
				},
			},
		},
	},
});

export type IOption = {
	label: string;
	value: string;
};

type IMyMultipleMenuProps = {
	children: React.ReactNode;
	options: Array<IOption>; // 选项
	value: Array<string>; // 选中的项values
	setValue?: any; // 设置选中的项values
	showSelectAll?: boolean; // 是否显示全选
	selectAllText?: string; // 全选的文字
	showHiddenIcon?: boolean; // 是否显示 展开收起箭头
	iconColor?: string; // 展开收起箭头的颜色
};

const MyMultipleMenu = (props: IMyMultipleMenuProps) => {
	const {
		children,
		options,
		value,
		setValue,
		showSelectAll = false,
		selectAllText = "全部",
		showHiddenIcon = true,
		iconColor,
	} = props;
	// 下拉框展示的相对位置参考元素
	const [anchorEl, setAnchorEl] = React.useState<null | HTMLElement>(null);
	// 展示/隐藏下拉框切换
	const handleClick = (event: React.MouseEvent<HTMLDivElement>) => {
		setAnchorEl(event.currentTarget);
	};
	// 隐藏下拉框
	const handleClose = (value: string) => {
		setAnchorEl(null);
	};

	// 选择/取消选择
	const handleMenuClick = (clickValue: string) => {
		const resultArr = _.cloneDeep(value);
		const valueIndex = value.indexOf(clickValue);
		if (valueIndex === -1) {
			resultArr.push(clickValue);
			setValue && setValue(resultArr);
		} else {
			resultArr.splice(valueIndex, 1);
			setValue && setValue(resultArr);
		}
	};

	// 全选
	const handleMenuAllClick = () => {
		if (value.length === options.length) {
			setValue([]);
		} else {
			setValue(options.map((option) => option.value));
		}
	};

	// 是否选中
	const getCheckedStatus = (
		checkBoxItemValue: any,
		valueArr: Array<any>
	): boolean => {
		const result = valueArr.indexOf(checkBoxItemValue);
		return result === -1 ? false : true;
	};

	const randerShowHiddenIcon = () => {
		if (Boolean(anchorEl)) {
			return (
				<ArrowDropUpIcon sx={{ color: iconColor || "rgba(19, 112, 255, 1)" }} />
			);
		} else {
			return (
				<ArrowDropDownIcon
					sx={{ color: iconColor || "rgba(19, 112, 255, 1)" }}
				/>
			);
		}
	};

	return (
		<ThemeProvider theme={theme}>
			<div>
				<div
					onClick={handleClick}
					style={{
						display: "flex",
						justifyContent: "flex-start",
						alignItems: "center",
					}}
				>
					{children}
					{showHiddenIcon && randerShowHiddenIcon()}
				</div>
				<Menu
					id="basic-menu"
					anchorEl={anchorEl}
					open={Boolean(anchorEl)}
					onClose={handleClose}
					MenuListProps={{
						"aria-labelledby": "basic-button",
					}}
				>
					{showSelectAll && (
						<MenuItem onClick={() => handleMenuAllClick()} key="indeterminate">
							<Checkbox
								checked={value.length === options.length}
								indeterminate={value.length !== options.length}
							/>
							<span>{selectAllText}</span>
						</MenuItem>
					)}
					{options.map((option, index) => {
						return (
							<MenuItem
								onClick={() => handleMenuClick(option.value)}
								key={index}
							>
								<Checkbox
									checked={getCheckedStatus(option.value, value)}
									value={option.value}
								/>
								<span>{option.label}</span>
							</MenuItem>
						);
					})}
				</Menu>
			</div>
		</ThemeProvider>
	);
};

export default MyMultipleMenu;
