import { useEffect, useState } from "react";
import { observer } from "mobx-react";
import SearchInput from "@/components/BusinessComponents/SearchInput";
import MyButton from "@/components/mui/MyButton";
import MySelect from "@/components/mui/MySelect";
import Add from "@mui/icons-material/Add";
import CardTable from "@/components/CommonComponents/CardTable";
import useMyRequest from "@/hooks/useMyRequest";
import { getWorkflowspecList, deleteWorkflowspec } from "@/api/resourceCenter";
import MyDialog from "@/components/mui/MyDialog";
import templateIcon from "@/assets/resourceCenter/templateIcon.svg";
import { useStores } from "@/store";
import { toJS } from "mobx";
import ProductSelect from "@/components/BusinessComponents/ProductSelect";
import WorkFlowEdit from "@/views/WorkFlowEdit";
import { useMessage } from "@/components/MySnackbar";
import TemplateDetail from "./TemplateDetail";
import style from "./index.module.css";

const UserResourcesTemplate = observer(() => {
	const [title, setTitle] = useState("");
	const Message = useMessage();
	const [showAddTemplate, setShowAddTemplate] = useState(false); // 新增、编辑模板弹窗
	const [showProductSelect, setShowProductSelect] = useState(false);
	const [showDeleteDialog, setShowDeleteDialog] = useState(false);
	const [showDetail, setShowDetail] = useState(false); // 详情
	const [templateId, setTemplateId] = useState("");
	const [product, setProduct] = useState(""); // 搜索列表用
	const [productId, setProductId] = useState(""); // 新增模板用
	const [list, setList] = useState([]);
	const { productListStore } = useStores();
	const getProductName = (productId: string) => {
		let res = "-";
		toJS(productListStore.productList).forEach((item) => {
			if (item.value === productId) {
				res = item.label;
			}
		});
		return res;
	};
	const renderItem = (item: any) => {
		return (
			<div className={style.templateBox}>
				<div className={style.templateTop}>
					<div className={style.templateTopItem}>
						{getProductName(item.productId)}
					</div>
					<div className={style.templateTopLine}></div>
					<div className={style.templateTopItem}>{item.version}</div>
					<div className={style.templateTopLine}></div>
					<div className={style.templateTopItem}>{item.updatedTime}</div>
				</div>
				<div className={style.templateTitleBox}>
					<img src={templateIcon} alt="" />
					<div className={style.templateTitle}>{item.title}</div>
				</div>
				<div className={style.templateDesc}>{item.description}</div>
				<div className={style.templateButtons}>
					<div className={style.templateBL}>
						<span
							className={style.button}
							onClick={() => {
								setTemplateId(item.id);
								setProductId(item.productId);
								setShowAddTemplate(true);
							}}
						>
							升级
						</span>
					</div>
					<div className={style.templateBR}>
						<MyButton
							text="删除"
							variant="outlined"
							style={{
								border: "1px solid rgba(221, 225, 230, 1)",
								color: "rgba(138, 144, 153, 1)",
								marginRight: "12px",
							}}
							onClick={() => {
								setShowDeleteDialog(true);
								setTemplateId(item.id);
							}}
						></MyButton>
						<MyButton
							text="查看"
							variant="outlined"
							onClick={() => {
								setTemplateId(item.id);
								setShowDetail(true);
							}}
						></MyButton>
					</div>
				</div>
			</div>
		);
	};

	// 获取模板列表
	const { run } = useMyRequest(getWorkflowspecList, {
		onSuccess: (result: any) => {
			setList(result.data);
		},
	});

	useEffect(() => {
		if (!showAddTemplate) {
			run({
				productId: product === "all" ? "" : product,
				title,
			});
		}
	}, [run, title, product, showAddTemplate]);

	const { run: deleteWorkflowspecFn } = useMyRequest(deleteWorkflowspec, {
		onSuccess: () => {
			Message.success("删除成功");
			setShowDeleteDialog(false);
			run({
				productId: product === "all" ? "" : product,
				title,
			});
		},
	});

	const deleteConfirm = () => {
		deleteWorkflowspecFn({ id: templateId });
	};

	return (
		<div className={style.template}>
			<div className={style.top}>
				<div className={style.topLeft}>
					<SearchInput
						sx={{ width: 340, marginRight: "16px" }}
						onKeyUp={(e: any) => {
							if (e.keyCode === 13) {
								setTitle(e.target.value);
							}
						}}
					></SearchInput>
					<MySelect
						title="产品类型"
						isTitle={true}
						options={
							[
								{ label: "全部", value: "all" },
								...productListStore?.productList,
							] || []
						}
						value={product}
						onChange={(e: any) => setProduct(e)}
						sx={{ width: "150px", height: "32px" }}
					></MySelect>
				</div>
				<div className={style.topRight}>
					<MyButton
						text="新建自定义模板"
						onClick={() => setShowProductSelect(true)}
						startIcon={<Add />}
					></MyButton>
				</div>
			</div>
			<div className={style.tableBox}>
				<CardTable
					data={list}
					renderItem={renderItem}
					itemMinWidth={377}
				></CardTable>
			</div>
			{showAddTemplate && (
				<WorkFlowEdit
					id={templateId}
					propsProductId={productId}
					onBack={() => {
						setShowAddTemplate(false);
					}}
				></WorkFlowEdit>
			)}
			{showProductSelect && (
				<ProductSelect
					open={showProductSelect}
					setOpen={setShowProductSelect}
					productId={productId}
					setProductId={setProductId}
					onConfirm={() => {
						setTemplateId("");
						setShowAddTemplate(true);
					}}
				></ProductSelect>
			)}
			{showDeleteDialog && (
				<MyDialog
					onClose={() => setShowDeleteDialog(false)}
					onConfirm={() => deleteConfirm()}
					open={showDeleteDialog}
					isText={true}
					title="提示"
				>
					确定要删除这个模板吗？
				</MyDialog>
			)}
			{showDetail && (
				<TemplateDetail
					id={templateId}
					setShowDetail={setShowDetail}
				></TemplateDetail>
			)}
		</div>
	);
});

export default UserResourcesTemplate;
