import React, { useMemo } from "react";

import {
	Dialog,
	DialogActions,
	DialogContent,
	DialogTitle,
	DialogProps,
} from "@mui/material";
import CloseIcon from "@mui/icons-material/Close";
import { ThemeProvider, createTheme } from "@mui/material/styles";

import MyButton from "./MyButton";

export interface IDialogProps extends DialogProps {
	/** 自定义类名 */
	className?: string;
	/** 自定义样式 */
	style?: React.CSSProperties;
	/** 弹窗的标题 */
	title?: string;
	/** 是否显示弹窗 */
	open: boolean;
	isHideHeader?: boolean;
	/** 是否隐藏弹窗底部按钮部分 */
	isHideFooter?: boolean;
	/** 自定义底部按钮 */
	footerRender?: () => React.ReactNode;
	/** 是否显示取消按钮 */
	showCancel?: boolean;
	/** 是否显示确定按钮 */
	showConfirm?: boolean;
	/** 关闭弹窗时的回调函数 */
	onClose?: () => void;
	/** 点击确定按钮时的回调函数 */
	onConfirm?: () => void;
	/** 取消按钮文案 */
	cancelText?: string;
	/** 确认按钮文案 */
	okText?: string;
	/** 是否禁用确认按钮 */
	disabledConfirm?: boolean;
	children: React.ReactNode;
	/** 点击遮罩是否关闭  默认为false*/
	clickMaskClose?: boolean;
	/** 确认按钮样式*/
	okColor?:
		| "inherit"
		| "primary"
		| "secondary"
		| "success"
		| "error"
		| "info"
		| "warning"; //按钮颜色风格
	loading?: boolean; // 确认按钮是否处于loading状态
	isText?: boolean; // 是否文本弹窗
	leftSideOfButtonContent?: React.ReactNode; // 取消确认按钮前的内容
}

const theme = createTheme({
	components: {
		MuiPaper: {
			styleOverrides: {
				root: {
					"&.MuiDialog-paper": {
						boxShadow:
							"0px 12px 15px -7px rgb(0 0 0 / 11%), 0px 24px 38px 3px rgb(0 0 0 / 9%), 0px 9px 46px 8px rgb(0 0 0 / 9%)",
					},
				},
			},
		},
		MuiDialogActions: {
			styleOverrides: {
				root: {
					justifyContent: "space-between",
				},
			},
		},
	},
});

const MyDialog: React.FunctionComponent<IDialogProps> = (props) => {
	const {
		title,
		open,
		style,
		onClose,
		onConfirm,
		isHideFooter,
		isHideHeader,
		children,
		footerRender,
		className,
		showCancel = true,
		/** 是否显示确定按钮 */
		showConfirm = true,
		cancelText,
		okText,
		disabledConfirm,
		clickMaskClose = false,
		loading = false,
		okColor = "primary",
		isText = false,
		leftSideOfButtonContent = null,
		...other
	} = props;

	const handelClose = (
		event: {},
		reason: "backdropClick" | "escapeKeyDown"
	) => {
		if (reason === "backdropClick" && !clickMaskClose) {
			return;
		}
		onClose && onClose();
	};

	const Footer = useMemo(() => {
		if (isHideFooter) return null;
		return footerRender ? (
			footerRender()
		) : (
			<DialogActions style={{ padding: "16px 24px 16px 24px" }}>
				<div>{leftSideOfButtonContent && leftSideOfButtonContent}</div>
				<div>
					{showCancel ? (
						<MyButton
							text={cancelText || "取消"}
							onClick={onClose}
							variant="outlined"
							color="secondary"
						/>
					) : null}
					{showConfirm ? (
						<MyButton
							text={okText || "确定"}
							onClick={onConfirm}
							variant="contained"
							color={okColor}
							disabled={disabledConfirm}
							isLoadingButton={true}
							loading={loading}
							style={{ marginLeft: "12px" }}
						/>
					) : null}
				</div>
			</DialogActions>
		);
	}, [
		disabledConfirm,
		okColor,
		okText,
		onConfirm,
		onClose,
		cancelText,
		showCancel,
		footerRender,
		isHideFooter,
		loading,
		showConfirm,
		leftSideOfButtonContent,
	]);

	const contentPadding = useMemo(() => {
		if (isText) {
			return "8px 24px 24px"; // 文本
		} else {
			return "8px 24px 12px"; // 表单
		}
	}, [isText]);
	return (
		<ThemeProvider theme={theme}>
			<Dialog
				open={open}
				onClose={handelClose}
				style={style}
				className={className}
				aria-labelledby="alert-dialog-title"
				aria-describedby="alert-dialog-description"
				sx={{
					"& .MuiDialog-container": {
						"& .MuiPaper-root": {
							// 设置最大宽度， 实际宽度让子元素撑大
							maxWidth: "1920px",
							borderRadius: "8px",
						},
					},
				}}
				{...other}
			>
				{isHideHeader ? null : (
					<DialogTitle id="alert-dialog-title" sx={{ padding: "20px 24px" }}>
						<div
							style={{
								display: "flex",
								justifyContent: "space-between",
								alignItems: "center",
								fontWeight: 600,
							}}
						>
							<span
								style={{ fontSize: 16, lineHeight: "24px", color: "#1E2633" }}
							>
								{title}
							</span>
							<CloseIcon
								onClick={onClose}
								sx={{
									fontSize: "18px",
									color: "#C2C6CC",
									cursor: "pointer",
									":hover": { background: "#f0f2f5", borderRadius: "2px" },
								}}
							/>
						</div>
					</DialogTitle>
				)}
				<DialogContent style={{ minWidth: 388, padding: contentPadding }}>
					{children}
				</DialogContent>
				{Footer}
			</Dialog>
		</ThemeProvider>
	);
};

export default MyDialog;
