import { useCallback, useMemo } from "react";
import Table from "@mui/material/Table";
import TableBody from "@mui/material/TableBody";
import TableCell from "@mui/material/TableCell";
import TableContainer from "@mui/material/TableContainer";
import TableHead from "@mui/material/TableHead";
import TableRow from "@mui/material/TableRow";
import Paper from "@mui/material/Paper";
import Checkbox from "@mui/material/Checkbox";
import MyPagination from "@/components/mui/MyPagination";
import ascIcon from "@/assets/project/ascIcon.svg";
import descIcon from "@/assets/project/descIcon.svg";
import sortIcon from "@/assets/project/sort.svg";
import CircularProgress from "@mui/material/CircularProgress";
import { createTheme, ThemeProvider } from "@mui/material";
import MyTooltip from "./MyTooltip";
import noFile from "@/assets/project/noFile.svg";

type Order = "ASC" | "DESC"; // 升序为asc，降序为desc。

export type sortState = {
	field: string | null | undefined | ""; // 根据哪个属性来排序
	order: Order | null | undefined | "";
};

interface IMyTableProps {
	rows: Array<any>; // 表格数据
	headCells: Array<any>; // 表头配置
	tableKey?: string; // 表格数据的key
	loading?: boolean; // 是否正在加载数据
	hasCheckbox?: boolean; // 是否有复选框
	selectItems?: Array<any>; // 选中的项
	setSelectItems?: any; // 设置选中的项
	fixedHead?: boolean; // 是否是固定表头
	noDataText?: string; // 无数据提示文案
	hasTableFooter?: boolean; // 是否有分页组件
	page?: number; // 当前页
	pageChange?: any; // 页码改变
	count?: number; // 总页数
	totalElements?: number; // 数据总量 不止是列表渲染的长度
	sortState?: sortState; // 排序状态
	setSortState?: any; // 设置排序状态
	paginationType?: "simple" | "complex"; // 分页组件的类型   simple简洁式  complex复杂、带每页数量切换、总数等
	rowsPerPage?: number; // 每页多少条数据
	handleChangeRowsPerPage?: any; // 每页多少条数据变化
	nodataText?: any; // 无数据文案
	handleRow?: any; // 点击一行
	activeId?: string; // 选中的一行的id
	disableFn?: any; // 禁用时根据disableFn来判断是否禁用
}

const MyTable = (props: IMyTableProps) => {
	const {
		rows,
		headCells,
		tableKey = "id",
		loading = false,
		hasCheckbox = false,
		selectItems = [],
		setSelectItems,
		fixedHead,
		hasTableFooter = false,
		page = 0,
		pageChange = () => {},
		count = 1,
		sortState,
		setSortState,
		paginationType = "simple",
		rowsPerPage = 10,
		handleChangeRowsPerPage,
		totalElements = 0,
		nodataText,
		handleRow,
		activeId,
		disableFn,
	} = props;

	const theme = useMemo(() => {
		return createTheme({
			components: {
				MuiTable: {
					styleOverrides: {
						root: {
							// minHeight: "100px",
						},
					},
				},
				MuiTableHead: {
					styleOverrides: {
						root: {
							background: "rgba(247, 248, 250, 1)",
							"& .MuiTableRow-head": {
								borderRadius: "4px 4px 0 0",
								borderBottom: "1px solid rgba(240, 242, 245, 1)",
							},
						},
					},
				},
				MuiTableRow: {
					styleOverrides: {
						root: {
							opacity: loading ? 0.5 : 1,
							":hover": {
								background: "rgba(245, 246, 247, 1)",
							},
							"& .MuiTableCell-root": {
								"&.MuiTableCell-head": {
									fontSize: "12px",
									lineHeight: "20px",
									color: "rgba(138, 144, 153, 1)",
									padding: "12px 16px",
									whiteSpace: "nowrap",
								},
								"&.MuiTableCell-head:nth-of-type(1)": {
									paddingRight: hasCheckbox ? "8px" : "16px",
								},
							},
							"&.MuiTableRow-footer": {
								// 分页的那一行不要hover效果
								":hover": {
									background: "#fff",
								},
							},
						},
					},
				},
				MuiPaper: {
					styleOverrides: {
						root: {
							// minHeight: "200px",
							position: "relative",
							"&.MuiTableContainer-root": {
								boxShadow: "none",
							},
						},
					},
				},
				MuiTableCell: {
					styleOverrides: {
						root: {
							boxSizing: "border-box",
							fontSize: "14px",
							lineHeight: "22px",
							color: "rgba(30, 38, 51, 1)",
							padding: "16px",
							borderBottom: "1px solid rgba(240, 242, 245, 1)",
						},
					},
				},
				MuiTableBody: {
					styleOverrides: {
						root: {
							minHeight: "200px",
							position: "relative",
							"& .MuiTableRow-root:nth-last-of-type(1) .MuiTableCell-root": {
								// 有分页的话 表格最后一行就不要下边框
								borderBottom: hasTableFooter
									? "none"
									: "1px solid rgba(240, 242, 245, 1)",
							},
						},
					},
				},

				// 复选框样式
				MuiSvgIcon: {
					styleOverrides: {
						root: {
							color: "rgba(209, 214, 222, 1)",
							fontSize: "18px",
						},
					},
				},
				MuiCheckbox: {
					styleOverrides: {
						root: {
							color: "rgba(209, 214, 222, 1)",
							"&.MuiCheckbox-indeterminate .MuiSvgIcon-root": {
								color: "rgba(19, 112, 255, 1)",
							},
							"&.Mui-checked .MuiSvgIcon-root": {
								color: "rgba(19, 112, 255, 1)",
							},
						},
					},
				},
				MuiButtonBase: {
					styleOverrides: {
						root: {
							"&.MuiCheckbox-root": {
								padding: 0,
							},
						},
					},
				},

				MuiCircularProgress: {
					styleOverrides: {
						root: {
							position: "absolute",
							top: 0,
							bottom: 0,
							left: 0,
							right: 0,
							margin: "auto",
							zIndex: 1,
						},
					},
				},
			},
		});
	}, [hasCheckbox, hasTableFooter, loading]);

	const onSelectAllClick = useCallback(
		(e: any) => {
			if (e.target.checked) {
				setSelectItems && setSelectItems(rows.map((row) => row[tableKey]));
			} else {
				setSelectItems && setSelectItems([]);
			}
		},
		[setSelectItems, tableKey, rows]
	);

	const onSelectRowClick = useCallback(
		(e: any, itemValue: string) => {
			if (e.target.checked) {
				setSelectItems && setSelectItems([...selectItems, itemValue]);
			} else {
				const selectItemIndex = selectItems.indexOf(itemValue);
				const newSelectItems = [
					...selectItems.slice(0, selectItemIndex),
					...selectItems.slice(selectItemIndex + 1, selectItems.length),
				];
				setSelectItems && setSelectItems(newSelectItems);
			}
		},
		[selectItems, setSelectItems]
	);

	const handleRowFn = useCallback(
		(row: any) => {
			if (!disableFn) {
				handleRow && handleRow(row);
			} else {
				!disableFn(row) && handleRow && handleRow(row);
			}
		},
		[disableFn, handleRow]
	);

	const handleSort = useCallback(
		(field: string) => {
			if (sortState?.field === field) {
				if (sortState?.order === "ASC") {
					setSortState({
						field,
						order: "DESC",
					});
				} else if (sortState?.order === "DESC") {
					setSortState({
						field,
						order: "ASC",
					});
				} else {
					setSortState({
						field,
						order: "DESC",
					});
				}
			} else {
				setSortState({
					field,
					order: "DESC",
				});
			}
		},
		[sortState, setSortState]
	);

	const randerTableHead = useMemo(() => {
		return (
			<TableHead>
				<TableRow>
					{hasCheckbox && (
						<TableCell align="left" sx={{ width: "18px", paddingRight: "8px" }}>
							<Checkbox
								indeterminate={
									selectItems.length > 0 && selectItems.length < rows.length
								}
								checked={rows.length > 0 && selectItems.length === rows.length}
								onChange={(e) => onSelectAllClick(e)}
							/>
						</TableCell>
					)}
					{headCells.map((headCell, index) => (
						<TableCell key={index} align="left" width={headCell.width}>
							{headCell.sort && (
								<>
									<span>{headCell.label}</span>
									<img
										src={
											sortState?.field === headCell.id
												? sortState?.order === "ASC"
													? ascIcon
													: sortState?.order === "DESC"
													? descIcon
													: sortIcon
												: sortIcon
										}
										alt=""
										onClick={() => handleSort(headCell.id)}
										style={{
											marginLeft: "8px",
											cursor: "pointer",
											position: "relative",
											top: "3px",
										}}
									/>
								</>
							)}
							{!headCell.sort && headCell.label}
						</TableCell>
					))}
				</TableRow>
			</TableHead>
		);
	}, [
		hasCheckbox,
		headCells,
		selectItems,
		onSelectAllClick,
		rows.length,
		handleSort,
		sortState,
	]);

	const randerTableBody = useMemo(() => {
		if (rows.length === 0) {
			if (loading) {
				return (
					<tbody>
						<tr>
							<td></td>
						</tr>
					</tbody>
				);
			}
			return (
				<tbody>
					<tr>
						<td>
							<img
								src={noFile}
								alt=""
								style={{
									position: "absolute",
									top: "50%",
									left: "50%",
									transform: "translate(-50%, -100%)",
								}}
							/>
							<span
								style={{
									position: "absolute",
									top: "50%",
									left: "50%",
									transform: "translate(-50%, 8px)",
									fontSize: "14px",
									lineHeight: "22px",
									color: "#8a9099",
								}}
							>
								{nodataText || "暂无数据"}
							</span>
						</td>
					</tr>
				</tbody>
			);
		}
		return (
			<TableBody>
				{rows.map((row, rowIndex) => (
					<TableRow
						key={row[tableKey] || rowIndex}
						// onClick={() => handleRow && handleRow(row)}
						onClick={() => handleRowFn(row)}
						sx={{
							background:
								activeId === row[tableKey] ? "rgba(237, 244, 255, 1)" : "",
							cursor: disableFn && disableFn(row) ? "no-drop" : "",
							opacity: disableFn && disableFn(row) ? "0.3" : "",
						}}
					>
						{hasCheckbox && (
							<TableCell
								align="left"
								sx={{ width: "18px", paddingRight: "8px" }}
							>
								<Checkbox
									checked={
										selectItems.filter(
											(selectItem) => selectItem === row[tableKey]
										).length > 0
									}
									onChange={(e) => onSelectRowClick(e, row[tableKey])}
								/>
							</TableCell>
						)}
						{headCells.map((headCell, index) => (
							<TableCell key={index} align="left" width={headCell.width}>
								{headCell.showOverflowTooltip && (
									<MyTooltip title={row[headCell.id]}>
										<div
											style={{
												width: Number(headCell.width) - 32 || "",
												overflow: "hidden",
												textOverflow: "ellipsis",
												whiteSpace: "nowrap",
											}}
										>
											{row[headCell.id]}
										</div>
									</MyTooltip>
								)}
								{!headCell.showOverflowTooltip && (
									<div
										style={{
											width: headCell.width ? Number(headCell.width) - 32 : "",
										}}
									>
										{row[headCell.id]}
									</div>
								)}
							</TableCell>
						))}
					</TableRow>
				))}
			</TableBody>
		);
	}, [
		rows,
		tableKey,
		hasCheckbox,
		selectItems,
		onSelectRowClick,
		headCells,
		nodataText,
		loading,
		handleRowFn,
		activeId,
		disableFn,
	]);

	const randerTableFooter = useMemo(() => {
		if (hasTableFooter) {
			return (
				<div
					style={{
						padding: "20px 0",
						display: "flex",
						justifyContent: "flex-end",
						alignItems: "center",
						borderTop: "1px solid rgba(240, 242, 245, 1)",
					}}
				>
					<MyPagination
						page={page}
						pageChange={pageChange}
						count={count}
						totalElements={totalElements}
						type={paginationType}
						rowsPerPage={rowsPerPage}
						handleChangeRowsPerPage={handleChangeRowsPerPage}
					/>
				</div>
			);
		} else {
			return null;
		}
	}, [
		hasTableFooter,
		page,
		count,
		pageChange,
		paginationType,
		rowsPerPage,
		handleChangeRowsPerPage,
		totalElements,
	]);

	const randerCircularProgress = useMemo(() => {
		if (loading) {
			return <CircularProgress></CircularProgress>;
		} else {
			return null;
		}
	}, [loading]);

	return (
		<ThemeProvider theme={theme}>
			{!fixedHead && (
				<>
					<TableContainer component={Paper} sx={{ minHeight: "200px" }}>
						{randerCircularProgress}
						<Table>
							{randerTableHead}
							{randerTableBody}
						</Table>
					</TableContainer>
					{randerTableFooter}
				</>
			)}
			{fixedHead && (
				<div
					style={{
						height: "100%",
						display: "flex",
						flexDirection: "column",
						position: "relative",
					}}
				>
					<TableContainer
						component={Paper}
						sx={{
							minHeight: "45px",
							overflowY: "scroll",
							background: "rgba(247, 248, 250, 1)",
						}}
					>
						<Table sx={{}}>{randerTableHead}</Table>
					</TableContainer>
					<TableContainer
						component={Paper}
						sx={{ flex: 1, minHeight: "200px", overflowY: "scroll" }}
					>
						<Table>{randerTableBody}</Table>
					</TableContainer>
					{randerTableFooter}
					{randerCircularProgress}
				</div>
			)}
		</ThemeProvider>
	);
};

export default MyTable;
