import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import MyCircularProgress from "@/components/mui/MyCircularProgress";
import style from "./index.module.css";

interface ICardTableProps {
	data: Array<any>; // 列表数据
	renderItem: any; // 单个卡片的渲染函数
	itemMinWidth?: number; // 单个卡片的最小宽度，有这个参数时numberOfColumns参数失效，效果为根据屏幕大小和单个卡片的最小宽度来适配每行渲染个数
	tableKey?: string; // 表格数据的key
	numberOfColumns?: number; // 列数   每行渲染几个
	horizontalSpacing?: number; // 水平方向的间隔
	verticalSpacing?: number; // 垂直方向的间隔
	renderBefore?: any;
	loading?: boolean;
}

const CardTable = (props: ICardTableProps) => {
	const {
		data=[],
		renderItem,
		tableKey = "id",
		numberOfColumns: propsNumberOfColumns = 3,
		horizontalSpacing = 20,
		verticalSpacing = 20,
		itemMinWidth,
		renderBefore,
		loading = false,
	} = props;

	const [numberOfColumns, setNumberOfColumns] = useState(3);
	const tableBoxRef: any = useRef(null);

	const getNumberOfColumns = useCallback(() => {
		if (itemMinWidth) {
			const boxWidth = tableBoxRef?.current?.offsetWidth;
			if (boxWidth) {
				setNumberOfColumns(Math.floor(boxWidth / itemMinWidth));
			} else {
				setNumberOfColumns(propsNumberOfColumns);
			}
		} else {
			setNumberOfColumns(propsNumberOfColumns);
		}
	}, [itemMinWidth, propsNumberOfColumns]);

	useEffect(() => {
		getNumberOfColumns();
	}, [getNumberOfColumns]);

	const boxWidth = useMemo(() => {
		return `${100 / numberOfColumns}%`;
	}, [numberOfColumns]);

	window.onresize = () => {
		getNumberOfColumns();
	};

	return (
		<div
			className={style.tableBox}
			style={{
				marginLeft: `-${horizontalSpacing / 2}px`,
				marginRight: `-${horizontalSpacing / 2}px`,
			}}
			ref={tableBoxRef}
		>
			<MyCircularProgress loading={loading} />
			{renderBefore && renderBefore() && (
				<div
					className={style.itemBox}
					style={{
						width: boxWidth,
						paddingLeft: `${horizontalSpacing / 2}px`,
						paddingRight: `${horizontalSpacing / 2}px`,
						paddingBottom: `${verticalSpacing}px`,
						boxSizing: "border-box",
					}}
				>
					{renderBefore()}
				</div>
			)}
			{data?.map((item, index) => {
				return (
					<div
						className={style.itemBox}
						key={item[tableKey] ? item[tableKey] : index}
						style={{
							width: boxWidth,
							paddingLeft: `${horizontalSpacing / 2}px`,
							paddingRight: `${horizontalSpacing / 2}px`,
							paddingBottom: `${verticalSpacing}px`,
							boxSizing: "border-box",
						}}
					>
						{renderItem(item, index)}
					</div>
				);
			})}
		</div>
	);
};

export default CardTable;
