import { useMemo, useCallback, useState } from "react";
import { useNavigate } from "react-router-dom";
import MyProgress from "@/components/mui/MyProgress";
import { getDatasetPath } from "../../ProjectJobDetail/utils";
import style from "./index.module.css";
import { useStores } from "@/store";
import { toJS } from "mobx";
import { observer } from "mobx-react-lite";

import runTime from "../../../../assets/project/runTime.svg";
import jobCostImg from "../../../../assets/project/jobCost.svg";
import jobSue from "../../../../assets/project/jobSue.svg";
import jobFail from "../../../../assets/project/jobFail.svg";
import jobRun from "../../../../assets/project/jobRun.svg";
import jobCadence from "../../../../assets/project/jobCadence.svg";
import fileIcon from "@/assets/project/fileIcon.svg";
import dataSetIcon from "@/assets/project/dataSetIcon.svg";

type TaskCardProps = {
	id: string;
	name: string;
	creator: string;
	state: string;
	completeNum: number;
	totalNum: number;
	costTime: string;
	jobCost: number;
	outputs: Array<any>;
};

const TaskCard = observer((props: TaskCardProps) => {
	const {
		id,
		name,
		creator,
		state,
		completeNum,
		totalNum,
		costTime,
		jobCost,
		outputs,
	} = props;
	const navigate = useNavigate();
	const [displayTitleBlue, setDisplayTitleBlue] = useState<boolean>(false);
	const { currentProjectStore } = useStores();
	const productId = toJS(currentProjectStore.currentProductInfo.id); // 产品ID

	const randerOutputs = useMemo(() => {
		if (outputs) {
			let result = Object.keys(outputs);
			let arr = result.map((item: any) => {
				let type = "file";
				if (outputs[item].indexOf("dataset") !== -1) {
					type = "dataset";
				}
				const nameArr = outputs[item]?.split('/') || []
				return {
					name: nameArr.length ? nameArr[nameArr.length - 1] : '',
					type,
					path: outputs[item],
				};
			});
			return arr;
		} else {
			return [];
		}
	}, [outputs]);
	// 结果文件跳转
	const goToProjectData = (info: any) => {
		let { path = "", type = "" } = info;
		if (type === "file") {
			const lastIndex = path.lastIndexOf("/");
			if (lastIndex !== -1) {
				path = path.slice(0, lastIndex);
			}
			path = path.slice(12);
		} else {
			path = getDatasetPath(path);
		}
		navigate(`/product/${productId || "cadd"}/projectData`, {
			state: { pathName: path, dataType: type },
		});
	};
	// 跳转详情页
	const gotoDetail = (id: string) => {
		navigate(`/product/${productId || "cadd"}/projectJobDetail`, {
			state: { taskId: id, from: "projectOverview" },
		});
	};
	// 渲染状态图标
	const renderStatusIcon = (data: string) => {
		switch (data) {
			case "SUBMITTED":
				return jobRun;
			case "RUNNING":
				return jobRun;
			case "ABORTED":
				return jobCadence;
			case "FAILED":
				return jobFail;
			case "SUCCEEDED":
				return jobSue;
			default:
				return jobCadence;
		}
	};
	// 渲染状态
	const renderStatusText = (data: string) => {
		switch (data) {
			case "SUBMITTED":
				return "正在启动";
			case "RUNNING":
				return "正在运行";
			case "ABORTED":
				return "运行终止";
			case "FAILED":
				return "运行失败";
			case "SUCCEEDED":
				return "运行成功";
			default:
				return "未知";
		}
	};
	// 渲染字体颜色
	const renderTextColor = (data: string) => {
		switch (data) {
			case "SUBMITTED":
				return "#1370FF";
			case "RUNNING":
				return "#1370FF";
			case "ABORTED":
				return "#C2C6CC";
			case "FAILED":
				return "#FF4E4E";
			case "SUCCEEDED":
				return "#0DD09B";
			default:
				return "#C2C6CC";
		}
	};
	// 渲染状态框背景颜色
	const renderBackgroundColor = (data: string) => {
		switch (data) {
			case "SUBMITTED":
				return "#EBF3FF";
			case "RUNNING":
				return "#EBF3FF";
			case "ABORTED":
				return "#F5F6F7";
			case "FAILED":
				return "#FFECE8";
			case "SUCCEEDED":
				return "#E8FFF1";
			default:
				return "#C2C6CC";
		}
	};
	//渲染进度条颜色
	const renderProgressColor = useCallback((data: any) => {
		switch (data) {
			case "SUBMITTED":
				return "info";
			case "RUNNING":
				return "info";
			case "ABORTED":
				return "disable";
			case "FAILED":
				return "error";
			case "SUCCEEDED":
				return "success";
			default:
				return "disable";
		}
	}, []);
	return (
		<div className={style.cardBox}>
			<div
				className={style.cardLeft}
				onClick={() => gotoDetail(id)}
				onMouseOver={() => setDisplayTitleBlue(true)}
				onMouseOut={() => setDisplayTitleBlue(false)}
			>
				<div className={style.topLine}>
					<div
						className={style.taskName}
						style={displayTitleBlue ? { color: "#1370ff" } : undefined}
					>
						{name}
					</div>
					<div
						className={style.taskStatus}
						style={{
							color: renderTextColor(state),
							background: renderBackgroundColor(state),
						}}
					>
						<img
							src={renderStatusIcon(state)}
							alt=""
							className={style.statusImg}
						/>
						<span>{renderStatusText(state)}</span>
					</div>
				</div>
				<div className={style.taskCreator}>{creator}</div>
				<div className={style.taskProgress}>
					<div className={style.progressInfo}>
						<div>Progress</div>
						<div style={{ color: renderTextColor(state) }}>
							{completeNum + "/" + totalNum}
						</div>
					</div>
					<MyProgress
						color={renderProgressColor(state)}
						value={(completeNum / totalNum) * 100}
					/>
				</div>
				<div className={style.bottomLine}>
					<img alt="" src={runTime} />
					<div className={style.bottomInfo}>{costTime}</div>
					<img alt="" src={jobCostImg} />
					<div className={style.bottomInfo}>{jobCost?.toFixed(2)}元</div>
				</div>
			</div>
			<div className={style.dividingLine}>
				<div className={style.arrow}></div>
			</div>
			<div className={style.cardRight}>
				{randerOutputs.length === 0 ? (
					<div className={style.noResult}>暂无结果文件</div>
				) : (
					<div className={style.resultBox}>
						{randerOutputs.map((item, index) => {
							return (
								<div
									key={index}
									className={style.result}
									onClick={() => goToProjectData(item)}
								>
									<img
										className={style.outputLeftImg}
										src={item?.type === "file" ? fileIcon : dataSetIcon}
										alt=""
									/>
									<span className={style.outputTitle}>{item?.name}</span>
								</div>
							);
						})}
					</div>
				)}
			</div>
		</div>
	);
});

export default TaskCard;
