import { useEffect, useState, useMemo } from "react";
import { observer } from "mobx-react-lite";
import classNames from "classnames";
import useMyRequest from "@/hooks/useMyRequest";
import { getWorkflowspecDetail } from "@/api/resourceCenter";
import { IResponse } from "@/api/http";
import goback from "@/assets/project/goback.svg";
import { ITaskInfo } from "@/views/Project/ProjectSubmitWork/interface";
import Flow from "@/views/Project/components/Flow";
import moment from "moment";
import MyTooltip from "@/components/mui/MyTooltip";
import { toJS } from "mobx";
import { useStores } from "@/store";

import styles from "./index.module.css";

interface ITemplateDetailProps {
	id: string;
	setShowDetail: any;
}

const TemplateDetail = observer((props: ITemplateDetailProps) => {
	const { id, setShowDetail } = props;
	const [templateInfo, setTemplateInfo] = useState<any>({});

	const [overviewActive, setOverviewActive] = useState(true);
	/** 选中的node Id */
	const [activeFlowIndex, setActiveFlowIndex] = useState<string>("");

	const { productListStore } = useStores();
	const getProductName = (productId: string) => {
		let res = "-";
		toJS(productListStore.productList).forEach((item) => {
			if (item.value === productId) {
				res = item.label;
			}
		});
		return res;
	};
	const taskInfo = useMemo(() => {
		if (activeFlowIndex) {
			return templateInfo.tasks.filter(
				(task: any) => task.id === activeFlowIndex
			)[0];
		} else {
			return {};
		}
	}, [templateInfo, activeFlowIndex]);

	const randerParameters = useMemo(() => {
		return taskInfo.parameters;
	}, [taskInfo]);

	/** 获取模版数据 */
	const { run } = useMyRequest(getWorkflowspecDetail, {
		onSuccess: (res: IResponse<ITaskInfo>) => {
			console.log(res);
			setTemplateInfo(res.data);
		},
	});

	useEffect(() => {
		run(id);
	}, [id, run]);

	/** 返回事件 */
	const onBack = () => {
		setShowDetail(false);
	};

	const setExternalSelectedNodeId = (id: string) => {
		setActiveFlowIndex(id);
	};

	const handleParams = () => {
		setOverviewActive(false);
	};

	return (
		<div className={styles.swBox}>
			<div className={styles.swHeader}>
				<div className={styles.swHeaderLeft}>
					<div className={styles.goBackIconBox}>
						<img
							onClick={onBack}
							className={styles.goBackIcon}
							src={goback}
							alt=""
						/>
					</div>
					<div className={styles.swTemplateTitle}>{templateInfo.title}</div>
				</div>
			</div>
			<div className={styles.swContent}>
				<div className={styles.swFormBox}>
					{!activeFlowIndex && (
						<div>
							<div className={styles.title}>基础信息</div>
							<div className={styles.taskInfoLi}>
								<div className={styles.taskInfoParams}>模板名称</div>
								<div className={styles.taskInfoValue}>{templateInfo.title}</div>
							</div>
							<div className={styles.taskInfoLi}>
								<div className={styles.taskInfoParams}>描述</div>
								<div className={styles.taskInfoValue}>
									{templateInfo.description}
								</div>
							</div>
							<div className={styles.taskInfoLi}>
								<div className={styles.taskInfoParams}>创建时间</div>
								<div className={styles.taskInfoValue}>
									{moment(new Date(templateInfo.createdAt)).format(
										"yyyy-MM-DD hh:mm"
									)}
								</div>
							</div>
							<div className={styles.taskInfoLi}>
								<div className={styles.taskInfoParams}>创建人</div>
								<div className={styles.taskInfoValue}>
									{templateInfo.creator}
								</div>
							</div>
							<div className={styles.taskInfoLi}>
								<div className={styles.taskInfoParams}>版本</div>
								<div className={styles.taskInfoValue}>
									{templateInfo.version}
								</div>
							</div>
							<div className={styles.taskInfoLi}>
								<div className={styles.taskInfoParams}>所属产品</div>
								<div className={styles.taskInfoValue}>
									{getProductName(templateInfo.productId)}
								</div>
							</div>
						</div>
					)}
					{activeFlowIndex && (
						<div className={styles.suanziInfo}>
							<MyTooltip>
								<div className={styles.title}>{taskInfo?.title}</div>
							</MyTooltip>
							<div className={styles.tabs}>
								<div
									className={classNames({
										[styles.tabLi]: true,
										[styles.tabLiAcitve]: overviewActive,
									})}
									onClick={() => setOverviewActive(true)}
								>
									概览
								</div>
								<div
									className={classNames({
										[styles.tabLi]: true,
										[styles.tabLiAcitve]: !overviewActive,
									})}
									onClick={() => handleParams()}
								>
									参数
								</div>
							</div>
							{overviewActive && (
								<div className={styles.overview}>
									<div className={styles.taskInfoLi}>
										<div className={styles.taskInfoParams}>描述</div>
										<div
											className={classNames({
												[styles.taskInfoValue]: true,
												[styles.taskInfoValueShowAll]: true,
											})}
										>
											{taskInfo?.description}
										</div>
									</div>
									<div className={styles.taskInfoLi}>
										<div className={styles.taskInfoParams}>算子版本</div>
										<div className={styles.taskInfoValue}>
											{taskInfo?.version || "-"}
										</div>
									</div>
								</div>
							)}
							{!overviewActive && (
								<div className={styles.params}>
									{randerParameters?.map((parameter: any) => {
										return (
											<div className={styles.taskInfoLi} key={parameter.name}>
												<div className={styles.taskInfoParams}>
													{parameter.name}
												</div>
												<div className={styles.taskInfoValue}>
													{parameter.defaultValue || "-"}
												</div>
											</div>
										);
									})}
								</div>
							)}
						</div>
					)}
				</div>
				<div className={styles.swFlowBox}>
					<Flow
						externalSelectedNodeId={activeFlowIndex}
						tasks={templateInfo?.tasks}
						setExternalSelectedNodeId={setExternalSelectedNodeId}
					/>
				</div>
			</div>
		</div>
	);
});

export default TemplateDetail;
