import React, { useState, useEffect } from "react";
import { observer } from "mobx-react-lite";
import _ from "lodash";
import moment from "moment";
import classNames from "classnames";

import useMyRequest from "@/hooks/useMyRequest";
import { getOverviewInfo, getTaskOverview } from "@/api/project_api";
import { useStores } from "@/store/index";
import NoProject from "@/components/BusinessComponents/NoProject";
import RadioGroupOfButtonStyle from "@/components/CommonComponents/RadioGroupOfButtonStyle";
import Loading from "@/views/Loading";
import SearchInput from "@/components/BusinessComponents/SearchInput";
import TaskCard from "./TaskCard";

import projectImg from "@/assets/project/projectIconSmall.svg";
import noFile from "@/assets/project/noFile.svg";
import style from "./index.module.css";

const ProjectOverview = observer(() => {
	const { currentProjectStore } = useStores();
	// 概览基本信息
	const [overviewInfo, setOverviewInfo] = useState<any>({});
	// 任务概览列表
	const [taskList, setTaskList] = useState([]);
	const [jobName, setJobName] = useState("");
	const [page, setPage] = useState(0);
	const [size, setSize] = useState(999);
	// 选择近7天or近15天or近30天
	const [day, setDay] = useState("7");
	// 滚轮是否到顶,判断是否显示阴影
	const [isTop, setIsTop] = useState(true);
	// 滚动滚轮时监听是否到顶
	const onscroll = (e: any) => {
		if (e.target.scrollTop <= 0) {
			setIsTop(true);
		} else {
			setIsTop(false);
		}
	};
	// 获取概览基本信息
	const { run: getOverview, loading } = useMyRequest(getOverviewInfo, {
		onSuccess: (result: any) => {
			setOverviewInfo(result.data);
		},
	});
	// 获取任务概览
	const { run: getTaskOverviewList } = useMyRequest(getTaskOverview, {
		onSuccess: (result: any) => {
			setTaskList(result.data.content);
		},
	});

	useEffect(() => {
		// 切项目时重置为初始7
		setDay("7");
		if (currentProjectStore.currentProjectInfo.id) {
			getOverview({
				id: currentProjectStore.currentProjectInfo.id as string,
			});
		}
	}, [currentProjectStore.currentProjectInfo.id, getOverview]);

	useEffect(() => {
		if (currentProjectStore.currentProjectInfo.id) {
			getTaskOverviewList({
				projectId: currentProjectStore.currentProjectInfo.id as string,
				jobName: jobName,
				day: Number(day),
				page: page,
				size: size,
			});
		}
	}, [
		currentProjectStore.currentProjectInfo.id,
		getTaskOverviewList,
		day,
		jobName,
	]);

	const handleKeyWordChangeKeyUp = (e: any) => {
		if (e.keyCode === 13) {
			setJobName(e.target.value);
		}
	};

	const handleKeyWordChangeBlur = (e: any) => {
		setJobName(e.target.value);
	};

	const storageUnitFromB = (b: number) => {
		if (b <= 0) {
			return { data: "0.00", unit: "KB" };
		} else if (b < 1024 * 1024) {
			return { data: (b / 1024).toFixed(2), unit: "KB" };
		} else if (b < 1024 * 1024 * 1024) {
			return { data: (b / (1024 * 1024)).toFixed(2), unit: "MB" };
		} else if (b < 1024 * 1024 * 1024 * 1024) {
			return { data: (b / (1024 * 1024 * 1024)).toFixed(2), unit: "G" };
		} else {
			return { data: (b / (1024 * 1024 * 1024 * 1024)).toFixed(2), unit: "T" };
		}
	};

	if (currentProjectStore.currentProjectInfo.name) {
		if (loading) {
			return (
				<div className={style.loadingBox}>
					<Loading />
				</div>
			);
		} else {
			return (
				<div className={style.box} onScroll={onscroll}>
					<div
						className={classNames({
							[style.topFixed]: true,
							[style.topFixedShadow]: !isTop,
						})}
					>
						<div className={style.basicInformation}>
							<div>
								<div className={style.titleBox}>
									<img
										src={projectImg}
										alt="项目logo"
										style={{ width: 30, height: 30 }}
									/>
									<span className={style.projectName}>
										{currentProjectStore.currentProjectInfo.name}
									</span>
								</div>
								<div className={style.otherInformation}>
									<div className={style.otherInformationBox}>
										<span>角色：</span>
										<span className={style.informationcolor}>
											{overviewInfo.projectRoleExhibit}
										</span>
									</div>
									<div className={style.otherInformationBox}>
										<span>计算区：</span>
										<span className={style.informationcolor}>
											{overviewInfo.zoneIdExhibit}
										</span>
									</div>
									<div className={style.otherInformationBox}>
										<span>创建日期：</span>
										<span className={style.informationcolor}>
											{moment(overviewInfo.createdAt).format("yyyy-MM-DD")}
										</span>
									</div>
								</div>
							</div>
							<div className={style.basicInformationRight}>
								<div>
									<div className={style.otherInformationBoxRight}>
										项目总消费
									</div>
									<div>
										<span className={style.numberDisplay}>
											{overviewInfo.projectCost?.toFixed(2)}
										</span>
										元
									</div>
								</div>
								<div className={style.verticalLine}></div>
								<div>
									<div className={style.otherInformationBoxRight}>
										项目剩余预算
									</div>
									<div>
										<span className={classNames({
											[style.numberDisplay]: true,
											[style.red]:overviewInfo.projectRemainingBudget?.toFixed(2) < 100
										})}>
											{overviewInfo.projectRemainingBudget?.toFixed(2)}
										</span>
										元
									</div>
								</div>
								<div className={style.verticalLine}></div>
								<div>
									<div
										className={style.otherInformationBoxRight}
										style={{ marginRight: "20px" }}
									>
										项目存储大小
									</div>
									<div>
										<span className={style.numberDisplay}>
											{overviewInfo.projectStorage !== undefined
												? storageUnitFromB(overviewInfo.projectStorage).data
												: ""}
										</span>
										{overviewInfo.projectStorage !== undefined
											? storageUnitFromB(overviewInfo.projectStorage).unit
											: ""}
									</div>
								</div>
							</div>
						</div>
						<div className={style.searchBox}>
							<div className={style.searchLineLeft}>
								<div className={style.taskOverview}>任务概览</div>
								<RadioGroupOfButtonStyle
									value={day}
									radioOptions={[
										{ value: "7", label: "近7天" },
										{ value: "15", label: "近15天" },
										{ value: "30", label: "近30天" },
									]}
									handleRadio={setDay}
								/>
							</div>
							<SearchInput
								onKeyUp={handleKeyWordChangeKeyUp}
								onBlur={handleKeyWordChangeBlur}
								sx={{ width: 340 }}
							/>
						</div>
					</div>
					<div className={style.taskDisplay}>
						{/* 任务列表为空展示 */}
						{taskList.length === 0 && (
							<div className={style.noDataBox}>
								<img className={style.noDataImg} src={noFile} alt="" />
								<span className={style.noDataText}>暂无任务</span>
							</div>
						)}
						{/* 任务列表卡片渲染 */}
						{taskList.length > 0 &&
							taskList.map((item: any) => {
								return (
									<TaskCard
										id={item.id}
										name={item.name}
										creator={item.creator}
										state={item.state}
										completeNum={item.completeNum}
										totalNum={item.totalNum}
										costTime={item.costTime}
										jobCost={item.jobCost}
										outputs={item.outputs}
										key={item.id}
									/>
								);
							})}
					</div>
				</div>
			);
		}
	} else {
		return <NoProject />;
	}
});
export default ProjectOverview;
