import { useState, useEffect, useMemo } from "react";
import MyDialog from "@/components/mui/MyDialog";
import { checkIsNumberLetterChinese } from "@/utils/util";
import { useMessage } from "@/components/MySnackbar";
import CloudEController from "@/api/fileserver/CloudEController";
import FileSelect from "@/components/BusinessComponents/FileSelect";
import fileSelectIcon from "@/assets/project/fileSelect.svg";
import MyInput from "@/components/mui/MyInput";
import { BACKEND_API_URI_PREFIX } from "@/api/api_prefix";
import InputAdornment from "@mui/material/InputAdornment";

type ISaveProps = {
	type: string;
	projectId: string;
	token: string;
	fileToken: string;
	path: string;
	name: string;
	selectIds: Array<any>;
	open: boolean;
	setOpen: any;
};

const Save = (props: ISaveProps) => {
	const {
		type,
		path,
		projectId,
		token,
		fileToken,
		selectIds,
		name,
		open,
		setOpen,
	} = props;
	const Message = useMessage();
	const [dname, setDname] = useState("");
	const [dpath, setDpath] = useState("/");
	const [fileSelectOpen, setFileSelectOpen] = useState(false);
	const [datasetNameCheck, setDatasetNameCheck] = useState({
		error: false,
		help: "",
	});
	const [dpathCheck, setDpathCheck] = useState({
		error: false,
		help: "",
	});

	useEffect(() => {
		if (open) {
			setDname("");
			setDatasetNameCheck({
				error: false,
				help: "",
			});
		}
	}, [open]);

	const showDpath = useMemo(() => {
		return `ProjectData${dpath === "/" ? "" : dpath}`;
	}, [dpath]);

	const handleSaveSubmit = () => {
		if (dname && !datasetNameCheck.error && dpath) {
			CloudEController.SaveDataset({
				type: type,
				projectId: projectId,
				token: token,
				filetoken: fileToken,
				name: name,
				path: path,
				dpath: dpath,
				dname: dname,
				id: selectIds.join(","),
				baseurl: BACKEND_API_URI_PREFIX,
			})
				?.then((res) => {
					Message.success("另存为成功！");
					setOpen(false);
				})
				?.catch((error) => {
					console.log(error);
					Message.error(error?.response?.data?.message || "文件服务发生错误!");
				});
		} else {
			if (!dpath) {
				setDpathCheck({
					error: true,
					help: "请选择另存为路径",
				});
			} else {
				setDpathCheck({
					error: false,
					help: "",
				});
			}
			if (!dname) {
				setDatasetNameCheck({
					error: true,
					help: "数据集名称不能为空",
				});
			}
		}
	};

	const handleDnameChange = (e: any) => {
		const dname = e.target.value;
		setDname(dname);
		if (!dname) {
			setDatasetNameCheck({
				error: true,
				help: "数据集名称不能为空",
			});
		} else if (!checkIsNumberLetterChinese(dname) || dname.length > 30) {
			setDatasetNameCheck({
				error: true,
				help: "格式不正确，必须在30字符以内，仅限大小写字母、数字、中文",
			});
		} else {
			setDatasetNameCheck({
				error: false,
				help: "",
			});
		}
	};

	const onFileSelectConfirm = (path: string) => {
		setFileSelectOpen(false);
		setDpath(path);
	};

	return (
		<>
			<MyDialog
				open={open}
				onClose={() => setOpen(false)}
				onConfirm={handleSaveSubmit}
				title="另存为"
			>
				<MyInput
					sx={{
						width: "388px",
						display: "block",
						margin: "20px 0",
					}}
					required
					error={datasetNameCheck.error}
					label="数据集名称"
					variant="outlined"
					value={dname}
					onChange={handleDnameChange}
					helperText={datasetNameCheck.help}
					InputProps={{
						endAdornment: (
							<InputAdornment position="end">{dname.length}/30</InputAdornment>
						),
					}}
					size="small"
				/>
				<MyInput
					sx={{
						width: "388px",
					}}
					value={showDpath}
					required
					label="保存路径"
					InputProps={{
						endAdornment: (
							<img
								onClick={() => {
									setFileSelectOpen(true);
								}}
								src={fileSelectIcon}
								alt="选择输出路径"
								style={{ cursor: "pointer" }}
							/>
						),
					}}
					error={dpathCheck.error}
					helperText={dpathCheck.help}
				></MyInput>
				{fileSelectOpen && (
					<FileSelect
						onClose={() => {
							setFileSelectOpen(false);
						}}
						open={fileSelectOpen}
						onConfirm={onFileSelectConfirm}
					/>
				)}
			</MyDialog>
		</>
	);
};

export default Save;
