import { useCallback, useEffect, useMemo, useRef, useState } from "react";
import VirtuallyList from "../VirtuallyList";
import style from "./index.module.scss";

interface IVrituallyCardTableProps {
	data: Array<any>; // 列表数据
	renderItem: any; // 单个卡片的渲染函数 这里面的元素的boxSizing属性最好是"border-box",
	rowHeight: number; // 行高 这个属性的高度最好是itemHeight + verticalSpacing
	itemMinWidth?: number; // 单个卡片的最小宽度，有这个参数时numberOfColumns参数失效，效果为根据屏幕大小和单个卡片的最小宽度来适配每行渲染个数
	tableKey?: string; // 表格数据的key
	numberOfColumns?: number; // 列数   每行渲染几个
	horizontalSpacing?: number; // 水平方向的间隔
	verticalSpacing?: number; // 垂直方向的间隔
	loading?: boolean;
}

const VrituallyCardTable = (props: IVrituallyCardTableProps) => {
	const {
		data,
		renderItem,
		tableKey = "id",
		rowHeight,
		numberOfColumns: propsNumberOfColumns = 3,
		horizontalSpacing = 20,
		verticalSpacing = 20,
		itemMinWidth,
		loading = false,
	} = props;

	const [numberOfColumns, setNumberOfColumns] = useState(3);
	const tableBoxRef: any = useRef(null);
	const getNumberOfColumns = useCallback(() => {
		if (itemMinWidth) {
			const boxWidth = tableBoxRef?.current?.offsetWidth;
			if (boxWidth) {
				setNumberOfColumns(Math.floor(boxWidth / itemMinWidth));
			} else {
				setNumberOfColumns(propsNumberOfColumns);
			}
		} else {
			setNumberOfColumns(propsNumberOfColumns);
		}
	}, [itemMinWidth, propsNumberOfColumns]);

	useEffect(() => {
		getNumberOfColumns();
	}, [getNumberOfColumns]);

	const boxWidth = useMemo(() => {
		return `${100 / numberOfColumns}%`;
	}, [numberOfColumns]);

	window.onresize = () => {
		getNumberOfColumns();
	};

	const listData = useMemo(() => {
		let resData: any = [[]];
		data.forEach((item) => {
			if (resData[resData.length - 1].length >= numberOfColumns) {
				resData.push([item]);
			} else {
				resData[resData.length - 1].push(item);
			}
		});
		return resData;
	}, [numberOfColumns, data]);

	const renderRow = ({
		index,
		isScrolling,
		isVisible,
		key,
		parent,
		style,
	}: any) => {
		return (
			<div key={key} style={style}>
				<div
					style={{
						display: "flex",
						justifyContent: "space-between",
						alignItems: "center",
						boxSizing: "border-box",
						height: "100%",
					}}
				>
					{listData[index].map((item: any, index: number) => {
						return (
							<div
								className={style.itemBox}
								key={item[tableKey] ? item[tableKey] : index}
								style={{
									width: boxWidth,
									paddingLeft: `${horizontalSpacing / 2}px`,
									paddingRight: `${horizontalSpacing / 2}px`,
									paddingBottom: `${verticalSpacing}px`,
									boxSizing: "border-box",
								}}
							>
								{renderItem(item, index)}
							</div>
						);
					})}
				</div>
			</div>
		);
	};

	return (
		<div
			className={style.tableBox}
			style={{
				marginLeft: `-${horizontalSpacing / 2}px`,
				marginRight: `-${horizontalSpacing / 2}px`,
				width: `calc(100% + ${horizontalSpacing}px)`,
			}}
			ref={tableBoxRef}
		>
			<VirtuallyList
				list={listData}
				renderRow={renderRow}
				rowHeight={rowHeight}
				loading={loading}
			></VirtuallyList>
		</div>
	);
};

export default VrituallyCardTable;
