import React from "react";

import {
	Dialog,
	DialogActions,
	DialogContent,
	DialogTitle,
} from "@mui/material";
import CloseIcon from "@mui/icons-material/Close";

import MyButton from "./MyButton";

export interface IDialogProps {
	/** 自定义类名 */
	className?: string;
	/** 自定义样式 */
	style?: React.CSSProperties;
	/** 弹窗的标题 */
	title?: string;
	/** 是否显示弹窗 */
	open: boolean;
	isHideHeader?: boolean;
	/** 是否隐藏弹窗底部按钮部分 */
	isHideFooter?: boolean;
	/** 自定义底部按钮 */
	footerRender?: () => React.ReactNode;
	/** 是否显示取消按钮 */
	showCancel?: boolean;
	/** 是否显示确定按钮 */
	showConfirm?: boolean;
	/** 关闭弹窗时的回调函数 */
	onClose?: () => void;
	/** 点击确定按钮时的回调函数 */
	onConfirm?: () => void;
	/** 取消按钮文案 */
	cancelText?: string;
	/** 确认按钮文案 */
	okText?: string;
	/** 是否禁用确认按钮 */
	disabledConfirm?: boolean;
	children: React.ReactNode;
	/** 点击遮罩是否关闭  默认为false*/
	clickMaskClose?: boolean;
	/** 确认按钮样式*/
	okSx?: any;
	loading?: boolean; // 确认按钮是否处于loading状态
}

const MyDialog: React.FunctionComponent<IDialogProps> = (props) => {
	const {
		title,
		open,
		style,
		onClose,
		onConfirm,
		isHideFooter,
		isHideHeader,
		children,
		footerRender,
		className,
		showCancel = true,
		/** 是否显示确定按钮 */
		showConfirm = true,
		cancelText,
		okText,
		disabledConfirm,
		clickMaskClose = false,
		loading = false,
		okSx = {},
	} = props;

	const handelClose = (
		event: {},
		reason: "backdropClick" | "escapeKeyDown"
	) => {
		if (reason === "backdropClick" && !clickMaskClose) {
			return;
		}
		onClose && onClose();
	};

	const Footer = () => {
		if (isHideFooter) return null;
		return footerRender ? (
			footerRender()
		) : (
			<DialogActions style={{ padding: "0 24px 24px 24px" }}>
				{showCancel ? (
					<MyButton
						text={cancelText || "取消"}
						onClick={onClose}
						variant="outlined"
						size="small"
						color="secondary"
					/>
				) : null}
				{showConfirm ? (
					<MyButton
						text={okText || "确定"}
						onClick={onConfirm}
						variant="contained"
						size="small"
						disabled={disabledConfirm}
						isLoadingButton={true}
						loading={loading}
						style={{ ...okSx }}
					/>
				) : null}
			</DialogActions>
		);
	};
	return (
		<Dialog
			open={open}
			onClose={handelClose}
			style={style}
			className={className}
			aria-labelledby="alert-dialog-title"
			aria-describedby="alert-dialog-description"
			sx={{
				"& .MuiDialog-container": {
					"& .MuiPaper-root": {
						// 设置最大宽度， 实际宽度让子元素撑大
						maxWidth: "1920px",
					},
				},
			}}
		>
			{isHideHeader ? null : (
				<DialogTitle id="alert-dialog-title">
					<div
						style={{
							display: "flex",
							justifyContent: "space-between",
							alignItems: "center",
							fontWeight: 600,
						}}
					>
						<span>{title}</span>
						<CloseIcon
							onClick={onClose}
							style={{ color: "#C2C6CC", cursor: "pointer" }}
						/>
					</div>
				</DialogTitle>
			)}
			<DialogContent style={{ minWidth: 388 }}>{children}</DialogContent>
			{Footer()}
		</Dialog>
	);
};

export default MyDialog;
