import { useState, useCallback, useEffect, useMemo } from "react";
import classnames from "classnames";
import style from "./index.module.css";
import MyButton from "@/components/mui/MyButton";
import MyTooltip from "@/components/mui/MyTooltip";
import InsertDriveFileOutlinedIcon from "@mui/icons-material/InsertDriveFileOutlined";
import CloudEController from "@/api/fileserver/CloudEController";
import { useStores } from "@/store";
import { toJS } from "mobx";
import FullScreenDrawer from "@/components/CommonComponents/FullScreenDrawer";
import ChevronLeftIcon from '@mui/icons-material/ChevronLeft';
import ChevronRightIcon from '@mui/icons-material/ChevronRight';
import useWindowSize from '@/hooks/useWindowSize'


type LogViewProps = {
    logs: any[];
    setShowLogView: any;
};

const LogView = (props: LogViewProps) => {
    const { logs, setShowLogView } = props;
    const { currentProjectStore } = useStores();
    const { width, height } = useWindowSize();
    const fileToken = toJS(currentProjectStore.currentProjectInfo.filetoken);
    const projectId = toJS(currentProjectStore.currentProjectInfo.id);
    // 当前选择的日志
    const [logCurrent, setLogCurrent] = useState<number>(0);
    // 当前日志的内容文本
    const [logText, setLogText] = useState('')
    // 当前日志路径
    const [logPath, setLogPath] = useState('')

    const [displacement, setDisplacement] = useState(0)

    const [middleDynamicWidth, setMiddleDynamicWidth] = useState(0)

    const [leftButtonColor, setLeftButtonColor] = useState('#585D62')
    const [rightButtonColor, setRightButtonColor] = useState('#585D62')

    useEffect(() => {
        setLogPath(logs[logCurrent]?.logPath)
    }, [logs]);

    // 请求日志文本
    useEffect(() => {
        if (logPath) {
            const path = logPath.slice(12)
            CloudEController.JobFileDownloadText(
                path,
                fileToken as string,
                projectId as string
            )?.then((res) => {
                setLogText(res.data)
            })
        } else {
            setLogText("")
        }
    }, [logPath]);

    // 选择日志时改变日志路径
    useEffect(() => {
        setLogPath(logs[logCurrent]?.logPath)
    }, [logCurrent]);

    //获取盒子的总宽度，用于滑动效果判断
    useEffect(() => {
        const box = document.getElementById('middleDynamic')
        setMiddleDynamicWidth(box ? box.offsetWidth : 0)
    }, [])

    useEffect(() => {
        if (middleDynamicWidth < width - 97) {
            setLeftButtonColor('#585D62')
            setRightButtonColor('#585D62')
        }
        if (displacement === 0) {
            setLeftButtonColor('#585D62')
        } else {
            setLeftButtonColor('#C0C5CD')
        }
        if (middleDynamicWidth > width - 97 && displacement !== -middleDynamicWidth + width - 97) {
            setRightButtonColor('#C0C5CD')
        } else {
            setRightButtonColor('#585D62')
        }
    }, [width, middleDynamicWidth, displacement])

    // 下载当前日志
    const handleDownLoad = () => {
        const path = logPath.slice(12)
        CloudEController.JobFileDownload(
            path,
            fileToken as string,
            projectId as string
        );
    }

    const rightClick = () => {
        if (middleDynamicWidth < width - 97) {
            return
        }
        if (-displacement > middleDynamicWidth - width * 1.8 + 97) {
            setDisplacement(-middleDynamicWidth + width - 97)
            return
        }
        const newDisplacement = displacement - width * 0.8;
        setDisplacement(newDisplacement)
    }

    const leftClick = () => {
        if (-displacement < width * 0.8) {
            setDisplacement(0)
            return
        }
        const newDisplacement = displacement + width * 0.8;
        setDisplacement(newDisplacement)
    }

    return (
        <FullScreenDrawer handleClose={setShowLogView} zIndex={1002}>
            <div className={style.logViewBox}>
                <div className={style.logViewContentMask}></div>
                <div className={style.logViewTop}>
                    <div
                        className={style.leftButton}
                        onClick={leftClick}
                        style={{ color: leftButtonColor, cursor: leftButtonColor === '#585D62' ? 'default' : 'pointer' }}
                    >
                        <ChevronLeftIcon />
                    </div>
                    <div className={style.middleFixed}>
                        <div className={style.middleDynamic} id='middleDynamic' style={{ left: `${displacement}px` }}>
                            {logs.map((item: any, index: number) => {
                                return <MyTooltip
                                    title={item.logName}
                                    placement="bottom"
                                    arrow={false}
                                    enterDelay={1000}
                                >
                                    <div
                                        key={index}
                                        onClick={() => { setLogCurrent(index) }}
                                        className={classnames({
                                            [style.logTitle]: true,
                                            [style.logTitleSelected]: index === logCurrent,
                                        })}>
                                        <InsertDriveFileOutlinedIcon className={style.fileIcon} />
                                        <span className={style.logName}>{item.logName}</span>
                                    </div>
                                </MyTooltip>
                            })}
                        </div>
                    </div>
                    <div className={style.gradientBox}></div>
                    <div
                        className={style.rightButton}
                        onClick={rightClick}
                        style={{ color: rightButtonColor, cursor: rightButtonColor === '#585D62' ? 'default' : 'pointer' }}
                    >
                        <ChevronRightIcon />
                    </div>
                </div>
                <div className={style.logViewContent}>
                    {logText}
                </div>
                <div className={style.logViewBottom}>
                    <MyButton text='下载当前日志' onClick={handleDownLoad} />
                </div>
            </div>
        </FullScreenDrawer>
    )
}

export default LogView