import { useCallback, useEffect, useState, useRef } from "react";
import { Column, Table } from "react-virtualized";
import style from "./index.module.scss";
import "react-virtualized/styles.css"; // only needs to be imported once
import Checkbox from "@mui/material/Checkbox";
import MyCircularProgress from "@/components/mui/MyCircularProgress";
import { createTheme, ThemeProvider } from "@mui/material";
import NoData from "@/components/BusinessComponents/NoData";
import ascIcon from "@/assets/project/ascIcon.svg";
import descIcon from "@/assets/project/descIcon.svg";
import sortIcon from "@/assets/project/sort.svg";

type Order = "ASC" | "DESC"; // 升序为asc，降序为desc。

export type sortState = {
	field: string | null | undefined | ""; // 根据哪个属性来排序
	order: Order | null | undefined | "";
};

interface IVirtuallyTableProps {
	rows: Array<any>; // 表格数据
	headCells: Array<any>; // 表头配置
	tableKey?: string; // 表格数据的key
	loading?: boolean; // 是否正在加载数据
	hasCheckbox?: boolean; // 是否有复选框
	selectItems?: Array<any>; // 选中的项
	setSelectItems?: any; // 设置选中的项
	noDataText?: string; // 无数据提示文案
	sortState?: sortState; // 排序状态
	setSortState?: any; // 设置排序状态
	nodataText?: any; // 无数据文案
	handleRow?: any; // 点击一行
	activeId?: string; // 选中的一行的id
	disableFn?: any; // 禁用时根据disableFn来判断是否禁用
	headerHeight?: number; // 表头高度
}

const theme = createTheme({
	components: {
		// 复选框样式
		MuiSvgIcon: {
			styleOverrides: {
				root: {
					color: "rgba(209, 214, 222, 1)",
					fontSize: "18px",
				},
			},
		},
		MuiCheckbox: {
			styleOverrides: {
				root: {
					color: "rgba(209, 214, 222, 1)",
					"&.MuiCheckbox-indeterminate .MuiSvgIcon-root": {
						color: "rgba(19, 112, 255, 1)",
					},
					"&.Mui-checked .MuiSvgIcon-root": {
						color: "rgba(19, 112, 255, 1)",
					},
				},
			},
		},
		MuiButtonBase: {
			styleOverrides: {
				root: {
					"&.MuiCheckbox-root": {
						padding: 0,
					},
				},
			},
		},
	},
});

const VirtuallyTable = (props: IVirtuallyTableProps) => {
	const {
		rows,
		headCells,
		tableKey = "id",
		loading = false,
		hasCheckbox = false,
		selectItems = [],
		setSelectItems,
		sortState,
		setSortState,
		nodataText,
		handleRow,
		activeId,
		disableFn,
		headerHeight = 59,
	} = props;

	const virtuallyTableBoxRef: any = useRef(null);

	const [width, setWidth] = useState(0);
	const [height, setHeight] = useState(0);

	const getTableWidthHeight = () => {
		setWidth(virtuallyTableBoxRef?.current?.offsetWidth || 1000);
		setHeight(virtuallyTableBoxRef?.current?.offsetHeight || 300);
	};

	useEffect(() => {
		getTableWidthHeight();
	}, []);

	window.onresize = () => {
		getTableWidthHeight();
	};

	const onSelectAllClick = useCallback(
		(e: any) => {
			if (e.target.checked) {
				setSelectItems && setSelectItems(rows.map((row) => row[tableKey]));
			} else {
				setSelectItems && setSelectItems([]);
			}
		},
		[setSelectItems, tableKey, rows]
	);

	const onSelectRowClick = useCallback(
		(e: any, itemValue: string) => {
			if (e.target.checked) {
				setSelectItems && setSelectItems([...selectItems, itemValue]);
			} else {
				const selectItemIndex = selectItems.indexOf(itemValue);
				const newSelectItems = [
					...selectItems.slice(0, selectItemIndex),
					...selectItems.slice(selectItemIndex + 1, selectItems.length),
				];
				setSelectItems && setSelectItems(newSelectItems);
			}
		},
		[selectItems, setSelectItems]
	);

	const handleSort = useCallback(
		(field: string) => {
			if (sortState?.field === field) {
				if (sortState?.order === "ASC") {
					setSortState({
						field,
						order: "DESC",
					});
				} else if (sortState?.order === "DESC") {
					setSortState({
						field,
						order: "ASC",
					});
				} else {
					setSortState({
						field,
						order: "DESC",
					});
				}
			} else {
				setSortState({
					field,
					order: "DESC",
				});
			}
		},
		[sortState, setSortState]
	);

	const handleRowFn = useCallback(
		(row: any) => {
			if (!disableFn) {
				handleRow && handleRow(row);
			} else {
				!disableFn(row) && handleRow && handleRow(row);
			}
		},
		[disableFn, handleRow]
	);

	return (
		<ThemeProvider theme={theme}>
			<div
				ref={virtuallyTableBoxRef}
				style={{ width: "100%", height: "100%", position: "relative" }}
			>
				<MyCircularProgress loading={loading} />
				{width && height && (
					<Table
						width={width}
						height={height}
						headerHeight={headerHeight}
						rowHeight={54}
						rowCount={rows.length}
						rowGetter={({ index }: any) => rows[index]}
						headerClassName={style.VTHeader}
						onRowClick={(data: any) => {
							handleRowFn(data.rowData);
						}}
						rowClassName={({ index }: any) => {
							if (index < 0) {
								return style.VTHeaderRow;
							} else {
								if (rows[index][tableKey] === activeId) {
									return style.VTActiveRow;
								} else {
									return style.VTRow;
								}
							}
						}}
						rowStyle={({ index }: any) => {
							if (index !== -1) {
								return {
									background:
										activeId === rows[index][tableKey]
											? "rgba(237, 244, 255, 1)"
											: "",
									cursor: disableFn && disableFn(rows[index]) ? "no-drop" : "",
									opacity: disableFn && disableFn(rows[index]) ? "0.3" : "",
								};
							}
						}}
					>
						{hasCheckbox && (
							<Column
								dataKey="checkbox"
								headerRenderer={() => {
									return (
										<Checkbox
											indeterminate={
												selectItems.length > 0 &&
												selectItems.length < rows.length
											}
											checked={
												rows.length > 0 && selectItems.length === rows.length
											}
											onChange={(e) => onSelectAllClick(e)}
										/>
									);
								}}
								headerStyle={{ margin: 0 }}
								style={{ margin: 0 }}
								width={50}
								cellRenderer={(data: any) => {
									return (
										<Checkbox
											checked={
												selectItems.filter(
													(selectItem) => selectItem === data.rowData[tableKey]
												).length > 0
											}
											onChange={(e) =>
												onSelectRowClick(e, data.rowData[tableKey])
											}
										/>
									);
								}}
								className={style.VTRowColumn}
							/>
						)}
						{headCells.map((headCell) => {
							return (
								<Column
									key={headCell.id}
									// label={headCell.label}
									headerRenderer={(data: any) => {
										if (headCell.sort) {
											return (
												<div>
													<span>{headCell.label}</span>
													<img
														src={
															sortState?.field === headCell.id
																? sortState?.order === "ASC"
																	? ascIcon
																	: sortState?.order === "DESC"
																	? descIcon
																	: sortIcon
																: sortIcon
														}
														alt=""
														onClick={() => handleSort(headCell.id)}
														style={{
															marginLeft: "8px",
															cursor: "pointer",
															position: "relative",
															top: "3px",
														}}
													/>
												</div>
											);
										} else {
											return headCell.label;
										}
									}}
									dataKey={headCell.id}
									width={headCell.width}
									flexGrow={headCell.flexGrow || 0}
									cellRenderer={
										headCell.cellRenderer
											? headCell.cellRenderer
											: (data: any) => {
													return data.cellData;
											  }
									}
									className={style.VTRowColumn}
								/>
							);
						})}
					</Table>
				)}
				{rows.length === 0 && (
					<NoData
						text={nodataText}
						noDataBoxStyle={{
							position: "absolute",
							bottom: 0,
							width: `${width}px`,
							height: `${height - headerHeight}px`,
						}}
					/>
				)}
			</div>
		</ThemeProvider>
	);
};

export default VirtuallyTable;
