package com.edgec.browserbackend.browser.task;

import com.alibaba.fastjson.JSONObject;
import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.domain.IpChargeRequestDto;
import com.edgec.browserbackend.account.domain.QueryIpUrlList;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.account.repository.QueryIpUrlListRepository;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.browser.domain.IpCountRecord;
import com.edgec.browserbackend.browser.domain.IpResource;
import com.edgec.browserbackend.browser.domain.IpType;
import com.edgec.browserbackend.browser.dto.IpBuyResultDto;
import com.edgec.browserbackend.browser.dto.IpInfoResultDto;
import com.edgec.browserbackend.browser.dto.ShopRequestDto;
import com.edgec.browserbackend.browser.repository.IpCountRecordRepository;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.common.commons.utils.NotifyUtils;
import com.edgec.browserbackend.common.utils.ThreadPoolUtils;
import com.edgec.browserbackend.common.utils.Trans;
import okhttp3.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.time.Instant;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import static com.edgec.browserbackend.browser.service.Impl.IpResourceServiceImpl.genRandom;
import static com.edgec.browserbackend.browser.service.Impl.IpResourceServiceImpl.region;

@Component
public class BrowserTask {

    private static final Logger log = LoggerFactory.getLogger(BrowserTask.class);

    private static String CLOUDAMURL = "https://www.cloudam.cn";
    private static String TESTURL = "http://112.74.13.2";


    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Autowired
    private AccountService accountService;

    @Autowired
    private IpAndShopService ipAndShopService;

    @Autowired
    private QueryIpUrlListRepository queryIpUrlListRepository;

    @Autowired
    private IpCountRecordRepository ipCountRecordRepository;

    @Autowired
    private AccountRepository accountRepository;

    @Value("${spring.profiles.active}")
    private String profiles;

    public Map<String, String> buildGetHeader() {
        Map<String, String> headers = new HashMap<>();
        headers.put("Content-Type", "application/json");
        if (profiles.equals("dev") || profiles.equals("staging"))
            headers.put("Authorization", "Bearer oq5tg3gMsflHcK5iZ2741G5R30XYd9blyOqH9qeBItKtrzfTsGIoy8AsxqqNXdcm");
        else if (profiles.equals("prod"))
            headers.put("Authorization", "Bearer tKWsuHzcngf0RQPMss70f9jgymDIwgQ9zbLfESJdcou3pZSNWl7lNTzto8VQgwaO");
        return headers;
    }

    public Map<String, String> buildPostHeader() {
        Map<String, String> headers = new HashMap<>();
        headers.put("Content-Type", "application/json");
        if (profiles.equals("dev") || profiles.equals("staging"))
            headers.put("Authorization", "Bearer oq5tg3gMsflHcK5iZ2741G5R30XYd9blyOqH9qeBItKtrzfTsGIoy8AsxqqNXdcm");
        else if (profiles.equals("prod"))
            headers.put("Authorization", "Bearer tKWsuHzcngf0RQPMss70f9jgymDIwgQ9zbLfESJdcou3pZSNWl7lNTzto8VQgwaO");
        return headers;
    }

    private IpChargeRequestDto buildIpChargeRequestDto(IpResource request, int chargeType, int payMethod) {
        IpChargeRequestDto ipChargeRequestDto = new IpChargeRequestDto();
        ipChargeRequestDto.setAmount(1);
        ipChargeRequestDto.setChargeType(chargeType);
        ipChargeRequestDto.setRegion(request.getRegion());
        ipChargeRequestDto.setPeriod(request.getPeriod());
        ipChargeRequestDto.setUnit(request.getUnit());
        ipChargeRequestDto.setPayMethod(0);
        return ipChargeRequestDto;
    }


    /**
     * 购买ip的定时任务，每分钟一次
     */
    @Scheduled(cron = "0 0/1 * * * ?")
    public void buyIpTasks() {
        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        long time = Instant.now().minusSeconds(300).toEpochMilli();
        List<IpResource> ipResources = ipResourceRepository.sampleTasks(6, time);
        log.info("buyIpTasks sample {} tasks", ipResources.size());
        List<CompletableFuture> futureList = new ArrayList<>();
        for (IpResource ipResource : ipResources) {
            long start = System.currentTimeMillis();

            CompletableFuture future = CompletableFuture.runAsync(
                    () -> {
                        if (ipResourceRepository.lockTask(ipResource)) {
                            try {
                                boolean result = false;
                                Map<String, String> header = buildPostHeader();
                                HashMap<String, Object> map = new HashMap<>();
                                map.put("name", ipResource.getUsername());
                                map.put("region", ipResource.getRegion());
                                map.put("period", String.valueOf(ipResource.getPeriod()));
                                map.put("provider", ipResource.getVendor());
                                map.put("unit", ipResource.getUnit());
                                map.put("amount", 1);
                                map.put("loginPassword", ipResource.getPassword());
                                map.put("startscript", "");
                                map.put("ipkeptperiod", 7);
                                map.put("persistSystemDiskOnTermination", false);
//                        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(map, header);
                                IpBuyResultDto ipBuyResultDto = null;
                                try {
                                    String requestResult = HttpClientutils.doPost(URL + "/intelligroup/ipresources?accountId=browser", header, JSONObject.toJSONString(map));
                                    ipBuyResultDto = JSONObject.parseObject(requestResult, IpBuyResultDto.class);
                                    if (StringUtils.isNotBlank(ipBuyResultDto.getErrorCode())) {
                                        log.error("fail to buy ip");
                                        log.error(ipBuyResultDto.getErrorCode());
                                    }
                                    if (ipBuyResultDto != null && ipBuyResultDto.getIplist() != null && ipBuyResultDto.getIplist().size() >= 1) {
                                        AtomicInteger index = new AtomicInteger();
                                        ipBuyResultDto.getIplist().forEach(
                                                x -> {
                                                    ipResource.setAddr(x.getIp());
                                                    ipResource.setStatus(3);
                                                    ipResource.setValidTime(Instant.parse(x.getValidTill()).toEpochMilli());
                                                    if (StringUtils.isNotBlank(ipResource.getRegion()) && region.contains(ipResource.getRegion())) {
                                                        ipResource.setProxyUsername(ipResource.getAddr());
                                                        ipResource.setProxyPassword(genRandom(3, 12));
                                                        ipResource.setSpecialLine(true);
                                                    }
                                                    ipResourceRepository.save(ipResource);

                                                    index.getAndIncrement();
                                                }
                                        );
                                        result = true;
                                    }
                                } catch (Exception e) {
                                    log.error(e.getMessage());
                                    result = false;
                                }
                                if (result == false && (ipResource.getPurchasedTime() < Instant.now().minusSeconds(7200).toEpochMilli())) {
                                    IpChargeRequestDto ipChargeRequestDto = buildIpChargeRequestDto(ipResource, 3, 0);
                                    accountService.chargeByMoney(ipResource.getUsername(), -ipResource.getPrice(), ipChargeRequestDto);
                                    if (ipResource.getShopIds() != null && ipResource.getShopIds().size() > 0) {
                                        ShopRequestDto shopRequestDto = new ShopRequestDto();
                                        shopRequestDto.setIpId(ipResource.getId());
                                        shopRequestDto.setShopIds(ipResource.getShopIds());
                                        ipAndShopService.unBindShops(ipResource.getUsername(), shopRequestDto);
                                    }
                                    ipResourceRepository.deleteById(ipResource.getId());
                                }
                            } finally {
                                long end = System.currentTimeMillis();
                                log.debug("buyIpTask {} execution time is: " + (end - start) / 1000 + "s", ipResource.getId());
                                try {
                                    ipResourceRepository.unLockTask(ipResource.getId());
                                } catch (Throwable th) {
                                    log.error("unlock failed", th);
                                    //try again
                                    ipResourceRepository.unLockTask(ipResource.getId());
                                }
                            }
                        }
                    },
                    ThreadPoolUtils.buyIpTasksPool
            );

            futureList.add(future);
        }
        CompletableFuture.allOf(futureList.toArray(new CompletableFuture[0])).join();
    }

    @Scheduled(cron = "0 0/1 * * * ?")
    public void queryIpTasks() {
        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        long time = Instant.now().minusSeconds(300).toEpochMilli();
        List<IpResource> ipResources = ipResourceRepository.sampleTasks(3, time);
        for (IpResource ipResource : ipResources) {
            long start = System.currentTimeMillis();
            CompletableFuture.runAsync(
                    () -> {
                        if (ipResourceRepository.lockTask(ipResource)) {
                            try {
                                String url = URL + "/ecc/ipinfo?accountId=browser&ip=" + ipResource.getAddr();
                                Map<String, String> header = buildGetHeader();
                                String rs = HttpClientutils.doGet(url, header);
                                IpInfoResultDto ipInfoResultDto = JSONObject.parseObject(rs, IpInfoResultDto.class);
                                if (ipInfoResultDto != null && StringUtils.isBlank(ipInfoResultDto.getErrorCode())) {
                                    if (StringUtils.isNotEmpty(ipInfoResultDto.getStatus())) {
                                        ipResourceRepository.updateStatus(ipResource.getId(), 0);
                                    }
                                }
                            } catch (Exception e) {
                                log.error(e.getMessage(), e);
                                NotifyUtils.sendMessage("浏览器后端 queryIpTasks() 又炸了，赶紧看啊", e, NotifyUtils.MsgType.WEBHOOK);
                            } finally {
                                long end = System.currentTimeMillis();
                                log.debug("queryIpTask {} execution time is: " + (end - start) / 1000 + "s", ipResource.getId());
                                try {
                                    ipResourceRepository.unLockTask(ipResource.getId());
                                } catch (Throwable th) {
                                    log.error("unlock failed", th);
                                    //try again
                                    ipResourceRepository.unLockTask(ipResource.getId());
                                }
                            }
                        }
                    },
                    ThreadPoolUtils.queryIpTasksPool
            );
        }
    }

    @Scheduled(cron = "0 0/5 * * * ?")
    public void healthCheck() {
        // 1. 随机获取未被删除的 20 个 ip 做健康检查
        List<IpResource> ipResources = ipResourceRepository.sampleTasks(Arrays.asList(0, 2));
        for (IpResource ipResource : ipResources) {
            long start = System.currentTimeMillis();
            // todo ipResourceRepository.healthLock(ipResource) 始终为 true，感觉有些多余
            if (ipResourceRepository.healthLock(ipResource)) {
                try {
                    int failTime = 0;
                    // 获取查询 ip 地址的 5个网址
                    List<QueryIpUrlList> queryIpUrlLists = queryIpUrlListRepository.findAll();
                    final int maxRetry = Math.max(queryIpUrlLists.size(), 5);

                    // 如果 ip 已经失效，且 ip 的状态不是 3:正在分配 和 6:未分配，则将 ip 的状态设置为 1 已过期
                    if (ipResource.getValidTime() <= Instant.now().toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                        ipResource.setStatus(1);
                        ipResourceRepository.save(ipResource);

                        // 如果 ip 为专线 ip，或者 ip 为用户自己的，但是使用了系统的专线功能
                    } else if (ipResource.isSpecialLine() || (ipResource.getIpType() == IpType.OWN && ipResource.isUsingSpecialLine())) {

                        Trans trans = new Trans(ipResource.getProxyUsername(), ipResource.getProxyPassword());
                        String sp_result = trans.get(getNextUrl(queryIpUrlLists, failTime).getUrl());
                        while (!sp_result.contains(ipResource.getAddr())) {
                            if (failTime > maxRetry) {
                                NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 专线代理异常:" + ipResource.getIpType(), NotifyUtils.MsgType.WEBHOOK);
                                log.error("防关联浏览器 ip " + ipResource.getAddr() + " 专线代理异常 " + sp_result);
                                break;
                            }
                            failTime++;
                            Thread.sleep(2000);
                            sp_result = trans.get(getNextUrl(queryIpUrlLists, failTime).getUrl());
                        }

                        // 如果 ip 类型为 vendor
                    } else if (ipResource.getIpType() == IpType.VENDOR) {
                        Trans trans = new Trans(ipResource.getAddr(), Integer.valueOf(ipResource.getPort().size() > 1 ? ipResource.getPort().get(1) : ipResource.getPort().get(0)), ipResource.getUsername(), ipResource.getPassword());
                        String result = trans.get(getNextUrl(queryIpUrlLists, failTime).getUrl());
                        if (!result.contains(ipResource.getAddr())) {
                            while (!result.contains(ipResource.getAddr())) {
                                if (failTime > maxRetry) {
                                    NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 代理异常:" + ipResource.getIpType(), NotifyUtils.MsgType.WEBHOOK);
                                    log.error("防关联浏览器 ip " + ipResource.getAddr() + " 代理异常 " + result);
                                    break;
                                }
                                failTime++;
                                Thread.sleep(2000);
                                result = trans.get(getNextUrl(queryIpUrlLists, failTime).getUrl());
                            }
                        }
                    }
                } catch (Exception e) {
                    // todo 将代理异常改为 健康检查异常
                    NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 健康检查异常", e, NotifyUtils.MsgType.WEBHOOK);
                    log.error(e.getMessage(), e);
                } finally {
                    long end = System.currentTimeMillis();
                    log.debug("queryIpTask {} execution time is: " + (end - start) / 1000 + "s", ipResource.getId());
                    try {
                        ipResourceRepository.unLockHealth(ipResource.getId());
                    } catch (Throwable th) {
                        log.error("unlock failed", th);
                        //try again
                        ipResourceRepository.unLockHealth(ipResource.getId());
                    }
                }
            }
        }
    }


    private static QueryIpUrlList getNextUrl(List<QueryIpUrlList> queryIpUrlLists, int count) {
        if (CollectionUtils.isEmpty(queryIpUrlLists)) {
            return null;
        }
        int size = queryIpUrlLists.size();
        int index = count % size;
        return queryIpUrlLists.get(index);
    }


    @Scheduled(cron = "0 0 0 * * ?")
    public void countIp() {
        List<Account> accounts = accountRepository.findAll();
        try {
            accounts.forEach(
                    x -> {
                        IpCountRecord ipCountRecord = new IpCountRecord();
                        ipCountRecord.setUsername(x.getName());
                        ipCountRecord.setTimestamp(Instant.now().toEpochMilli());
                        long ipcount_using = ipResourceRepository.countAllByOwnerAndIsDeletedAndValidTimeGreaterThan(x.getName(), false, Instant.now().toEpochMilli());
                        long ipcount_all = ipResourceRepository.countAllByOwner(x.getName());
                        ipCountRecord.setIp_all(ipcount_all);
                        ipCountRecord.setIp_using(ipcount_using);
                        ipCountRecordRepository.save(ipCountRecord);
                    }
            );
            IpCountRecord ipCountRecord = new IpCountRecord();
            ipCountRecord.setUsername("all");
            ipCountRecord.setTimestamp(Instant.now().toEpochMilli());
            long ipcount_using = ipResourceRepository.countAllByIsDeletedAndValidTimeGreaterThan(false, Instant.now().toEpochMilli());
            long ipcount_all = ipResourceRepository.count();
            ipCountRecord.setIp_using(ipcount_using);
            ipCountRecord.setIp_all(ipcount_all);
            ipCountRecordRepository.save(ipCountRecord);
        } catch (Exception e) {
            log.error("统计IpCount error", e);
        }

    }

    public static class HttpClientutils {
        static OkHttpClient.Builder builder = new OkHttpClient.Builder();

        static {
            builder.connectTimeout(300, TimeUnit.SECONDS);
            builder.readTimeout(300, TimeUnit.SECONDS);
        }

        static OkHttpClient client = new OkHttpClient(builder);

        public static String doGet(String url, Map<String, String> headers) throws IOException {
            Headers httpHeaders = Headers.of(headers);
            Request request = new Request.Builder()
                    .get()
                    .url(url)
                    .headers(httpHeaders)
                    .build();
            Call call = client.newCall(request);
            Response response = call.execute();
            ResponseBody responseBody = response.body();
            return responseBody.string();
        }

        public static String doPost(String url, Map<String, String> headers, String body) throws IOException {
            Headers httpHeaders = Headers.of(headers);
            Request request = new Request.Builder()
                    .post(RequestBody.create(body, okhttp3.MediaType.get("application/json")))
                    .url(url)
                    .headers(httpHeaders)
                    .build();
            Call call = client.newCall(request);
            Response response = call.execute();
            ResponseBody responseBody = response.body();
            String rs = responseBody.string();
            response.close();
            return rs;
        }
    }
}
