package com.edgec.browserbackend.account.service.impl;

import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.dto.AccountPromotionDto;
import com.edgec.browserbackend.account.dto.BillQueryResultDto;
import com.edgec.browserbackend.account.dto.IpCountQueryResultDto;
import com.edgec.browserbackend.account.dto.PromotionQueryResultDto;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.*;
import com.edgec.browserbackend.account.service.AdministratorService;
import com.edgec.browserbackend.auth.service.UserService;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.*;


@Service
@ComponentScan("com.edgec.browserbackend.account.repository")
public class AdministratorServiceImpl implements AdministratorService {

    private final Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    private AdministratorRepository administratorRepository;

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private UserPrePaidBillingRepository userPrePaidBillingRepository;

    @Autowired
    private UserBalanceRepository userBalanceRepository;

    @Autowired
    private UserBillingRepository userBillingRepository;

    @Autowired
    private UserService userService;

    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Autowired
    private WhiteSiteRepository whiteSiteRepository;

    @Override
    public Administrator createAdministrator(Administrator administrator) {
        Administrator administrator1 = new Administrator();
        if(StringUtils.isEmpty(administrator.getName())){
            throw new ClientRequestException(AccountErrorCode.NAMEEMPTY, "userName connot be empty " );
        }

        administrator1.setName(administrator.getName());
        User newUser = new User();
        newUser.setUsername(administrator.getName());
        newUser.setPassword(administrator.getPassword());
        newUser.setEnabled(true);

        userService.create(new com.edgec.browserbackend.auth.domain.User(newUser));

        administrator1.setPassword("");
        administratorRepository.save(administrator1);

        return administrator1;
    }

    @Override
    public List<Administrator> getAllAdministrator() {
        return administratorRepository.findAll();
    }

    @Override
    public Account getAccountByName(String name) {
        Account account = accountRepository.findByName(name);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);
        return account;
    }

    @Override
    public Account getAccountByPhoneNumber(String phoneNumber) {
        Account account = accountRepository.findByPhoneNumberAndParentIsNull(phoneNumber);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.PHONENOTEXIST, "Can not find account with phone number" + phoneNumber);
        return account;
    }

    @Override
    public Account getAccountByEmail(String email) {
        Account account = accountRepository.findByEmail(email);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.EMAILNOTEXIST, "Email does not exist: " + email);
        return account;
    }


    @Override
    public BillQueryResultDto getUserBillingByName(String name) {
        Account account = accountRepository.findByName(name);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByUsernameAndPayMethodIn(name, Arrays.asList(1,2));
        double parentexpense = 0;
        if (userPrePaidBillings != null)
            parentexpense = userPrePaidBillings.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
        double parentwithdrawn = account.getPromotion().getWithdrawn();
        UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
        double parentbalanced = 0;
        double parentbalanceused = 0;
        if (userBalance != null) {
            parentbalanced = userBalance.getBalanced();
            parentbalanceused = userBalance.getUsed();
        }
        List<UserPrePaidBilling> userPrePaidBillings1 = userPrePaidBillingRepository.findByUsernameAndPayMethod(name, 3);
        double parentbanktransfer = 0;
        if (userPrePaidBillings1 != null)
            parentbanktransfer = userPrePaidBillings1.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();

        double childexpense = 0;
        double childwithdrawn = 0;
        double childbalanced = 0;
        double childbanktransfer = 0;
        double childbalanceused = 0;
        if (account.getParent() == null) {
            List<Account> children = accountRepository.findByParent(account.getName());
            if (children != null && children.size() > 0) {
                for (Account child : children) {
                    List<UserPrePaidBilling> userPrePaidBillings_child = userPrePaidBillingRepository.findByUsernameAndPayMethodIn(child.getName(), Arrays.asList(1,2));
                    if (userPrePaidBillings_child != null)
                        childexpense += userPrePaidBillings_child.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                    UserBalance userBalance_child = userBalanceRepository.findById(child.getName()).orElse(null);
                    if (userBalance_child != null) {
                        childbalanced += userBalance_child.getBalanced();
                        childbalanceused += userBalance_child.getUsed();
                    }
                    List<UserPrePaidBilling> userPrePaidBillings1_child = userPrePaidBillingRepository.findByUsernameAndPayMethod(child.getName(), 3);
                    if (userPrePaidBillings1_child != null)
                        childbanktransfer += userPrePaidBillings1_child.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                }
            }
        }

        BillQueryResultDto billQueryResultDto = new BillQueryResultDto();
        billQueryResultDto.setChildExpense(childexpense);
        billQueryResultDto.setChildBankTransfer(childbanktransfer);
        billQueryResultDto.setChildBalanced(childbalanced);
        billQueryResultDto.setChildWithdraw(childwithdrawn);
        billQueryResultDto.setChildBalanceUsed(childbalanceused);

        billQueryResultDto.setParentExpense(parentexpense);
        billQueryResultDto.setParentBalanced(parentbalanced);
        billQueryResultDto.setParentBankTransfer(parentbanktransfer);
        billQueryResultDto.setParentWithdrawn(parentwithdrawn);
        billQueryResultDto.setParentBalanceUsed(parentbalanceused);

        Account promoter = null;
        if (account.getPromotionCode() != null)
            promoter = accountRepository.findByPromotion(account.getPromotionCode());
        if (promoter != null)
            billQueryResultDto.setPromoter(promoter.getName());

        billQueryResultDto.setPromotion(account.getPromotion());

        return billQueryResultDto;
    }

    @Override
    public Account unLockLockedAccount(String name,Account account) {
        Account newAccount = accountRepository.findByName(name);
        if (newAccount == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);

        User newUser = new User();
        newUser.setUsername(name);
        userService.unlock(new com.edgec.browserbackend.auth.domain.User(newUser),"unlock");

        return newAccount;
    }

    @Override
    public Account lockAbnormalAccount(String name, Account account) {
        Account abnormalAccount = accountRepository.findByName(name);
        if (abnormalAccount == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);

        abnormalAccount.setLockReason(account.getLockReason());

        User abnormalUser = new User();
        abnormalUser.setUsername(name);
        userService.lock(new com.edgec.browserbackend.auth.domain.User(abnormalUser),"lock");

        accountRepository.save(abnormalAccount);

        return abnormalAccount;
    }

    @Override
    public Page<Account> searchAllUserPage(Pageable pageable) {
        Page<Account> accounts = accountRepository.findAll(pageable);
        return accounts;
    }

    @Override
    public Page<UserPrePaidBilling> searchAllUserBillingPage(int page, int year1, int year2, int month1, int month2) {
        Pageable pageable = PageRequest.of(page, 15);
        Page<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findAllByYearBetweenAndMonthBetween(pageable, year1, year2, month1, month2);
        return userPrePaidBillings;
    }

    @Override
    public Page<Account> searchCreateAccountBetween(Pageable pageable, String  strDate1, String strDate2, int isAuthorized ) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        Page<Account> accounts = null;
        try {
            Date dateTime1 = formatter.parse(strDate1);
            Date dateTime2 = formatter.parse(strDate2);
            if (isAuthorized == 5)
                accounts = accountRepository.findAllBySignupDateBetweenAndParentIsNull(pageable, dateTime1, dateTime2);
            else
                accounts = accountRepository.findAllBySignupDateBetweenAndParentIsNullAndAuthorized(pageable, dateTime1, dateTime2, isAuthorized);
        }catch (ParseException e){
            e.printStackTrace();
        }
        return accounts;

    }
    @Override
    public void deleteAdministrator( String name) {
        Administrator administrator = administratorRepository.findByName(name);
        if (administrator == null) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }
        userService.deleteUser(name);
        administratorRepository.delete(administrator);

    }

    @Override
    public Administrator updateAdministrator(String name, String roles) {
        Administrator updating = administratorRepository.findByName(name);
        if (updating == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "can't find adminstrator with name " + name);
        }
        updating.setRole(roles);
        userService.updateRoles(name, roles);
        administratorRepository.save(updating);

        return updating;
    }

    @Override
    public List<CompanyEarningsDto> getCompanyEarningDetail(String stringdate) {
        StringTokenizer stringToInt = new StringTokenizer(stringdate, "-");
        int year = Integer.parseInt(stringToInt.nextToken());
        int month = Integer.parseInt(stringToInt.nextToken());

        List<CompanyEarningsDto> result = new ArrayList<>();

        List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByYearAndMonth(year, month);
        if (userPrePaidBillings != null && userPrePaidBillings.size() > 0) {
            for (UserPrePaidBilling userPrePaidBilling : userPrePaidBillings) {
                CompanyEarningsDto companyEarningsDto = new CompanyEarningsDto();

                companyEarningsDto.setUsername(userPrePaidBilling.getUsername());
                companyEarningsDto.setTotalBillingCost(userPrePaidBilling.getTotal());
                companyEarningsDto.setPayStatus(userPrePaidBilling.getStatus());

                Account account = accountRepository.findByName(userPrePaidBilling.getUsername());

                companyEarningsDto.setCompanyShouldEarning(userPrePaidBilling.getTotal());

                if (userPrePaidBilling.getStatus().equals(BillStatus.PAID)) {
                    companyEarningsDto.setCompanyActualEarning(userPrePaidBilling.getTotal());
                } else {
                    companyEarningsDto.setCompanyActualEarning(0);
                }

                result.add(companyEarningsDto);

            }
        }

        return result;
    }

    @Override
    public List<UserPrePaidBilling> getServiceFeeOwedUserInfo() {

        List<UserPrePaidBilling> result = userPrePaidBillingRepository.findByStatus(BillStatus.UNPAID);

        return result;
    }

    @Override
    public boolean getUserLockState(String name) {
        return userService.lockState(name);
    }


    @Override
    public void deleteUser(String username) {
        userService.deleteUser(username);
        accountRepository.deleteById(username);
    }

    @Override
    public UserBalance findUserBalance(String username) {
        return userBalanceRepository.findById(username).orElse(null);
    }

    @Override
    public boolean userbillingTransfer() {
        List<UserBilling> userBillings = userBillingRepository.findByUsername("51tou");
        userBillings.forEach(x ->{
             generateUserPrePaidBillingByUserBilling(x);
        });
        return true;
    }

    private void generateUserPrePaidBillingByUserBilling(UserBilling userBilling) {
        UserPrePaidBilling userPrePaidBilling = new UserPrePaidBilling();
        userPrePaidBilling.setYear(userBilling.getYear());
        userPrePaidBilling.setMonth(userBilling.getMonth());
        userPrePaidBilling.setServices(Services.INTELLIGROUP);
        if (userBilling.getStatus() != BillStatus.PAID) {
            userPrePaidBilling.setStatus(BillStatus.UNPAID);
        } else {
            userPrePaidBilling.setStatus(BillStatus.PAID);
        }
        userPrePaidBilling.setDeductionRecords(userBilling.getDeductionRecords());
        userPrePaidBilling.setTotal(userBilling.getTotalBillingCost());
        userPrePaidBilling.setTimestamp(ZonedDateTime.of(userBilling.getYear(), userBilling.getMonth(), 1, 0, 0, 0, 0, ZoneOffset.systemDefault()).toEpochSecond() * 1000);
        userPrePaidBilling.setPrepaid(false);
        userPrePaidBilling.setChargeType(9);
        userPrePaidBilling.setUnit("month");
        userPrePaidBilling.setUsername(userBilling.getUsername());
        userPrePaidBillingRepository.save(userPrePaidBilling);

    }

    @Override
    public void addPromotionCode(String username, String promotionCode) {
        Account account = accountRepository.findByName(username);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        account.setPromotionCode(promotionCode);
        accountRepository.save(account);
    }

    @Override
    public IpCountQueryResultDto queyrIpCount(String username) {
        IpCountQueryResultDto ipCountQueryResultDto = new IpCountQueryResultDto();
        if (StringUtils.isBlank(username)) {
            long ipcount_using = ipResourceRepository.countAllByIsDeleted(false);
            long ipcount_all = ipResourceRepository.count();
            ipCountQueryResultDto.setIpCount_using(ipcount_using);
            ipCountQueryResultDto.setIpCount_all(ipcount_all);
        } else {
            long ipcount_using = ipResourceRepository.countAllByOwnerAndIsDeleted(username, false);
            long ipcount_all = ipResourceRepository.countAllByOwner(username);
            ipCountQueryResultDto.setIpCount_using(ipcount_using);
            ipCountQueryResultDto.setIpCount_all(ipcount_all);
        }
        return ipCountQueryResultDto;
    }

    @Override
    public PromotionQueryResultDto queryPromotion(Pageable pageable, String username, String promotionCode, String strDate1, String strDate2) {
        if (StringUtils.isBlank(username) && StringUtils.isBlank(promotionCode))
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        Account account = null;
        if (!StringUtils.isBlank(username))
            account = accountRepository.findByName(username);
        else account = accountRepository.findByPromotion(promotionCode);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getParent() != null)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        PromotionQueryResultDto promotionQueryResultDto = new PromotionQueryResultDto(account);
        promotionQueryResultDto.setUsername(username);
        try {
            Date dateTime1 = formatter.parse(strDate1);
            Date dateTime2 = formatter.parse(strDate2);
            Page<Account> accountPage = accountRepository.findAllBySignupDateBetweenAndPromotionCodeAndParentIsNull(pageable, dateTime1, dateTime2, account.getPromotion().getCode());
            List<AccountPromotionDto> accountPromotionDtos = new ArrayList<>();
            if (accountPage.getNumberOfElements() > 0) {
                accountPage.get().forEach(account1 -> {
                    accountPromotionDtos.add(new AccountPromotionDto(account1));
                });
            }

            Page<AccountPromotionDto> accountPromotionDtoPage = new PageImpl<AccountPromotionDto>(accountPromotionDtos);
            promotionQueryResultDto.setUsers(accountPromotionDtoPage);

            List<Account> accounts = accountRepository.findAllBySignupDateBetweenAndPromotionCodeAndParentIsNull(dateTime1, dateTime2, account.getPromotion().getCode());
            Promotion promotion = new Promotion();
            Account finalAccount = account;
            if (accounts != null && accounts.size() > 0) {
                accounts.forEach(x -> {
                    promotion.setInvitedUsers(promotion.getInvitedUsers() + 1);
                    List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByAdministratorAndPayMethodInAndTimestampBetween(x.getName(), Arrays.asList(1,2,3), dateTime1.getTime(), dateTime2.getTime());
                    double totalCommission = 0;
                    if (userPrePaidBillings != null && userPrePaidBillings.size() > 0)
                        totalCommission = userPrePaidBillings.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                    promotion.setTotalCommission(promotion.getTotalCommission() + (int)totalCommission);
                    promotion.setCommission(promotion.getCommission() + x.getPromotion().getCommission());
                    double secondCommission = 0;
                    if (finalAccount.getPromotion().isSale() && x.getParent() == null) {
                        List<Account> secondPromotes = accountRepository.findByPromotionCode(x.getPromotion().getCode());
                        if (secondPromotes != null && secondPromotes.size() > 0) {
                            for (Account secondPromote : secondPromotes) {
                                List<UserPrePaidBilling> userPrePaidBillings1 = userPrePaidBillingRepository.findByAdministratorAndPayMethodInAndTimestampBetween(secondPromote.getName(), Arrays.asList(1,2,3), dateTime1.getTime(), dateTime2.getTime());
                                if (userPrePaidBillings1 != null && userPrePaidBillings1.size() > 0)
                                    secondCommission += userPrePaidBillings1.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                            }
                        }
                    }
                    if (finalAccount.getPromotion().isSale())
                        promotion.setAllGift(promotion.getAllGift() + totalCommission * 0.1 + secondCommission * 0.02);
                    else
                        promotion.setAllGift(promotion.getAllGift() + totalCommission * 0.08);
                    promotion.setWithdrawn(promotion.getWithdrawn() + secondCommission);
                    promotion.setGift(promotion.getGift() + x.getPromotion().getGift());
                    promotion.setCommissionLastMonth(promotion.getCommissionLastMonth() + x.getPromotion().getCommissionLastMonth());
                });
            }
            promotionQueryResultDto.setPromotion(promotion);

            return promotionQueryResultDto;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void addWhiteList(String website) {
        WhiteSite whiteSite = new WhiteSite();
        whiteSite.setWhite(website);
        whiteSiteRepository.save(whiteSite);
    }
}









