package com.edgec.browserbackend.browser.service.Impl;

import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.BindHistory;
import com.edgec.browserbackend.browser.domain.IpResource;
import com.edgec.browserbackend.browser.domain.Shop;
import com.edgec.browserbackend.browser.domain.UserShop;
import com.edgec.browserbackend.browser.dto.ShopRequestDto;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.browser.repository.ShopRepository;
import com.edgec.browserbackend.browser.repository.UserShopRepository;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

@Service
public class IpAndShopServiceImpl implements IpAndShopService {

    private final Logger logger = LoggerFactory.getLogger(IpAndShopServiceImpl.class);

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private UserShopRepository userShopRepository;

    @Autowired
    private ShopRepository shopRepository;

    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Override
    public void bindShop(String username, ShopRequestDto shopRequestDto) {
        // 1. 根据id 来获取账户
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        String shopId = shopRequestDto.getShopId();
        // 2.获取当前账户下指定的商铺
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        // 3.根据shopId获取商铺的详情
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));

        // 4.根据传入的id或者addr查询是否有指定的 ip 资源
        IpResource ipResource = null;
        if (StringUtils.isNotBlank(shopRequestDto.getIpAddr())) {
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(shopRequestDto.getIpAddr(), false);
        }
        if (StringUtils.isNotBlank(shopRequestDto.getIpId())) {
            ipResource = ipResourceRepository.findByIdAndIsDeleted(shopRequestDto.getIpId(), false);
        }
        if (ipResource == null) {
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        }

        // 5.根据shopId来查询指定 isDeleted 为 false 的 ip资源
        IpResource bind = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shopId, false);
        if (bind != null) {
            throw new ClientRequestException(BrowserErrorCode.SHOP_BINDED);
        }

        try {
            // 6.绑定ip，将shopId添加到ip资源中，且将 bind 状态设置为 true
            ipResourceRepository.addShopId(ipResource.getId(), shopId);
        } catch (Exception e) {
            logger.error("fail to bind shop and ip", e);
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void unBindShop(String username, ShopRequestDto shopRequestDto) {
        // 1. 检查当前账户是否存在，是否具备操作权限
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        String shopId = shopRequestDto.getShopId();

        // 2. 查找当前用户是否已经绑定商铺 以及 关联商铺的绑定 （一个商铺可以被多个用户持有）
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }

        // 3. 查找商铺信息
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));

        // 4. 查找当前商铺绑定的 ip 资源是否存在
        IpResource ipResource = null;
        if (StringUtils.isNotBlank(shopRequestDto.getIpAddr())) {
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(shopRequestDto.getIpAddr(), false);
        }
        if (StringUtils.isNotBlank(shopRequestDto.getIpId())) {
            ipResource = ipResourceRepository.findByIdAndIsDeleted(shopRequestDto.getIpId(), false);
        }
        if (ipResource == null) {
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        }

        // 5. 若存在，ip资源是否绑定了该店铺（一个ip资源可以被多个店铺绑定）
        if (ipResource.getShopIds() == null && !ipResource.getShopIds().contains(shop.getShopId())) {
            throw new ClientRequestException(BrowserErrorCode.IPNOTBINDTOSHOP);
        }


        // 封装 ip 资源绑定的历史信息
        BindHistory bindHistory = new BindHistory();
        bindHistory.setIp(ipResource.getAddr());
        bindHistory.setPlatform(shop.getShopPlatform());
        bindHistory.setShopName(shop.getShopName());
        bindHistory.setUnbindTime(ZonedDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));

        try {
            // 6. IP资源解除绑定的店铺并更新ip资源的绑定历史
            ipResourceRepository.deleteShopId(ipResource.getId(), shopId, bindHistory);

            // 7. 更新 ip 资源的绑定状态为 未绑定
            IpResource newIp = ipResourceRepository.findById(ipResource.getId()).orElse(null);
            if (newIp.getShopIds() == null || !newIp.getShopIds().contains(shopId)) {
                ipResourceRepository.updateBind(ipResource.getId(), false);
            }
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void unBindShops(String username, ShopRequestDto shopRequestDto) {
        List<String> shopIds = null;
        if (shopRequestDto.getShopIds() != null && shopRequestDto.getShopIds().size() > 0)
            shopIds = shopRequestDto.getShopIds();
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        IpResource ipResource = null;
        if (StringUtils.isNotBlank(shopRequestDto.getIpAddr())) {
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(shopRequestDto.getIpAddr(), false);
        } else if (StringUtils.isNotBlank(shopRequestDto.getIpId())) {
            ipResource = ipResourceRepository.findByIdAndIsDeleted(shopRequestDto.getIpId(), false);
        }
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        for (String shopId : shopIds) {
            UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
            if (account.getPermission() < 4 || userShop == null) {
                continue;
            }
            Shop shop = shopRepository.findById(shopId).orElse(null);
            if (shop == null)
                continue;
            if (ipResource.getShopIds() == null && !ipResource.getShopIds().contains(shop.getShopId()))
                continue;
            try {
                BindHistory bindHistory = new BindHistory();
                bindHistory.setIp(ipResource.getAddr());
                bindHistory.setPlatform(shop.getShopPlatform());
                bindHistory.setShopName(shop.getShopName());
                bindHistory.setUnbindTime(ZonedDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
                ipResourceRepository.deleteShopId(ipResource.getId(), shopId, bindHistory);
                IpResource newIp = ipResourceRepository.findById(ipResource.getId()).orElse(null);
                if (newIp.getShopIds() == null || newIp.getShopIds().size() == 0)
                    ipResourceRepository.updateBind(ipResource.getId(), false);
            } catch (Exception e) {
                logger.error("fail to unbind", e.getMessage());
                throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
            }
        }
    }
}
