package com.edgec.browserbackend.auth.config;


import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.account.repository.OtpRepository;
import com.edgec.browserbackend.auth.repository.UserRepository;
import com.edgec.browserbackend.auth.service.MongoTokenStore;
import com.edgec.browserbackend.auth.service.security.MongoUserDetailsService;
import java.util.ArrayList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.crypto.password.NoOpPasswordEncoder;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.config.annotation.configurers.ClientDetailsServiceConfigurer;
import org.springframework.security.oauth2.config.annotation.web.configuration.AuthorizationServerConfigurerAdapter;
import org.springframework.security.oauth2.config.annotation.web.configuration.EnableAuthorizationServer;
import org.springframework.security.oauth2.config.annotation.web.configurers.AuthorizationServerEndpointsConfigurer;
import org.springframework.security.oauth2.config.annotation.web.configurers.AuthorizationServerSecurityConfigurer;
import org.springframework.security.oauth2.provider.CompositeTokenGranter;
import org.springframework.security.oauth2.provider.OAuth2RequestFactory;
import org.springframework.security.oauth2.provider.TokenGranter;
import org.springframework.security.oauth2.provider.TokenRequest;
import org.springframework.security.oauth2.provider.client.ClientCredentialsTokenGranter;
import org.springframework.security.oauth2.provider.code.AuthorizationCodeServices;
import org.springframework.security.oauth2.provider.code.AuthorizationCodeTokenGranter;
import org.springframework.security.oauth2.provider.implicit.ImplicitTokenGranter;
import org.springframework.security.oauth2.provider.password.ResourceOwnerPasswordTokenGranter;
import org.springframework.security.oauth2.provider.refresh.RefreshTokenGranter;
import org.springframework.security.oauth2.provider.token.AuthorizationServerTokenServices;

/**
 * @author cdov
 */
@Configuration
@EnableAuthorizationServer
public class OAuth2AuthorizationNewConfig extends AuthorizationServerConfigurerAdapter {


    @Autowired
    private MongoTokenStore mongoTokenStore;

    @Autowired
    @Qualifier("authenticationManagerBean")
    private AuthenticationManager authenticationManager;

    @Autowired
    private MongoUserDetailsService userDetailsService;

    @Autowired
    private Environment env;

    @Autowired
    private OAuthResponseExceptionTranslator oAuthResponseExceptionTranslator;

    @Autowired
    private OtpRepository otpRepository;

    @Autowired
    private UserRepository repository;

    @Override
    public void configure(ClientDetailsServiceConfigurer clients) throws Exception {

        // TODO persist clients details

        // @formatter:off
        clients.inMemory()
               .withClient("browser")
               .authorizedGrantTypes("refresh_token", "password","sms")
               .scopes("browser")
               .accessTokenValiditySeconds(7 * 24 * 60 * 60)
               .refreshTokenValiditySeconds(30 * 24 * 60 * 60)
               .and()
               .withClient("cloudam-browser")
               .secret(env.getProperty("ACCOUNT_SERVICE_PASSWORD"))
               .authorizedGrantTypes("client_credentials", "refresh_token")
               .scopes("server");
        // @formatter:on
    }

    @Override
    public void configure(AuthorizationServerEndpointsConfigurer endpoints) throws Exception {
        endpoints
                .tokenStore(mongoTokenStore)
            .tokenGranter(tokenGranter(endpoints))
                .authenticationManager(authenticationManager)
                .userDetailsService(userDetailsService).exceptionTranslator(oAuthResponseExceptionTranslator);
    }

    @Override
    public void configure(AuthorizationServerSecurityConfigurer oauthServer) throws Exception {
        oauthServer
                .tokenKeyAccess("permitAll()")
                .checkTokenAccess("isAuthenticated()")
                .passwordEncoder(NoOpPasswordEncoder.getInstance())
               .allowFormAuthenticationForClients();
    }

    /**
     * 自定义TokenGranter
     */
    private TokenGranter tokenGranter(AuthorizationServerEndpointsConfigurer endpoints) {
        TokenGranter tokenGranter = new TokenGranter() {
            private CompositeTokenGranter delegate;

            @Override
            public OAuth2AccessToken grant(String grantType, TokenRequest tokenRequest) {
                if (delegate == null) {
                    delegate = new CompositeTokenGranter(getDefaultTokenGranters(endpoints));
                }
                return delegate.grant(grantType, tokenRequest);
            }
        };
        return tokenGranter;
    }

    /**
     * 覆盖原来的List<TokenGranter>,方便我们添加自定义的授权方式,比如SMSCodeTokenGranter短信验证码授权
     */
    private List<TokenGranter> getDefaultTokenGranters(AuthorizationServerEndpointsConfigurer endpoints) {
        AuthorizationServerTokenServices tokenServices = endpoints.getDefaultAuthorizationServerTokenServices();
        AuthorizationCodeServices authorizationCodeServices = endpoints.getAuthorizationCodeServices();
        OAuth2RequestFactory requestFactory = endpoints.getOAuth2RequestFactory();
        List<TokenGranter> tokenGranters = new ArrayList<TokenGranter>();
        tokenGranters.add(new AuthorizationCodeTokenGranter(tokenServices,
            authorizationCodeServices, endpoints.getClientDetailsService(), requestFactory));
        tokenGranters.add(new RefreshTokenGranter(tokenServices, endpoints.getClientDetailsService(), requestFactory));
        ImplicitTokenGranter implicit = new ImplicitTokenGranter(tokenServices, endpoints.getClientDetailsService(),
            requestFactory);
        tokenGranters.add(implicit);
        tokenGranters.add(
            new ClientCredentialsTokenGranter(tokenServices, endpoints.getClientDetailsService(), requestFactory));
        if (authenticationManager != null) {
            tokenGranters.add(new ResourceOwnerPasswordTokenGranter(authenticationManager,
                tokenServices, endpoints.getClientDetailsService(), requestFactory));
        }
        tokenGranters.add(new SMSCodeTokenGranter(tokenServices, endpoints.getClientDetailsService(), requestFactory, otpRepository, repository, "sms"));
        return tokenGranters;
    }



}
