package com.edgec.browserbackend.browser.service.Impl;

import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.domain.LoginHistory;
import com.edgec.browserbackend.browser.domain.OperationHistory;
import com.edgec.browserbackend.browser.domain.WinHistory;
import com.edgec.browserbackend.browser.dto.*;
import com.edgec.browserbackend.browser.repository.BrowserWinLogRepository;
import com.edgec.browserbackend.browser.repository.LoginHistoryRepository;
import com.edgec.browserbackend.browser.repository.OperationHistoryRepository;
import com.edgec.browserbackend.browser.service.HistoryService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.time.Instant;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;

@Service
public class HistoryServiceImpl implements HistoryService {

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private LoginHistoryRepository loginHistoryRepository;

    @Autowired
    private OperationHistoryRepository operationHistoryRepository;

    @Autowired
    private BrowserWinLogRepository browserWinLogRepository;


    @Override
    public void addLoginHistory(String username, LoginHistoryDto loginHistoryDto) {
        Account account = accountRepository.findById(username)
                                           .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        int num = loginHistoryRepository.countByUsername(username);
        if (num > 10000) {
            loginHistoryRepository.delete(loginHistoryRepository.findFirstByUsername(username));
        }
        LoginHistory loginHistory = new LoginHistory(loginHistoryDto);
        loginHistory.setLoginTime(Instant.now().toEpochMilli());
        loginHistory.setUsername(username);
        if (account.getParent() == null) {
            loginHistory.setAdministrator(username);
        } else {
            loginHistory.setAdministrator(account.getParent());
        }

        loginHistory.setNickname(account.getNickname());
        loginHistoryRepository.save(loginHistory);
    }

    @Override
    public void addOperationHistory(String username, OperationHistoryDto operationHistoryDto) {
        Account account = accountRepository.findById(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        int num = operationHistoryRepository.countByUsername(username);
        if (num > 10000) {
            operationHistoryRepository.delete(operationHistoryRepository.findFirstByUsername(username));
        }

        OperationHistory operationHistory = new OperationHistory(operationHistoryDto);
        operationHistory.setOperationTime(Instant.now().toEpochMilli());
        operationHistory.setUsername(username);
        operationHistory.setNickname(account.getNickname());
        if (account.getParent() == null) {
            operationHistory.setAdministrator(username);
        } else {
            operationHistory.setAdministrator(account.getParent());
        }
        operationHistoryRepository.save(operationHistory);
    }

    @Override
    public HistoryPageResultDto getLoginHistories(String username, HistoryListRequestDto historyListRequestDto) {
        Account account = accountRepository.findById(username).orElse(null);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getPermission() < 8)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        if (historyListRequestDto.getAmount() > 100)
            historyListRequestDto.setAmount(100);
        Pageable pageable = PageRequest.of(historyListRequestDto.getPage(), historyListRequestDto.getAmount());
        HistoryPageResultDto historyPageResultDto = new HistoryPageResultDto();
        Page<LoginHistory> loginHistoryPage;
        long time = Instant.now().atZone(ZoneOffset.UTC).minusDays(historyListRequestDto.getDay()).toInstant().toEpochMilli();
        if (account.getParent() == null) {
            loginHistoryPage = loginHistoryRepository.findByAdministratorAndLoginTimeGreaterThanOrderByLoginTimeDesc(username, time, pageable);
        } else {
            loginHistoryPage = loginHistoryRepository.findByUsernameAndLoginTimeGreaterThanOrderByLoginTimeDesc(username, time, pageable);
        }

        List<LoginHistoryDto> loginHistoryDtoList = new ArrayList<>();
        loginHistoryPage.getContent().forEach(x -> {
            loginHistoryDtoList.add(new LoginHistoryDto(x));
        });
        historyPageResultDto.setHistory(loginHistoryDtoList);
        PageInfo pageInfo = new PageInfo();
        pageInfo.setCurrentPage(loginHistoryPage.getNumber());
        pageInfo.setTotalPages(loginHistoryPage.getTotalPages());
        pageInfo.setTotalItems((int) loginHistoryPage.getTotalElements());
        historyPageResultDto.setHistoryPage(pageInfo);
        return historyPageResultDto;
    }

    @Override
    public HistoryPageResultDto getOperationHistories(String username, HistoryListRequestDto historyListRequestDto) {
        Account account = accountRepository.findById(username).orElse(null);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getPermission() < 8)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        if (historyListRequestDto.getAmount() > 100)
            historyListRequestDto.setAmount(100);
        Pageable pageable = PageRequest.of(historyListRequestDto.getPage(), historyListRequestDto.getAmount());
        HistoryPageResultDto historyPageResultDto = new HistoryPageResultDto();
        Page<OperationHistory> operationHistories;
        long time = Instant.now().atZone(ZoneOffset.UTC).minusDays(historyListRequestDto.getDay()).toInstant().toEpochMilli();
        if (account.getParent() == null) {
            operationHistories = operationHistoryRepository.findByAdministratorAndOperationTimeGreaterThanOrderByOperationTimeDesc(username, time, pageable);
        } else {
            operationHistories = operationHistoryRepository.findByUsernameAndOperationTimeGreaterThanOrderByOperationTimeDesc(username, time, pageable);
        }

        List<OperationHistoryDto> operationHistoryDtos = new ArrayList<>();
        operationHistories.getContent().forEach(x -> {
            operationHistoryDtos.add(new OperationHistoryDto(x));
        });
        historyPageResultDto.setHistory(operationHistoryDtos);
        PageInfo pageInfo = new PageInfo();
        pageInfo.setCurrentPage(operationHistories.getNumber());
        pageInfo.setTotalPages(operationHistories.getTotalPages());
        pageInfo.setTotalItems((int) operationHistories.getTotalElements());
        historyPageResultDto.setHistoryPage(pageInfo);
        return historyPageResultDto;
    }

    @Override
    public void addBrowserWinLog(String username, WinHistory winHistory) {
        browserWinLogRepository.deleteBeforeWinLog(username, 1);
        browserWinLogRepository.addWinLog(username, winHistory);
    }
}
