package com.edgec.browserbackend.browser.service.Impl;

import com.alibaba.fastjson.JSONObject;
import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.dto.*;
import com.edgec.browserbackend.browser.repository.*;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.utils.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 商铺操作
 *
 * @author JMW
 */
@Transactional
@Service
public class ShopServiceImpl implements ShopService {

    private IpAndShopService ipAndShopService;
    private IpResourceService ipResourceService;
    private ShopRepository shopRepository;
    private AccountRepository accountRepository;
    private UserShopRepository userShopRepository;
    private GroupRepository groupRepository;
    private IpResourceRepository ipResourceRepository;
    private SpecialLineRepository specialLineRepository;
    private ShopUARepository shopUaRepository;

    @Autowired
    public void setIpAndShopService(IpAndShopService ipAndShopService) {
        this.ipAndShopService = ipAndShopService;
    }

    @Autowired
    public void setIpResourceService(IpResourceService ipResourceService) {
        this.ipResourceService = ipResourceService;
    }

    @Autowired
    public void setShopRepository(ShopRepository shopRepository) {
        this.shopRepository = shopRepository;
    }

    @Autowired
    public void setAccountRepository(AccountRepository accountRepository) {
        this.accountRepository = accountRepository;
    }

    @Autowired
    public void setUserShopRepository(UserShopRepository userShopRepository) {
        this.userShopRepository = userShopRepository;
    }

    @Autowired
    public void setGroupRepository(GroupRepository groupRepository) {
        this.groupRepository = groupRepository;
    }

    @Autowired
    public void setIpResourceRepository(IpResourceRepository ipResourceRepository) {
        this.ipResourceRepository = ipResourceRepository;
    }

    @Autowired
    public void setSpecialLineRepository(SpecialLineRepository specialLineRepository) {
        this.specialLineRepository = specialLineRepository;
    }

    @Autowired
    public void setShopUaRepository(ShopUARepository shopUaRepository) {
        this.shopUaRepository = shopUaRepository;
    }

    private final Logger logger = LoggerFactory.getLogger(ShopServiceImpl.class);

    /**
     * 新增店铺
     *
     * @param username      username
     * @param shopResultDto shopResultDto
     * @return ID
     */
    @Override
    public String addShop(String username, ShopResultDto shopResultDto) {
        if (shopResultDto.getGroup() == null) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        if (shopResultDto.getGroup() != null) {
            groupRepository.findById(shopResultDto.getGroup()).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST));
        }
        Account account = judgeAccountForAddShop(username);
        return saveShopAndGetId(account, shopResultDto);
    }

    private final static String XLSX = ".xlsx";

    /**
     * 批量新增店铺
     *
     * @param username username
     * @param file     file
     * @return IDS
     * @throws IOException IOException
     */
    @Override
    public List<String> addShops(String username, MultipartFile file) throws IOException {
        String filename = file.getOriginalFilename();
        if (StringUtils.isEmpty(filename) || filename.length() < 6 || !filename.substring(filename.length() - 5).equals(XLSX)) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE, "文件格式错误");
        }
        Account account = judgeAccountForAddShop(username);
        List<List<Object>> shops = FileUtil.readExcel(file.getInputStream());
        return shops.stream()
                .map(shopObject -> saveShopAndGetId(account, getShopResultDto(username, shopObject)))
                .collect(Collectors.toList());
    }

    private final static int PERMISSION_04 = 4;
    private final static int SHOP_MAX = 10000;

    /**
     * 新增店铺时账户判断
     * 对当前用户的 account 信息进行校验
     *
     * @param username username
     * @return account
     */
    private Account judgeAccountForAddShop(String username) {
        Account account = judgeAccount(username);
        if (account.getPermission() < PERMISSION_04) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        if (account.getShopCount() >= SHOP_MAX) {
            throw new ClientRequestException(AccountErrorCode.SHOPMAX);
        }
        return account;
    }

    /**
     * 账户判断
     *
     * @param username username
     * @return account
     */
    private Account judgeAccount(String username) {
        return accountRepository.findByName(username)
                .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
    }

    /**
     * 保存店铺信息并获取ID
     *
     * @param account       account
     * @param shopResultDto shopResultDto
     * @return ID
     */
    private String saveShopAndGetId(Account account, ShopResultDto shopResultDto) {
        String logs = "【saveShopAndGetId】 ";
        try {
            shopResultDto.setOwner(account.getName());
            Shop shop = new Shop();
            shop.of(shopResultDto);
            shop.setCreateTime(Instant.now().toEpochMilli());

            if (StringUtils.isEmpty(shop.getShopUA())) {
                List<String> uaList = shopUaRepository.findAll().stream()
                        .map(ShopUA::getUaList)
                        .map(String::valueOf)
                        .collect(Collectors.toList());
                shop.setShopUA(uaList.get(new Random().nextInt(uaList.size())));
            }

            return saveUserShopAndAccount(shopRepository.save(shop).getShopId(), shopResultDto.getGroup(), account);
        } catch (Exception e) {
            logger.error("{}, Exception : {}", logs, e.getMessage(), e);
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
    }

    /**
     * 封装 userShop 并保存
     * 更新当前 account 的商铺数
     *
     * @param id       id
     * @param groupId  groupId
     * @param account  account
     */
    private String saveUserShopAndAccount(String id, String groupId, Account account) {
        UserShop userShop = new UserShop();
        userShop.setShopId(id);
        userShop.setUsername(account.getName());
        userShop.setGroupId(groupId);
        userShopRepository.save(userShop);

        account.setShopCount(account.getShopCount() + 1);
        accountRepository.save(account);

        if (!StringUtils.isEmpty(account.getParent())) {
            saveUserShopAndAccount(id,"-1", judgeAccount(account.getParent()));
        }

        return id;
    }

    private final static int SHOP_NAME = 0;
    private final static int SHOP_PLAT_FORM = 1;
    private final static int SHOP_ACCOUNT = 2;
    private final static int SHOP_PASSWORD = 3;

    /**
     * 将Excel文件中的ShopResultDto封装并返回
     * 批量导入店铺时，默认不分组
     *
     * @param username   username
     * @param shopObject shopObject
     * @return shopResultDto
     */
    private ShopResultDto getShopResultDto(String username, List<Object> shopObject) {
        ShopResultDto shopResultDto = new ShopResultDto();
        shopResultDto.setGroup("-1");
        shopResultDto.setOwner(username);
        shopResultDto.setShopName(shopObject.get(SHOP_NAME).toString());
        shopResultDto.setShopPlatform(shopObject.get(SHOP_PLAT_FORM).toString());
        String shopAccount = shopObject.get(SHOP_ACCOUNT).toString();
        String shopPassword = shopObject.get(SHOP_PASSWORD).toString();
        if (!StringUtils.isEmpty(shopAccount)) {
            shopResultDto.setShopAccount(shopAccount);
        }
        if (!StringUtils.isEmpty(shopPassword)) {
            shopResultDto.setShopPassword(shopPassword);
        }
        return shopResultDto;
    }

    @Override
    public String updateShop(String username, ShopResultDto shopResultDto) {
        // 1.校验参数、account 、权限
        if (StringUtils.isEmpty(shopResultDto.getShopId())) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        // 2.查询当前店铺是否已经绑定当前用户
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
        if (userShop == null) {
            shopResultDto.setShopCookie("");
            logger.warn("{} step-1, Exception : {},{}", "【updateShop】 ", username, JSONObject.toJSONString(shopResultDto));
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }

        // 3. 获取旧的店铺信息，并封装新的数据
        Shop shop_old = shopRepository.findById(shopResultDto.getShopId()).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        shop_old = shop_old.of(shopResultDto);

        try {
            // 4. 更新店铺信息
            shopRepository.save(shop_old);
        } catch (Exception e) {
            logger.error("fail to update", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        return shop_old.getShopId();
    }

    @Override
    public void deleteShop(String username, String shopId) {
        // 1. 校验参数、account 、权限、店铺是否已经和用户绑定
        if (shopId == null) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }

        // 2. 获取店铺信息
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));

        // 3. 如果当前店铺已经绑定 未被删除的 ip 资源，则让IP资源先解绑店铺
        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shop.getShopId(), false);
        if (ipResource != null) {
            ShopRequestDto shopRequestDto = new ShopRequestDto();
            shopRequestDto.setShopId(shopId);
            shopRequestDto.setIpId(ipResource.getId());
            ipAndShopService.unBindShop(username, shopRequestDto);
        }

        // 4. 获取当前店铺关联的所有的 usershop信息 并删除数据库中 usershop信息
        List<UserShop> userShops = userShopRepository.findByShopId(shopId);
        boolean result = userShopRepository.deleteByShopId(shopId);
        if (result) {
            // 5. 删除当前店铺，并更新 account信息 todo
            shopRepository.deleteById(shopId);

            // 6. 更新 和店铺关联的所有的 account 的信息
            List<Account> accountList = accountRepository.findByNameIn(userShops.stream().map(UserShop::getUsername).collect(Collectors.toList()));
            for (Account a : accountList) {
                a.setShopCount(account.getShopCount() - 1);
                accountRepository.save(a);
            }
        } else {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
    }

    @Override
    public void transferShop(String username, String shopId, String groupId) {
        // 1. 校验当前用户是否存在账户信息
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        // 2. 校验 shop 是否存在
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        // 3. 校验 分组信息是否存在 以及 group 是否是当前 用户创建
        Group group = groupRepository.findById(groupId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST));
        if (!group.getId().equals("-1") && group.getOwner() != null && !group.getOwner().equals(username)) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        // 4. 获取 usershop 信息 并校验
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(BrowserErrorCode.USER_NOT_BIND_SHOP);
        }


        try {
            // 5.更该 店铺 分组 并保存
            userShop.setGroupId(groupId);
            userShopRepository.save(userShop);
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }

    }

    @Override
    public void assignShops(String username, List<String> shopIds, List<String> users) {
        // 1. 查询当前登录用户信息，并校验是否具备分配权限
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 8) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        logger.info("assign  shops step-1 " + account.getPermission());

        // 2. 校验当前用户是否绑定了对应的 shops，并获取 usershops信息
        List<UserShop> userShops = userShopRepository.findByUsernameAndShopIdIn(username, shopIds);
        if (userShops == null || userShops.size() < 1) {
            throw new ClientRequestException(BrowserErrorCode.USER_NOT_BIND_SHOP);
        }

        logger.info("assign  shops step-2 " + userShops.stream().map(UserShop::getShopId).collect(Collectors.toList()).toArray().toString());

        // 3. 查询 目标 账户是否存在，并获取账户信息。注意:目标账户只能是当前用户的子账户
        List<Account> accounts = accountRepository.findByNameIn(users);
        if (accounts == null || accounts.size() != users.size()) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        for (Account ac : accounts) {
            if (ac.getParent() == null || !ac.getParent().equals(username)) {
                throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
            }
            logger.info("assign  shops step-3 " + ac.getName());
        }

        // 4. 获取 shop 信息 并校验
        Pageable pageable = PageRequest.of(0, 100);
        List<Shop> shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(shopIds, pageable).getContent();
        if (shops == null || shops.size() < 1) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }
        logger.info("assign  shops step-4 " + shops.size());
        try {
            if (shops.size() == 1) {
                // 5. 删除除当前用户外的所有 usershop 信息
                userShopRepository.deleteByShopIdExceptOwner(shops.get(0).getShopId(), shops.get(0).getOwner());
                for (Account account1 : accounts) {
                    // todo 这个地方有点多余
                    UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shops.get(0).getShopId());
                    logger.info("assign  shops step-5.1 " + (userShop1 == null));
                    if (userShop1 != null) {
                        continue;
                    }

                    // 6. 封装 目标账户的 usershop信息并保存
                    userShop1 = new UserShop(account1.getName(), shops.get(0).getShopId(), "-1");
                    userShopRepository.save(userShop1);
                    logger.info("assign  shops step-5.2 " + userShop1.toString());
                }
            } else {
                shops.forEach(
                        shop -> {
                            for (Account account1 : accounts) {
                                UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shop.getShopId());
                                logger.info("assign  shops step-5.1 " + (userShop1 == null));
                                if (userShop1 != null) {
                                    continue;
                                }
                                userShop1 = new UserShop(account1.getName(), shop.getShopId(), "-1");
                                userShopRepository.save(userShop1);
                                logger.info("assign  shops step-5.2 " + userShop1.toString());
                            }
                        }
                );
            }

            logger.info("---------assign  shops-----------");
        } catch (Exception e) {
            logger.error("fail to assign", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    public static List<String> region = Arrays.asList(
            "asiapa", "hongkong", "japan", "s-korea", "us", "malaysia",
            "yajiada", "singapore", "australia", "germany", "uk", "brazil",
            "moscow", "canada", "france", "sweden", "s-korea", "india", "meast",
            "brazil", "virginia", "ohio", "california", "oregon", "ireland", "london", "ireland"
    );

    @Override
    public ShopResultDto queryShop(String username, String shopId) {
        // 1. 校验 账户、商铺、IP资源、usershop 信息
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shopId, false);
        if (ipResource == null) {
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        // 2. 封装 ip 资源信息
        IpResourceDto ipResourceDto = null;
        if (ipResource.isSpecialLine()) {
            SpecialLine specialLine = specialLineRepository.findAll().get(0);
            ipResourceDto = new IpResourceDto(ipResource, null, false, specialLine);

        } else {
            ipResourceDto = new IpResourceDto(ipResource, null, false);
        }

        //海外设置KCP端口
        if (org.apache.commons.lang3.StringUtils.isNotBlank(ipResource.getRegion()) && region.contains(ipResource.getRegion())) {
            ipResourceDto.setSecondaryProxyPort("20020");
        }

        //刚购买14分钟内的专线设置为不可用
        if (ipResource.getStatus() == 0 && ipResource.getPurchasedTime() > (Instant.now().minusSeconds(14 * 60).toEpochMilli())) {
            ipResource.setSpecialLine(false);
        }

        // 3. 封装 ShopResultDto 信息并返回
        return ShopResultDto.of(shop, userShop.getGroupId(), ipResourceDto);
    }

    @Override
    public List<Shop> getShopListByOwner(String owner, String shopName) {
        return shopRepository.findByOwnerAndShopName(owner, shopName);
    }

    @Override
    public String updateShopUa(String id, String shopUa) {
        Shop shop = shopRepository.findById(id).orElseThrow(
                () -> new ClientRequestException(BrowserErrorCode.SHOP_ERROR, "shop not exists!"));
        shop.setShopUA(shopUa);
        return shopRepository.save(shop).getShopId();
    }


    @Override
    public ShopPageResultDto getShopList(String username, String groupId, int pageNum, int amount, ShopFilterDto shopFilterDto, String tag) {
        long start = System.currentTimeMillis();
        // 1. 校验当前登录用户的账户是否存在
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        logger.info("getshoplist step-1:{},{}", username, System.currentTimeMillis() - start);

        // 2. 如有有分组校验当前查询的分组信息是否正确
        start = System.currentTimeMillis();
        Group group = null;
        if (groupId != null) {
            group = groupRepository.findById(groupId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST));
            if (group.getOwner() != null && !group.getOwner().equals(username)) {
                throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
            }
        }
        logger.info("getshoplist step-2:{},{}", username, System.currentTimeMillis() - start);

        // 3. 根据 groupId 与 username 来查询 shopIds (如果当前用户是父账户，则查询结果包含子账户的shopId)
        start = System.currentTimeMillis();
        List<String> allIds = null;
        if (!"penghai".equals(tag)) {
            if ("-1".equals(groupId) || "-2".equals(groupId) ) {
                allIds = userShopRepository.findByUsername(username).stream().map(UserShop::getShopId).collect(Collectors.toList());
            }
            if (!"-1".equals(groupId) && !"-2".equals(groupId)) {
                // 如果分组
                allIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream().map(UserShop::getShopId).collect(Collectors.toList());
            }
        }

        if ("penghai".equals(tag)) {
            allIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream().map(UserShop::getShopId).collect(Collectors.toList());
        }
        logger.info("getshoplist step-3:{},{}", username, System.currentTimeMillis() - start);

        // 4. 根据传入的过滤条件得到 shopIds
        start = System.currentTimeMillis();
        List<String> shopIds = null;
        if (shopFilterDto.getBindIp() == 0) {
            shopIds = allIds;
        }
        // 绑定了 ip 资源的店铺
        if (shopFilterDto.getBindIp() == 1) {
            shopIds = ipResourceRepository.findShopIdInList(allIds, false).stream()
                    .flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
        }
        // 未绑定 ip 资源的店铺
        if (shopFilterDto.getBindIp() == 2) {
            // 店铺绑定的 ip 资源被删除的 店铺
            for (String id : allIds) {
                IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, true);
                if (ipResource != null) {
                    shopIds.add(id);
                }
            }
        }

        // 临时试用一下
        if (shopFilterDto.getBindIp() == 3) {
            List<String> shopIds2 = ipResourceRepository.findShopIdInList(allIds, false).stream()
                    .flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
            allIds.removeAll(shopIds2);
            shopIds = allIds;
        }
        logger.info("getshoplist step-4:{},{}", username, System.currentTimeMillis() - start);

        amount = amount > 100 ? 100 : amount;
        Pageable pageable = PageRequest.of(pageNum, amount);
        // 5. 根据过滤后的商铺ids 与 其他过滤条件 得到商铺信息并分页
        start = System.currentTimeMillis();
        Page<Shop> shops = getShopsByFilter(shopFilterDto, shopIds, pageable);
        logger.info("getshoplist step-5:{},{}", username, System.currentTimeMillis() - start);

        // 6. 封装 shopResultDtos 信息并返回
        start = System.currentTimeMillis();
        ShopPageResultDto<ShopResultDto> shopPageResultDto = new ShopPageResultDto<>();
        if (shops != null && shops.getNumberOfElements() >= 1) {
            List<ShopResultDto> shopResultDtos = new ArrayList<>();

            SpecialLine specialLine = specialLineRepository.findAll().get(0);
            List<IpResource> pageIpResourceListToSave = new ArrayList<>();

            shops.getContent().stream().forEach(
                    // 设置ip资源状态 并 封装 shopResultDto信息
                    x -> {
                        long start1 = System.currentTimeMillis();
                        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(x.getShopId(), false);
                        // 如果 ip资源非空 且 addr 也非空
                        if (ipResource != null && !StringUtils.isEmpty(ipResource.getAddr())) {
                            // 1. ip资源在未来七天内到期 且 ip 资源的状态不是 3（正在分配）、5（已失效）、6（未分配），则将 ip 资源设置为 2（即将过期）
                            if (ipResource.getValidTime() <= Instant.now().plusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getValidTime() > Instant.now().toEpochMilli()) {
                                if (ipResource.getStatus() != 5 && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                    ipResource.setStatus(2);
                                    //ipResourceRepository.save(ipResource);
                                    pageIpResourceListToSave.add(ipResource);
                                }

                                // 2. ip资源在七天前到期，且 ip 资源的状态不是 3（正在分配）、6（未分配），则删除 ip 资源
                            } else if (ipResource.getValidTime() <= Instant.now().minusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                IpResourceRequestDto ipResourceRequestDto1 = new IpResourceRequestDto();
                                if (ipResource.getIpType() == IpType.VENDOR) {
                                    ipResourceRequestDto1.setAddr(Arrays.asList(ipResource.getAddr()));
                                } else {
                                    ipResourceRequestDto1.setIpId(Arrays.asList(ipResource.getId()));
                                }

                                try {
                                    ipResourceService.deleteExpiredIp(username, ipResourceRequestDto1);
                                } catch (Exception e) {
                                    logger.error(e.getMessage(), e);
                                }

                                return;

                                // 3. ip资源到期，且 ip 资源的状态不是 3（正在分配）、6（未分配），则设置 ip 资源的状态为 1（已过期）
                            } else if (ipResource.getValidTime() <= Instant.now().toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                ipResource.setStatus(1);
                                //ipResourceRepository.save(ipResource);
                                pageIpResourceListToSave.add(ipResource);

                                // 4. 其他，将ip资源状态设置为 未使用
                            } else {
                                if ((ipResource.getStatus() == 0 || ipResource.getStatus() == 1 || ipResource.getStatus() == 2) && ipResource.getVendor() != Vendor.local) {
                                    ipResource.setStatus(0);
                                } else if (ipResource.getIpType().equals(IpType.LOCAL) && (ipResource.getStatus() == 1 || ipResource.getStatus() == 2) && ipResource.getAddr().equals("本地Ip未使用")) {
                                    ipResource.setStatus(4);
                                }
                                // ipResourceRepository.save(ipResource);
                                pageIpResourceListToSave.add(ipResource);
                            }
                            if("-2".equals(groupId)){
                                return;
                            }
                        }

                        //ShopResultDto shopResultDto = getShopResultDto(username, x, ipResource, specialLine);
                        ShopResultDto shopResultDto = getShopResultDto(groupId, x, ipResource, specialLine);

                        shopResultDtos.add(shopResultDto);
                        logger.info("getshoplist step-6.x:{},{},{}", username, System.currentTimeMillis() - start1, Thread.currentThread().getName());
                    }
            );
            ipResourceRepository.saveAll(pageIpResourceListToSave);

            Page<ShopResultDto> shopDtoPage = new PageImpl<>(shopResultDtos, pageable, shopIds.size());
            shopPageResultDto.setShopList(shopDtoPage.getContent());
            PageInfo pageInfo = new PageInfo(shopDtoPage.getPageable().getPageNumber(), shopDtoPage.getTotalPages(), shopIds.size());
            shopPageResultDto.setShopPage(pageInfo);
        }
        logger.info("getshoplist step-6:{},{}", username, System.currentTimeMillis() - start);
        return shopPageResultDto;
    }

    @Override
    public ShopSummary getShopSummary(String username) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        ShopSummary shopSummary = new ShopSummary();
        List<String> allShopIds = userShopRepository.findByUsername(username).stream()
                .map(UserShop::getShopId).collect(Collectors.toList());
        List<String> unbind = new ArrayList<>();
        if (!allShopIds.isEmpty()) {
            for (String id : allShopIds) {
                IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, false);
                if (ipResource == null) {
                    unbind.add(id);
                }
            }
        }
        shopSummary.setUnbind(unbind.size());

        List<String> shopIds = userShopRepository.findByUsername(username).stream()
                .map(UserShop::getShopId).collect(Collectors.toList());
        List<String> bind = ipResourceRepository.findShopIdInList(shopIds, false).stream()
                .map(IpResource::getId).collect(Collectors.toList());
        int expired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(1, bind, false);
        int willexpired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(2, bind, false);
        shopSummary.setExpired(expired);
        shopSummary.setWillExpire(willexpired);
        shopSummary.setTotal(shopIds.size());
        return shopSummary;
    }

    @Override
    public List<String> getShopUsers(String username, String shopId) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getParent() != null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        // 获取当前登录用户绑定了当前商铺的 所有子用户名
        List<String> shopUsers = userShopRepository.findByShopId(shopId).stream()
                .map(x -> x.getUsername())
                .filter(x -> !x.equals(username))
                .collect(Collectors.toList());
        return shopUsers;
    }

    @Override
    public List<String> getBatchShopUsers(String username, List<String> shopIds) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getParent() != null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        List<String> shopUsers = new ArrayList<>();
        if (shopIds != null && shopIds.size() > 0) {
            String maxShopId = null;
            int max = 0;
            for (String shopId : shopIds) {
                int userCount = userShopRepository.countByShopId(shopId);
                if (userCount > max) {
                    max = userCount;
                    maxShopId = shopId;
                }
            }
            if (maxShopId == null) {
                return shopUsers;
            }

            List<String> users = userShopRepository.findByShopId(maxShopId).stream().map(x -> x.getUsername()).filter(x -> !x.equals(username)).collect(Collectors.toList());
            for (String user : users) {
                int shopCount = userShopRepository.countByUsernameAndShopIdIn(user, shopIds);
                if (shopCount < shopIds.size()) {
                    continue;
                } else {
                    shopUsers.add(user);
                }

            }
        }
        return shopUsers;
    }

    @Override
    public Integer dealDirtyData() {
        List<Account> all = accountRepository.findAll();
        List<Account> parents = all.stream().filter(x -> StringUtils.isEmpty(x.getParent())).collect(Collectors.toList());
        // 移除 父账户，剩下的都是子账户了
        all.removeAll(parents);

        // 父账户， 子账户列表
        Map<Account, List<Account>> map = new HashMap<>();
        for (Account account : parents) {
            List<Account> list = new ArrayList<>();
            for (Account child : all) {
                if (child.getParent().equals(account.getName())) {
                    list.add(child);
                }
            }
            if (list.size() != 0) {
                map.put(account, list);
            }
        }

        Integer result = 0;
        Iterator<Map.Entry<Account, List<Account>>> entries = map.entrySet().iterator();
        while (entries.hasNext()) {
            Map.Entry<Account, List<Account>> entry = entries.next();
            List<Account> children = entry.getValue();
            // 查找子类所拥有的店铺
            List<Shop> childrenShop = shopRepository.findByOwnerIn(children.stream().map(Account::getName).collect(Collectors.toList()));
            if (childrenShop.size() == 0) {
                continue;
            }
            List<String> ids = childrenShop.stream().map(x -> x.getShopId()).collect(Collectors.toList());

            // 查出来 父账户已有的 usershop信息
            List<UserShop> parentUserShops = userShopRepository.findByUsernameAndShopIdIn(entry.getKey().getName(), ids);
            List<String> parentShopIds = parentUserShops.stream().map(x -> x.getShopId()).collect(Collectors.toList());

            // 剩下的就是 父账户的 usershop 缺失的信息
            ids.removeAll(parentShopIds);

            // 构建父账户缺失的 usershop 信息并保存
            List<UserShop> userShops = new ArrayList<>();
            for (String id : ids) {
                UserShop userShop = new UserShop();
                userShop.setShopId(id);
                userShop.setUsername(entry.getKey().getName());
                userShop.setGroupId("-1");

                userShops.add(userShop);
            }

            if (userShops.size() != 0) {
                // 将父账户缺失的 usershop 信息补充完整
                List<UserShop> userShops1 = userShopRepository.saveAll(userShops);
                result += userShops1.size();
            }

        }

        return result;
    }

    @Override
    public String queryShopCookieById(String id) {
        Shop shop = shopRepository.findById(id).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        return shop.getShopCookie();
    }

    @Override
    public Integer deleteShopCookieById(String id) {
        Shop shop = shopRepository.findById(id).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        shop.setShopCookie("");
        Shop save = shopRepository.save(shop);

        Integer i = 1;
        if (StringUtils.isEmpty(save.getShopCookie())) {
            i = 0;
        }
        return i;
    }

    @Override
    public List<FavoriteUrl> getFavoritesByShopId(String shopId) {
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        return shop.getFavoritesUrls();
    }

    @Override
    public boolean delShopUa(String uaflag) {
        List<ShopUA> all = shopUaRepository.findAll();
        for(ShopUA shopUa : all){
            List<String> uaList = shopUa.getUaList();
            List<String> newUaList = uaList.stream().filter((p) -> !p.contains(uaflag)).collect(Collectors.toList());
            shopUa.setUaList(newUaList);
            shopUaRepository.save(shopUa);
        }
        return true;
    }

    @Override
    public boolean saveFavoritesByShopId(String shopId, FavoriteUrl favoriteUrl) {
        return shopRepository.saveFavoritesUrls(shopId, favoriteUrl);
    }

    @Override
    public boolean deleteFavoritesByShopId(String shopId, FavoriteUrl favoriteUrl) {
        return shopRepository.deleteFavoritesByShopId(shopId, favoriteUrl);
    }

    private Page<Shop> getShopsByFilter(ShopFilterDto shopFilterDto, List<String> shopIds, Pageable pageable) {
        Page<Shop> shops = null;
        if (!StringUtils.isEmpty(shopFilterDto.getIpRegion())) {
            List<String> filter = ipResourceRepository.findShopIdInListAndRegionLike(shopIds, false, shopFilterDto.getIpRegion())
                    .stream().flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
            shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(filter, pageable);
        }
        if (!StringUtils.isEmpty(shopFilterDto.getShopAccount())) {
            shops = shopRepository.findByShopIdInAndShopAccountLikeOrderByCreateTimeDesc(shopIds, shopFilterDto.getShopAccount(), pageable);
        }
        if (!StringUtils.isEmpty(shopFilterDto.getShopName())) {
            shops = shopRepository.findByShopIdInAndShopNameLikeOrderByCreateTimeDesc(shopIds, shopFilterDto.getShopName(), pageable);
        }
        if (shopFilterDto.isEmpty()) {
            shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(shopIds, pageable);
        }
        return shops;
    }

    /*private ShopResultDto getShopResultDto(String username, Shop x, IpResource ipResource, SpecialLine specialLine) {
        if (ipResource == null) {
            ipResource = new IpResource();
        }
        String group1 = userShopRepository.findByUsernameAndShopId(username, x.getShopId()).getGroupId();
        ShopResultDto shopResultDto = null;
        if (ipResource.isSpecialLine()) {
            shopResultDto = ShopResultDto.of(x, group1, new IpResourceDto(ipResource, null, false, specialLine));
        } else {
            shopResultDto = ShopResultDto.of(x, group1, new IpResourceDto(ipResource, null, false));
        }
        return shopResultDto;
    }*/

    private ShopResultDto getShopResultDto(String groupId, Shop x, IpResource ipResource, SpecialLine specialLine) {
        if (ipResource == null) {
            ipResource = new IpResource();
        }

        ShopResultDto shopResultDto = null;
        if (ipResource.isSpecialLine()) {
            shopResultDto = ShopResultDto.of(x, groupId, new IpResourceDto(ipResource, null, false, specialLine));
        } else {
            shopResultDto = ShopResultDto.of(x, groupId, new IpResourceDto(ipResource, null, false));
        }
        return shopResultDto;
    }
}
