package com.edgec.browserbackend.account.repository;

import com.edgec.browserbackend.account.domain.BillStatus;
import com.edgec.browserbackend.account.domain.DeductionRecord;
import com.edgec.browserbackend.account.domain.Services;
import com.edgec.browserbackend.account.domain.UserPrePaidBilling;
import com.edgec.browserbackend.account.dto.BillQueryCriteriaDto;
import com.mongodb.client.result.UpdateResult;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.*;
import org.springframework.data.mongodb.core.query.BasicQuery;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.CriteriaDefinition;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.util.ObjectUtils;

import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.springframework.data.mongodb.core.query.Criteria.where;

public class UserPrePaidBillingRepositoryCustomImpl implements UserPrePaidBillingRepositoryCustom {

    @Autowired
    private MongoTemplate mongoTemplate;


    @Override
    public long countPrepaidOrderNum(int year, int month) {

        MatchOperation matchOperation = Aggregation.match(where("year").is(year).and("month").is(month).and("chargeType").in(Arrays.asList(1,2)));
        CountOperation countOperation = Aggregation.count().as("count");
        AggregationResults<HashMap> result = mongoTemplate.aggregate(Aggregation.newAggregation(matchOperation, countOperation),
                UserPrePaidBilling.class, HashMap.class);

        if (result.getMappedResults() == null || result.getMappedResults().size() == 0) {
            return 0;
        } else {
            HashMap map = result.getMappedResults().get(0);
            if (map == null) {
                return 0;
            } else {
                Integer num = (Integer) map.get("count");
                if (num == null) {
                    return 0;
                } else {
                    return num.longValue();
                }

            }
        }
    }

    @Override
    public float companyExpenseAmount(int year, int month) {
        Criteria criteria = new Criteria();
        criteria.where("year").is(year).and("month").is(month).orOperator(where("payMethod").is(3), where("chargeType").is(4));
        MatchOperation matchOperation = Aggregation.match(criteria);
        GroupOperation groupOperation = Aggregation.group("month").sum("total").as("prepaidAmount");

        AggregationResults<Map> totalSum = mongoTemplate.aggregate(Aggregation.newAggregation(matchOperation, groupOperation),
                UserPrePaidBilling.class, Map.class);

        if (totalSum.getMappedResults() == null || totalSum.getMappedResults().size() == 0) {
            return 0;
        } else {
            Map map = totalSum.getMappedResults().get(0);

            if (map == null) {
                return 0;
            } else {
                Double sum = (Double) map.get("prepaidAmount");
                if (sum == null) {
                    return 0;
                } else {
                    return sum.floatValue();
                }
            }
        }
    }

    @Override
    public float companyIncomeAmount(int year, int month) {

        MatchOperation matchOperation = Aggregation.match(where("year").is(year).and("month").is(month).and("payMethod").in(Arrays.asList(1,2)));
        GroupOperation groupOperation = Aggregation.group("month").sum("total").as("prepaidAmount");

        AggregationResults<Map> totalSum = mongoTemplate.aggregate(Aggregation.newAggregation(matchOperation, groupOperation),
                UserPrePaidBilling.class, Map.class);

        if (totalSum.getMappedResults() == null || totalSum.getMappedResults().size() == 0) {
            return 0;
        } else {
            Map map = totalSum.getMappedResults().get(0);

            if (map == null) {
                return 0;
            } else {
                Double sum = (Double) map.get("prepaidAmount");
                if (sum == null) {
                    return 0;
                } else {
                    return sum.floatValue();
                }
            }
        }
    }

    @Override
    public float companyWithdrawAmount(int year, int month) {
        MatchOperation matchOperation = Aggregation.match(where("year").is(year).and("month").is(month).and("chargeType").is(4));
        GroupOperation groupOperation = Aggregation.group("month").sum("total").as("prepaidAmount");

        AggregationResults<Map> totalSum = mongoTemplate.aggregate(Aggregation.newAggregation(matchOperation, groupOperation),
                UserPrePaidBilling.class, Map.class);

        if (totalSum.getMappedResults() == null || totalSum.getMappedResults().size() == 0) {
            return 0;
        } else {
            Map map = totalSum.getMappedResults().get(0);

            if (map == null) {
                return 0;
            } else {
                Double sum = (Double) map.get("prepaidAmount");
                if (sum == null) {
                    return 0;
                } else {
                    return sum.floatValue();
                }
            }
        }
    }

    @Override
    public float companyBankTransferAmount(int year, int month) {
        MatchOperation matchOperation = Aggregation.match(where("year").is(year).and("month").is(month).and("payMethod").is(3));
        GroupOperation groupOperation = Aggregation.group("month").sum("total").as("prepaidAmount");

        AggregationResults<Map> totalSum = mongoTemplate.aggregate(Aggregation.newAggregation(matchOperation, groupOperation),
                UserPrePaidBilling.class, Map.class);

        if (totalSum.getMappedResults() == null || totalSum.getMappedResults().size() == 0) {
            return 0;
        } else {
            Map map = totalSum.getMappedResults().get(0);

            if (map == null) {
                return 0;
            } else {
                Double sum = (Double) map.get("prepaidAmount");
                if (sum == null) {
                    return 0;
                } else {
                    return sum.floatValue();
                }
            }
        }
    }

    @Override
    public Page<UserPrePaidBilling> findByUsernameAndChargeTypes(Pageable pageable, String username, String[] chargeTypes, int year, int month, int day, String zoneId) {

        int pageSize = pageable.getPageSize();
        int pageNumber = pageable.getPageNumber();

        Integer[] chargeType = new Integer[chargeTypes.length];
        for (int i = 0; i < chargeTypes.length; i++) {
            chargeType[i] = Integer.parseInt(chargeTypes[i]);
        }
        zoneId = zoneId.replace(" ", "+");
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        CriteriaDefinition criteriaDefinition;
        MatchOperation match;
        if (day != 0) {
            //日开始时间
            ZonedDateTime zonedDateTimeDayBegin = ZonedDateTime.of(year, month, day, 0, 0, 0, 0, zoneOffset);
            long dayBeginTime = zonedDateTimeDayBegin.withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
            //日结束时间
            long dayEndTime = zonedDateTimeDayBegin.plusDays(1).withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;

            if (chargeTypes == null || chargeTypes.length == 0) {
                criteriaDefinition = where("username").is(username)
                        .and("timestamp").gte(dayBeginTime).lt(dayEndTime);
            } else {
                criteriaDefinition = where("chargeType").in(chargeType)
                        .and("username").is(username).and("timestamp")
                        .gte(dayBeginTime).lt(dayEndTime);
            }
            match = new MatchOperation(criteriaDefinition);
        } else {

            if (chargeTypes == null || chargeTypes.length == 0) {
                criteriaDefinition = where("username").is(username)
                        .and("year").is(year).and("month").is(month);
            } else {
                criteriaDefinition = where("chargeType").in(chargeType)
                        .and("username").is(username)
                        .and("year").is(year).and("month").is(month);
            }
            match = new MatchOperation(criteriaDefinition);
        }


        SortOperation sort = Aggregation.sort(Sort.Direction.DESC, "timestamp");
        SkipOperation skip = Aggregation.skip(pageNumber * pageSize);
        LimitOperation limit = Aggregation.limit(pageSize);
        CountOperation count = Aggregation.count().as("count");

        AggregationResults<Map> totalSize = mongoTemplate.aggregate(Aggregation.newAggregation(match, count), UserPrePaidBilling.class, Map.class);
        AggregationResults<UserPrePaidBilling> aggregate = mongoTemplate.aggregate(Aggregation.newAggregation(match, sort, skip, limit), UserPrePaidBilling.class, UserPrePaidBilling.class);

        if (totalSize.getMappedResults().size() == 0) {
            List<UserPrePaidBilling> userPrePaidBillings = Arrays.asList();
            Page<UserPrePaidBilling> userPrePaidBillingPage = new PageImpl<>(userPrePaidBillings, pageable, 0);
            return userPrePaidBillingPage;
        }
        Integer total = (Integer) (totalSize.getMappedResults().get(0).get("count"));

        Page<UserPrePaidBilling> userPrePaidBillingPage = new PageImpl<>(aggregate.getMappedResults(), pageable, total);

        return userPrePaidBillingPage;
    }

    @Override
    public Page<UserPrePaidBilling> findBillsByCondition(Pageable pageable, String username, BillQueryCriteriaDto billQueryCriteriaDto, long dayBeginTime, long dayEndTime) {
        int pageSize = pageable.getPageSize();
        int pageNumber = pageable.getPageNumber();
        Criteria criteria = where("username").is(username).and("timestamp").gte(dayBeginTime).lt(dayEndTime);
        if (billQueryCriteriaDto.getBillStatuses().size() != 0) {
            criteria.and("status").in(billQueryCriteriaDto.getBillStatuses());
        }
        if (billQueryCriteriaDto.getChargeTypes().size() != 0) {
            criteria.and("chargeType").in(billQueryCriteriaDto.getChargeTypes());
        }
        if (billQueryCriteriaDto.getIsPrepaid().size() != 0) {
            criteria.and("isPrepaid").in(billQueryCriteriaDto.getIsPrepaid());
        }
        if (billQueryCriteriaDto.getServices().size() != 0) {
            criteria.and("services").in(billQueryCriteriaDto.getServices());
        }
        MatchOperation match = Aggregation.match(criteria);
        SortOperation sort = Aggregation.sort(Sort.Direction.DESC, "timestamp");
        SkipOperation skip = Aggregation.skip(pageNumber * pageSize);
        LimitOperation limit = Aggregation.limit(pageSize);
        CountOperation count = Aggregation.count().as("count");
        AggregationResults<Map> totalSize = mongoTemplate.aggregate(Aggregation.newAggregation(match, count), UserPrePaidBilling.class, Map.class);
        if (totalSize.getMappedResults().size() == 0) {
            List<UserPrePaidBilling> userPrePaidBillings = Arrays.asList();
            return new PageImpl<>(userPrePaidBillings, pageable, 0);
        }
        Integer total = (Integer) (totalSize.getMappedResults().get(0).get("count"));
        AggregationResults<UserPrePaidBilling> aggregate = mongoTemplate.aggregate(Aggregation.newAggregation(match, sort, skip, limit), UserPrePaidBilling.class, UserPrePaidBilling.class);
        if (totalSize.getMappedResults().size() == 0) {
            List<UserPrePaidBilling> userPrePaidBillings = Arrays.asList();
            return new PageImpl<>(userPrePaidBillings, pageable, 0);
        }

        return new PageImpl<>(aggregate.getMappedResults(), pageable, total);
    }

    @Override
    public List<UserPrePaidBilling> findOverviewByYearAndMonthAndService(String username, int year, int month, Services service) {
        MatchOperation match;
        if (month == 0) {
            if (ObjectUtils.isEmpty(service)) {
                match = Aggregation.match(where("username").is(username).and("year").is(year));
            } else {
                match = Aggregation.match(where("username").is(username).and("year").is(year).and("services").is(service));
            }
        } else {
            if (ObjectUtils.isEmpty(service)) {
                match = Aggregation.match(where("username").is(username).and("year").is(year).and("month").is(month));
            } else {
                match = Aggregation.match(where("username").is(username).and("year").is(year).and("month").is(month).and("services").is(service));
            }
        }
        GroupOperation group = Aggregation.group("year", "month", "services", "isPrepaid", "chargeType")
                .sum("amount").as("amount")
                .sum("total").as("total");
        SortOperation sort = Aggregation.sort(Sort.Direction.DESC, "year").and(Sort.Direction.DESC, "month").and(Sort.Direction.ASC, "total");
        AggregationResults<UserPrePaidBilling> results = mongoTemplate.aggregate(Aggregation.newAggregation(match, group, sort), UserPrePaidBilling.class, UserPrePaidBilling.class);
        if (results.getMappedResults().size() != 0) {
            List<UserPrePaidBilling> userPrePaidBillings = results.getMappedResults();
            return userPrePaidBillings;
        }
        return Arrays.asList();
    }

    @Override
    public void upsertBillOwnCost(String username, String services, int year, int month, long timestamp, float potential, float actual, float ownCost, float total, BillStatus status, boolean isPrepaid, String user, int chargeType, String target) {

        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("username").is(username).and("services").is(services).and("year").is(year).and("month").is(month).and("user").is(user));

        Update update = new Update();
        update.set("timestamp", timestamp)
                .set("ownPotentialCost", potential)
                .set("ownActualCost", actual)
                .set("ownBillingCost", ownCost)
                .set("total", total)
                .set("status", status)
                .set("isPrepaid", isPrepaid)
                .set("chargeType", chargeType)
                .set("period", 1)
                .set("target", target);

        mongoTemplate.upsert(basicQuery, update, UserPrePaidBilling.class);
    }

    @Override
    public int updateBillStatus(String id, BillStatus fromStatus, BillStatus toStatus, List<DeductionRecord> deductionRecords) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("_id").is(id).and("status").is(fromStatus));

        Update update = new Update();
        update.set("status", toStatus).set("deductionRecords", deductionRecords);
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, UserPrePaidBilling.class);

        return new Long(result.getModifiedCount()).intValue();
    }

    @Override
    public List<UserPrePaidBilling> findBillStatisticsByUsernameAndService(String username, Services service) {
        MatchOperation match;
        if (ObjectUtils.isEmpty(service)) {
            match = Aggregation.match(where("username").is(username));
        } else {
            match = Aggregation.match(where("username").is(username).and("services").is(service));
        }
        GroupOperation group = Aggregation.group("year", "month", "services", "isPrepaid", "chargeType", "status")
                .sum("amount").as("amount")
                .sum("total").as("total");
        SortOperation sort = Aggregation.sort(Sort.Direction.DESC, "year").and(Sort.Direction.DESC, "month").and(Sort.Direction.ASC, "total");
        AggregationResults<UserPrePaidBilling> results = mongoTemplate.aggregate(Aggregation.newAggregation(match, group, sort), UserPrePaidBilling.class, UserPrePaidBilling.class);
        if (results.getMappedResults().size() != 0) {
            List<UserPrePaidBilling> userPrePaidBillings = results.getMappedResults();
            return userPrePaidBillings;
        }

        return Arrays.asList();
    }



    @Override
    public List<UserPrePaidBilling> findOverviewByYearAndMonth(String username, int year, int month, String zoneId, Services service) {

        UserPrePaidBilling StatisticsAmountAndTotal = new UserPrePaidBilling();
        StatisticsAmountAndTotal.setChargeType(-1);

        MatchOperation match = null;

        zoneId = zoneId.replace(" ", "+");
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);

        //月开始时间
        ZonedDateTime zonedDateTimeMonthBegin = ZonedDateTime
                .of(year, month, 1, 0, 0, 0, 0, zoneOffset);
        long mouthBeginTime = zonedDateTimeMonthBegin.withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
        //月结束时间
        long mouthEndTime = zonedDateTimeMonthBegin.plusMonths(1).withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
        Criteria criteria = where("username").is(username).and("timestamp").gte(mouthBeginTime).lt(mouthEndTime);
        if (service != null) {
            criteria.and("services").is(service.name());
        }
        match = Aggregation.match(criteria);
        GroupOperation group = Aggregation.group("services", "isPrepaid", "chargeType", "status")
                .sum("amount").as("amount").sum("total").as("total");
//        SortOperation sort = Aggregation.sort(Sort.Direction.ASC, "services").and(Sort.Direction.ASC, "status");
        AggregationResults<UserPrePaidBilling> results = mongoTemplate
                .aggregate(Aggregation.newAggregation(match, group), UserPrePaidBilling.class, UserPrePaidBilling.class);
        if (results.getMappedResults().size() != 0) {
            List<UserPrePaidBilling> userPrePaidBillings = results.getMappedResults();
            return userPrePaidBillings;
        }
        return Arrays.asList();
    }


}
