package com.edgec.browserbackend.auth.service.Impl;

import com.edgec.browserbackend.auth.domain.Roles;
import com.edgec.browserbackend.auth.domain.User;
import com.edgec.browserbackend.auth.domain.UserPasswordReset;
import com.edgec.browserbackend.auth.exception.AuthErrorCode;
import com.edgec.browserbackend.auth.repository.RolesRepository;
import com.edgec.browserbackend.auth.repository.UserRepository;
import com.edgec.browserbackend.auth.service.UserService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;

import java.util.Optional;

@Service
public class UserServiceImpl implements UserService {

    private static final BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
    private final Logger log = LoggerFactory.getLogger(getClass());
    @Autowired
    private UserRepository repository;

    @Autowired
    private RolesRepository rolesRepository;

    @Override
    public void verifyCode(String name, String code) {

        User existing = repository.findById(name).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + name));
        if (!StringUtils.isEmpty(code)) {
            if (existing.getVerificationCode().equals(code)) {
                existing.setEnabled(true);
                existing.setVerificationCode("");
                repository.save(existing);
                return;
            }
        }
        throw new ClientRequestException(AuthErrorCode.WRONGEMAILCODE, "Wrong email verification code: " + code);
    }

    @Override
    public void reset(User user) {
        User existing = repository.findById(user.getUsername()).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + user.getUsername()));
        existing.setVerificationCode(user.getVerificationCode());
        repository.save(existing);
    }

    @Override
    public void lock(User user, String lock) {
        User existing = repository.findById(user.getUsername()).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + user.getUsername()));
        existing.setEnabled(false);
        repository.save(existing);
    }

    @Override
    public void unlock(User user, String unlock) {
        User existing = repository.findById(user.getUsername()).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + user.getUsername()));
        existing.setEnabled(true);
        repository.save(existing);
    }

    public void deleteUser(String name) {
        User existing = repository.findById(name).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + name));
        repository.delete(existing);
    }

    @Override
    public boolean lockState(String name) {
        User existing = repository.findById(name).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + name));
        return existing.isEnabled();
    }

    @Override
    public void updateRoles(String name, String roles) {
        Roles role = new Roles();
        role.setRoles(roles);
        role.setUsername(name);
        rolesRepository.save(role);
    }

    @Override
    public void addRoles(String name, String roles) {
        Roles role = rolesRepository.findById(name).orElse(null);
        if (role != null) {
            if (!role.getRoles().contains("AUTHUSER")) {
                role.setRoles(role.getRoles() + "," + roles);
                rolesRepository.save(role);
            }
        }
    }

    @Override
    public void create(User user) {

        Optional<User> existing = repository.findById(user.getUsername());
        existing.ifPresent(it -> {
            throw new ClientRequestException(AuthErrorCode.NAMEEXIST, "user already exists: " + it.getUsername());
        });

        String hash = encoder.encode(user.getPassword());
        user.setPassword(hash);

        repository.save(user);

        log.info("new user has been created: {}", user.getUsername());
    }

    @Override
    public void resetUserPassword(User user) {
        User existing = repository.findById(user.getUsername()).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + user.getUsername()));
        if (!StringUtils.isEmpty(user.getPassword())) {
            String newhash = encoder.encode(user.getPassword());
            existing.setPassword(newhash);
            repository.save(existing);
        }
    }

    @Override
    public boolean changePassword(UserPasswordReset userPasswordReset) {
        User existing = repository.findById(userPasswordReset.getUsername()).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + userPasswordReset.getUsername()));
        if (!StringUtils.isEmpty(userPasswordReset.getPassword())) {
            //change password with old password
            String hash = encoder.encode(userPasswordReset.getPassword());

            if (encoder.matches(userPasswordReset.getPassword(), existing.getPassword())) {
                String newhash = encoder.encode(userPasswordReset.getNewPassword());
                existing.setPassword(newhash);
                repository.save(existing);
                return true;
            } else {
                throw new ClientRequestException(AuthErrorCode.AUTHENTICATION_ERROR, "Wrong password used.");
            }
        }

        if (!StringUtils.isEmpty(userPasswordReset.getVerificationCode())) {
            if (existing.getVerificationCode().equals(userPasswordReset.getVerificationCode())) {
                String newhash = encoder.encode(userPasswordReset.getNewPassword());
                existing.setPassword(newhash);
                existing.setVerificationCode("");
                repository.save(existing);
                return true;
            } else {
                throw new ClientRequestException(AuthErrorCode.AUTHENTICATION_ERROR, "Wrong verification code.");
            }
        }
        return false;
    }


    public void updateUser(String username, User user) {
        User existing = repository.findById(username).orElseThrow(() -> new ClientRequestException(AuthErrorCode.NAMENOTEXIST, "user does not exist: " + username));
        existing.setEmail(user.getEmail());
        repository.save(existing);
    }

}
