package com.edgec.browserbackend.browser.service.Impl;

import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.dto.*;
import com.edgec.browserbackend.browser.repository.*;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.utils.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

@Service
@Transactional
public class ShopServiceImpl implements ShopService {

    private final Logger logger = LoggerFactory.getLogger(ShopServiceImpl.class);

    @Autowired
    ShopRepository shopRepository;

    @Autowired
    AccountRepository accountRepository;

    @Autowired
    UserShopRepository userShopRepository;

    @Autowired
    GroupRepository groupRepository;

    @Autowired
    IpResourceRepository ipResourceRepository;

    @Autowired
    IpAndShopService ipAndShopService;

    @Autowired
    IpResourceService ipResourceService;

    @Autowired
    SpecialLineRepository specialLineRepository;

    @Override
    public String addShop(String username, ShopResultDto shopResultDto) {
        // 1. 对商铺的分组信息校验
        if (shopResultDto.getGroup() == null) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        if (shopResultDto.getGroup() != null) {
            Group group = groupRepository.findById(shopResultDto.getGroup())
                                         .orElseThrow(() -> new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST));
        }

        // 2. 对当前用户的 account 信息进行校验
        Account account = accountRepository.findByName(username)
                                           .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        if (account.getShopCount() >= 10000) {
            throw new ClientRequestException(AccountErrorCode.SHOPMAX);
        }

        try {
            // 3. 将新增的 shop 信息保存并返回 id
            String id = getShopId(username, shopResultDto);

            // 4. 封装 usershop 并保存到 usershop 表
            UserShop userShop = new UserShop();
            userShop.setShopId(id);
            userShop.setUsername(username);
            userShop.setGroupId(shopResultDto.getGroup());
            userShopRepository.save(userShop);

            // 5. 如果当前用户有父账户，则也封装usershop并保存到 usershop 表，区别是 username 与 groupId，同时更新 父账户的商铺数
            if (!StringUtils.isEmpty(account.getParent())) {
                UserShop userShop1 = new UserShop();
                userShop1.setShopId(id);
                userShop1.setUsername(account.getParent());
                userShop1.setGroupId("-1");
                userShopRepository.save(userShop1);

                Account parentAccount = accountRepository.findByName(account.getParent())
                                                         .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
                account.setShopCount(parentAccount.getShopCount() + 1);
                accountRepository.save(parentAccount);
            }

            // 6. 更新当前 account 的商铺数
            account.setShopCount(account.getShopCount() + 1);
            accountRepository.save(account);

            // 7. 返回新增商铺的 id
            return id;
        } catch (Exception e) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
    }

    @Override
    public List<String> addShops(String username, MultipartFile file) throws IOException {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        if (account.getShopCount() >= 10000) {
            throw new ClientRequestException(AccountErrorCode.SHOPMAX);
        }

        List<String> ids = new ArrayList<>();
        List<List<Object>> list = FileUtil.readExcel(file.getInputStream());
        for (List<Object> l : list) {
            ShopResultDto shopResultDto = new ShopResultDto();
            shopResultDto.setOwner(username);
            shopResultDto.setShopName(l.get(0).toString());
            shopResultDto.setShopPlatform(l.get(1).toString());
            if (!StringUtils.isEmpty(l.get(2).toString())) {
                shopResultDto.setShopAccount(l.get(2).toString());
            }
            if (!StringUtils.isEmpty(l.get(3).toString())) {
                shopResultDto.setShopPassword(l.get(3).toString());
            }
            // 批量导入店铺时，默认不分组
            shopResultDto.setGroup("-1");

            try {
                String id = getShopId(username, shopResultDto);

                UserShop userShop = new UserShop();
                userShop.setShopId(id);
                userShop.setGroupId(shopResultDto.getGroup());
                userShop.setUsername(username);
                userShopRepository.save(userShop);

                if (!StringUtils.isEmpty(account.getParent())) {
                    UserShop userShop1 = new UserShop();
                    userShop1.setShopId(id);
                    userShop1.setGroupId("-1");
                    userShop1.setUsername(account.getParent());
                    userShopRepository.save(userShop1);

                    Account parentAccount = accountRepository.findByName(account.getParent())
                                                             .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
                    account.setShopCount(parentAccount.getShopCount() + 1);
                    accountRepository.save(parentAccount);
                }

                account.setShopCount(account.getShopCount() + 1);
                accountRepository.save(account);

                ids.add(id);
            } catch (Exception e) {
                logger.error("fail to add shops", e.getMessage());
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            }
        }
        return ids;
    }

    @Override
    public String updateShop(String username, ShopResultDto shopResultDto) {
        // 1.校验参数、account 、权限
        if (StringUtils.isEmpty(shopResultDto.getShopId())) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        // 2.查询当前店铺是否已经绑定当前用户
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
        if (userShop == null) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }

        // 3. 获取旧的店铺信息，并封装新的数据
        Shop shop_old = shopRepository.findById(shopResultDto.getShopId()).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        shop_old = shop_old.of(shopResultDto);

        try {
            // 4. 更新店铺信息
            shopRepository.save(shop_old);
        } catch (Exception e) {
            logger.error("fail to update", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        return shop_old.getShopId();
    }

    @Override
    public void deleteShop(String username, String shopId) {
        // 1. 校验参数、account 、权限、店铺是否已经和用户绑定
        if (shopId == null) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }

        // 2. 获取店铺信息
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));

        // 3. 如果当前店铺已经绑定 未被删除的 ip 资源，则让IP资源先解绑店铺
        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shop.getShopId(), false);
        if (ipResource != null) {
            ShopRequestDto shopRequestDto = new ShopRequestDto();
            shopRequestDto.setShopId(shopId);
            shopRequestDto.setIpId(ipResource.getId());
            ipAndShopService.unBindShop(username, shopRequestDto);
        }

        // 4. 删除当前店铺关联的所有的 usershop信息
        boolean result = userShopRepository.deleteByShopId(shopId);
        if (result) {
            // 5. 删除当前店铺，并更新 account信息 todo
            shopRepository.deleteById(shopId);

            // 6. 更新 和店铺关联的所有的 account 的信息
            List<UserShop> userShops = userShopRepository.findByShopId(shopId);
            List<Account> accountList = accountRepository.findByNameIn(userShops.stream().map(UserShop::getUsername).collect(Collectors.toList()));
            for (Account a : accountList) {
                a.setShopCount(account.getShopCount() - 1);
                accountRepository.save(a);
            }

        } else {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
    }

    @Override
    public void transferShop(String username, String shopId, String groupId) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        Group group = groupRepository.findById(groupId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST));
        if (userShop == null || (!group.getId().equals("-1") && group.getOwner() != null && !group.getOwner().equals(username))) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
        try {
            userShop.setGroupId(groupId);
            userShopRepository.save(userShop);
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }

    }

    @Override
    public void assignShops(String username, List<String> shopIds, List<String> users) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        List<UserShop> userShops = userShopRepository.findByUsernameAndShopIdIn(username, shopIds);
        if (account.getPermission() < 8 || userShops == null || userShops.size() < 1) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Pageable pageable = PageRequest.of(0, 100);
        List<Shop> shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(shopIds, pageable).getContent();
        if (shops == null || shops.size() < 1)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        List<Account> accounts = accountRepository.findByNameIn(users);
        if (accounts == null || accounts.size() != users.size())
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        for (Account ac : accounts) {
            if (ac.getParent() == null || !ac.getParent().equals(username)) {
                throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
            }
        }
        if (shops.size() == 1) {
            try {
                userShopRepository.deleteByShopIdExceptOwner(shops.get(0).getShopId(), shops.get(0).getOwner());
                for (Account account1 : accounts) {
                    UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shops.get(0).getShopId());
                    if (userShop1 != null)
                        continue;
                    userShop1 = new UserShop(account1.getName(), shops.get(0).getShopId(), "-1");
                    userShopRepository.save(userShop1);
                }
            } catch (Exception e) {
                logger.error("fail to assign", e.getMessage());
                throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
            }
        } else {
            shops.forEach(shop -> {
                try {
                    for (Account account1 : accounts) {
                        UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shop.getShopId());
                        if (userShop1 != null)
                            continue;
                        userShop1 = new UserShop(account1.getName(), shop.getShopId(), "-1");
                        userShopRepository.save(userShop1);
                    }
                } catch (Exception e) {
                    logger.error("fail to assign", e.getMessage());
                    throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
                }
            });
        }
    }

    @Override
    public ShopResultDto queryShop(String username, String shopId) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }
        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shopId, false);
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        String group = userShopRepository.findByUsernameAndShopId(username, shop.getShopId()).getGroupId();
        ShopResultDto shopResultDto = null;
        if (ipResource.isSpecialLine()) {
            SpecialLine specialLine = specialLineRepository.findAll().get(0);
            shopResultDto = ShopResultDto.of(shop, group, new IpResourceDto(ipResource, null, false, specialLine));
        } else
            shopResultDto = ShopResultDto.of(shop, group, new IpResourceDto(ipResource, null, false));
        return shopResultDto;
    }

    @Override
    public ShopPageResultDto getShopList(String username, String groupId, int pageNum, int amount, ShopFilterDto shopFilterDto) {
        // 当前登录用户的账户是否存在
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));


        // 当前查询的分组信息是否正确
        Group group = null;
        if (groupId != null) {
            group = groupRepository.findById(groupId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST));

            if (group.getOwner() != null && !group.getOwner().equals(username)) {
                throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
            }
        }

        // 根据 groupId 与 username 来查询 shopIds (如果当前用户是父账户，则查询结果包含子账户的shopId)
        List<String> allIds = null;
        if ("-1".equals(groupId)) {
            allIds = userShopRepository.findByUsername(username).stream()
                                       .map(UserShop::getShopId).collect(Collectors.toList());
            // 如果分组
        } else {
            allIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream()
                                       .map(UserShop::getShopId).collect(Collectors.toList());
        }

        List<String> shopIds = null;
        if (shopFilterDto.getBindIp() == 0) {
            shopIds = allIds;

            // 这个地方有点奇怪，为啥不去 userShop 中找，而是在 ipresource 中找？ 可以查看下删除绑定状态的逻辑
        } else if (shopFilterDto.getBindIp() == 1) {
            // ip资源已经分配了 的店铺
            shopIds = ipResourceRepository.findShopIdInList(allIds, false).stream()
                                          .flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());

            // 这个地方的其他情况是指什么情况
        } else {
            // 店铺绑定的 ip 资源被删除的 店铺
            for (String id : allIds) {
                IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, false);
                if (ipResource == null) {
                    shopIds.add(id);
                }
            }
        }

        amount = amount > 100 ? 100 : amount;
        Pageable pageable = PageRequest.of(pageNum, amount);
        // 根据商铺ids 与 过滤条件 得到商铺
        Page<Shop> shops = getShopsByFilter(shopFilterDto, shopIds, pageable);

        ShopPageResultDto<ShopResultDto> shopPageResultDto = new ShopPageResultDto<>();
        if (shops != null && shops.getNumberOfElements() >= 1) {
            List<ShopResultDto> shopResultDtos = new ArrayList<>();
            shops.getContent().stream().forEach(
                    x -> {
                        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(x.getShopId(), false);
                        // 如果 ip资源非空 且 addr 也非空
                        if (ipResource != null && !StringUtils.isEmpty(ipResource.getAddr())) {
                            // 1. ip资源在未来七天内到期 且 ip 资源的状态不是 3（正在分配）、5（已失效）、6（未分配），则将 ip 资源设置为 2（即将过期）
                            if (ipResource.getValidTime() <= Instant.now().plusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getValidTime() > Instant.now().toEpochMilli()) {
                                if (ipResource.getStatus() != 5 && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                    ipResource.setStatus(2);
                                    ipResourceRepository.save(ipResource);
                                }

                                // 2. ip资源在七天前到期，且 ip 资源的状态不是 3（正在分配）、6（未分配），则删除 ip 资源
                            } else if (ipResource.getValidTime() <= Instant.now().minusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                if (ipResource.getIpType() == IpType.VENDOR) {
                                    IpResourceRequestDto ipResourceRequestDto1 = new IpResourceRequestDto();
                                    ipResourceRequestDto1.setAddr(Arrays.asList(ipResource.getAddr()));
                                    try {
                                        ipResourceService.deleteIp(username, ipResourceRequestDto1);
                                    } catch (Exception e) {
                                        logger.error(e.getMessage(), e);
                                    }
                                } else {
                                    IpResourceRequestDto ipResourceRequestDto1 = new IpResourceRequestDto();
                                    ipResourceRequestDto1.setIpId(Arrays.asList(ipResource.getId()));
                                    try {
                                        ipResourceService.deleteIp(username, ipResourceRequestDto1);
                                    } catch (Exception e) {
                                        logger.error(e.getMessage(), e);
                                    }
                                }
                                return;

                                // 3. ip资源到期，且 ip 资源的状态不是 3（正在分配）、6（未分配），则设置 ip 资源的状态为 1（已过期）
                            } else if (ipResource.getValidTime() <= Instant.now().toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                ipResource.setStatus(1);
                                ipResourceRepository.save(ipResource);

                                // 4. 其他
                            } else {
                                if ((ipResource.getStatus() == 0 || ipResource.getStatus() == 1 || ipResource.getStatus() == 2) && ipResource.getVendor() != Vendor.local) {
                                    ipResource.setStatus(0);
                                } else if (ipResource.getIpType().equals(IpType.LOCAL) && (ipResource.getStatus() == 1 || ipResource.getStatus() == 2) && ipResource.getAddr().equals("本地Ip未使用")) {
                                    ipResource.setStatus(4);
                                }
                                ipResourceRepository.save(ipResource);
                            }
                        }

                        ShopResultDto shopResultDto = getShopResultDto(username, x, ipResource);

                        shopResultDtos.add(shopResultDto);
                    }
            );

            Page<ShopResultDto> shopDtoPage = new PageImpl<>(shopResultDtos, pageable, shopIds.size());
            shopPageResultDto.setShopList(shopDtoPage.getContent());
            PageInfo pageInfo = new PageInfo(shopDtoPage.getPageable().getPageNumber(), shopDtoPage.getTotalPages(), shopIds.size());
            shopPageResultDto.setShopPage(pageInfo);
        }

        return shopPageResultDto;
    }

    @Override
    public ShopSummary getShopSummary(String username) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));


        ShopSummary shopSummary = new ShopSummary();
        List<String> allShopIds = userShopRepository.findByUsername(username).stream()
                                                    .map(UserShop::getShopId).collect(Collectors.toList());
        List<String> unbind = new ArrayList<>();
        if (!allShopIds.isEmpty()) {
            for (String id : allShopIds) {
                IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, false);
                if (ipResource == null) {
                    unbind.add(id);
                }
            }
        }
        shopSummary.setUnbind(unbind.size());

        List<String> shopIds = userShopRepository.findByUsername(username).stream()
                                                 .map(UserShop::getShopId).collect(Collectors.toList());
        List<String> bind = ipResourceRepository.findShopIdInList(shopIds, false).stream()
                                                .map(IpResource::getId).collect(Collectors.toList());
        int expired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(1, bind, false);
        int willexpired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(2, bind, false);
        shopSummary.setExpired(expired);
        shopSummary.setWillExpire(willexpired);
        shopSummary.setTotal(shopIds.size());
        return shopSummary;
    }

    @Override
    public List<String> getShopUsers(String username, String shopId) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getParent() != null)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        List<String> shopUsers = userShopRepository.findByShopId(shopId).stream().map(x -> x.getUsername()).filter(x -> !x.equals(username)).collect(Collectors.toList());
        return shopUsers;
    }

    @Override
    public List<String> getBatchShopUsers(String username, List<String> shopIds) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getParent() != null)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        List<String> shopUsers = new ArrayList<>();
        if (shopIds != null && shopIds.size() > 0) {
            String maxShopId = null;
            int max = 0;
            for (String shopId : shopIds) {
                int userCount = userShopRepository.countByShopId(shopId);
                if (userCount > max) {
                    max = userCount;
                    maxShopId = shopId;
                }
            }
            if (maxShopId == null)
                return new ArrayList<>();
            List<String> users = userShopRepository.findByShopId(maxShopId).stream().map(x -> x.getUsername()).filter(x -> !x.equals(username)).collect(Collectors.toList());
            for (String user : users) {
                int shopCount = userShopRepository.countByUsernameAndShopIdIn(user, shopIds);
                if (shopCount < shopIds.size())
                    continue;
                else
                    shopUsers.add(user);
            }
        }
        return shopUsers;
    }


    private String getShopId(String username, ShopResultDto shopResultDto) {
        Shop shop = new Shop();
        shopResultDto.setOwner(username);
        shop.of(shopResultDto);
        shop.setCreateTime(Instant.now().toEpochMilli());
        String id = shopRepository.save(shop).getShopId();
        return id;
    }

    private Page<Shop> getShopsByFilter(ShopFilterDto shopFilterDto, List<String> shopIds, Pageable pageable) {
        Page<Shop> shops = null;
        if (!StringUtils.isEmpty(shopFilterDto.getIpRegion())) {
            List<String> filter = ipResourceRepository.findShopIdInListAndRegionLike(shopIds, false, shopFilterDto.getIpRegion())
                                                      .stream().flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
            shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(filter, pageable);
        }
        if (!StringUtils.isEmpty(shopFilterDto.getShopAccount())) {
            shops = shopRepository.findByShopIdInAndShopAccountLikeOrderByCreateTimeDesc(shopIds, shopFilterDto.getShopAccount(), pageable);
        }
        if (!StringUtils.isEmpty(shopFilterDto.getShopName())) {
            shops = shopRepository.findByShopIdInAndShopNameLikeOrderByCreateTimeDesc(shopIds, shopFilterDto.getShopName(), pageable);
        }
        if (shopFilterDto.isEmpty()) {
            shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(shopIds, pageable);
        }
        return shops;
    }

    private ShopResultDto getShopResultDto(String username, Shop x, IpResource ipResource) {
        if (ipResource == null) {
            ipResource = new IpResource();
        }
        String group1 = userShopRepository.findByUsernameAndShopId(username, x.getShopId()).getGroupId();
        ShopResultDto shopResultDto = null;
        if (ipResource.isSpecialLine()) {
            SpecialLine specialLine = specialLineRepository.findAll().get(0);
            shopResultDto = ShopResultDto.of(x, group1, new IpResourceDto(ipResource, null, false, specialLine));
        } else {
            shopResultDto = ShopResultDto.of(x, group1, new IpResourceDto(ipResource, null, false));
        }
        return shopResultDto;
    }
}
