package com.edgec.browserbackend.account.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.edgec.browserbackend.account.dto.*;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.account.service.PaymentService;
import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.*;
import com.edgec.browserbackend.account.service.EmailService;
import com.edgec.browserbackend.account.service.SmsUtils;
import com.edgec.browserbackend.account.utils.AccountServicePool;
import com.edgec.browserbackend.auth.exception.AuthErrorCode;
import com.edgec.browserbackend.auth.repository.UserRepository;
import com.edgec.browserbackend.auth.service.UserAuthService;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.IpSummary;
import com.edgec.browserbackend.browser.domain.ShopSummary;
import com.edgec.browserbackend.browser.dto.PageInfo;
import com.edgec.browserbackend.browser.dto.ShopPageResultDto;
import com.edgec.browserbackend.browser.dto.ShopResultDto;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.utils.CommonStringUtils;
import com.edgec.browserbackend.common.utils.Aes;
import com.edgec.browserbackend.common.utils.FileUtil;
import com.mongodb.DB;
import com.mongodb.client.gridfs.GridFSBucket;
import com.mongodb.client.gridfs.GridFSDownloadStream;
import com.mongodb.client.gridfs.model.GridFSFile;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSInputFile;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.tomcat.util.http.fileupload.FileItem;
import org.apache.tomcat.util.http.fileupload.disk.DiskFileItem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.mongodb.gridfs.GridFsOperations;
import org.springframework.data.mongodb.gridfs.GridFsResource;
import org.springframework.data.mongodb.gridfs.GridFsTemplate;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;
import org.thymeleaf.util.StringUtils;

import java.io.*;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.YearMonth;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static org.springframework.data.mongodb.core.query.Query.query;
import static org.springframework.data.mongodb.gridfs.GridFsCriteria.whereFilename;

@Service
@Transactional
@ComponentScan("com.edgec.browserbackend.account.repository")
public class AccountServiceImpl implements AccountService {

    private static final int TRIVAL_MONTHS = 1;

    private final Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    private OtpRepository otpRepository;

    @Autowired
    private UserBalanceRepository userBalanceRepository;

    @Autowired
    private PreOrderRepository preOrderRepository;

    @Autowired
    private AccountRepository repository;

    @Autowired
    private EmailService emailService;

    @Autowired
    private UserBillingRepository billingRepository;

    @Autowired
    private UserPrePaidBillingRepository prePaidBillingRepository;

    @Autowired
    private UserPaymentRepository userPaymentRepository;

    @Autowired
    private InvoiceRepository invoiceRepository;

    @Autowired
    private UserAuthService userAuthService;

    @Autowired
    private UserRepository userRepository;

    @Autowired
    private ShopService shopService;

    @Autowired
    private IpResourceService ipResourceService;

    @Autowired
    private WhiteSiteRepository whiteSiteRepository;

    @Autowired
    private CompanyAuthorizeRepository companyAuthorizeRepository;

    @Autowired
    private QueryIpUrlListRepository queryIpUrlListRepository;

    @Autowired
    private PaymentService paymentService;

    @Override
    public List<UserBillList> getUserBills0(String name) {

        List<UserBillList> result = new ArrayList<>();

        List<UserBilling> byUsernameOrderByYearDescMonthDesc = billingRepository.findByUsernameOrderByYearDescMonthDesc(name);

        if (byUsernameOrderByYearDescMonthDesc != null && byUsernameOrderByYearDescMonthDesc.size() > 0) {
            for (UserBilling userBilling : byUsernameOrderByYearDescMonthDesc) {

                UserBillList list = new UserBillList();
                list.setStatus(userBilling.getStatus());
                list.setPeriod(userBilling.getYear() + "-" + (userBilling.getMonth() < 10 ? "0" + userBilling.getMonth() : userBilling.getMonth()));
                list.addUserBill(userBilling);
                result.add(list);

                List<Account> children = repository.findByParent(name);
                if (children != null && children.size() > 0) {
                    List<String> collect = children.stream().map(x -> x.getName()).collect(Collectors.toList());
                    List<UserBilling> childrenBills = billingRepository.findByUsernameInAndYearAndMonthOrderByYearDescMonthDesc(collect, userBilling.getYear(), userBilling.getMonth());

                    if (childrenBills != null && childrenBills.size() > 0)
                        childrenBills.forEach(x -> list.addUserBill(x));
                }
            }
        }
        return result;
    }

    @Override
    public List<UserBillList> getUserBills(String name, Services service) {
        List<UserBillList> result = new ArrayList<>();
        List<UserPrePaidBilling> userPrePaidBillings = prePaidBillingRepository.findBillStatisticsByUsernameAndService(name, service);
        if (userPrePaidBillings != null && userPrePaidBillings.size() > 0) {
            for (UserPrePaidBilling userBilling : userPrePaidBillings) {
                UserBillList list = new UserBillList();
                list.setStatus(userBilling.getStatus());
                list.setPeriod(userBilling.getYear() + "-" + (userBilling.getMonth() < 10 ? "0" + userBilling.getMonth() : userBilling.getMonth()));
                list.setBill(userBilling);
                result.add(list);
            }
        }
        return result;
    }

    @Override
    public PrePaidBillingPageResultDto getUserPrePaidBills(String name, UserPrePaidBillingRequestDto userPrePaidBillingRequestDto) {
        Account account = repository.findByName(name);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);

        if (userPrePaidBillingRequestDto.getAmount() > 100)
            userPrePaidBillingRequestDto.setAmount(100);
        Pageable pageable = PageRequest.of(userPrePaidBillingRequestDto.getPage(), userPrePaidBillingRequestDto.getAmount());
        long time = Instant.now().atZone(ZoneOffset.UTC).minusDays(userPrePaidBillingRequestDto.getDay()).toInstant().toEpochMilli();

        Page<UserPrePaidBilling> userPrePaidBillings;
        if (userPrePaidBillingRequestDto.getTradeType() == 1) {
            if (account.getParent() != null)
                userPrePaidBillings = prePaidBillingRepository.findByUsernameAndTimestampGreaterThanAndChargeTypeOrderByTimestampDesc(name, time, 0, pageable);
            else
                userPrePaidBillings = prePaidBillingRepository.findByAdministratorAndTimestampGreaterThanAndChargeTypeOrderByTimestampDesc(name, time, 0, pageable);
        } else if (userPrePaidBillingRequestDto.getTradeType() == 2 || userPrePaidBillingRequestDto.getTradeType() == 3 || userPrePaidBillingRequestDto.getTradeType() == 4) {
            if (account.getParent() != null)
                userPrePaidBillings = prePaidBillingRepository.findByUsernameAndTimestampGreaterThanAndChargeTypeIsNotAndPayMethodOrderByTimestampDesc(name, time, 0, userPrePaidBillingRequestDto.getTradeType() - 2, pageable);
            else
                userPrePaidBillings = prePaidBillingRepository.findByAdministratorAndTimestampGreaterThanAndChargeTypeIsNotAndPayMethodOrderByTimestampDesc(name, time, 0, userPrePaidBillingRequestDto.getTradeType() - 2, pageable);
        } else {
            if (account.getParent() != null)
                userPrePaidBillings = prePaidBillingRepository.findByUsernameAndTimestampGreaterThanOrderByTimestampDesc(name, time, pageable);
            else
                userPrePaidBillings = prePaidBillingRepository.findByAdministratorAndTimestampGreaterThanOrderByTimestampDesc(name, time, pageable);
        }

        PrePaidBillingPageResultDto prePaidBillingPageResultDto = new PrePaidBillingPageResultDto();
        prePaidBillingPageResultDto.setBillingList(userPrePaidBillings.getContent());
        PageInfo pageInfo = new PageInfo();
        pageInfo.setTotalItems((int) userPrePaidBillings.getTotalElements());
        pageInfo.setTotalPages(userPrePaidBillings.getTotalPages());
        pageInfo.setCurrentPage(userPrePaidBillings.getNumber());
        prePaidBillingPageResultDto.setBillingPage(pageInfo);
        return prePaidBillingPageResultDto;
    }

    @Override
    public IpChargeResultDto chargeByMoney(String name, double money, IpChargeRequestDto requestDto) {
        IpChargeResultDto charge = new IpChargeResultDto();
        CompletableFuture.runAsync(() -> {
            charge.setApprovedAmount(0);
            charge.setSuccess(true);

            Account account = repository.findByName(name);
            if (account != null) {
                UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
                if (userBalance == null) {
                    userBalance = new UserBalance();
                    userBalance.setBalanced(0);
                    userBalance.setUsed(0);
                    userBalance.setUsername(name);
                    userBalanceRepository.save(userBalance);
                }

                userBalanceRepository.incrementBalance(userBalance, -(float) money, (float) money);
                userBalance = userBalanceRepository.findById(name).orElse(null);

                charge.setBalance(Math.round(userBalance.getBalanced()));
                charge.setSuccess(true);

                UserPrePaidBilling bill = new UserPrePaidBilling();
                if (account.getParent() != null)
                    bill.setAdministrator(account.getParent());
                else
                    bill.setAdministrator(account.getName());
                bill.setTradeNo(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + SmsUtils.createRandom(true, 4));
                bill.setChargeType(requestDto.getChargeType());
                bill.setAmount(requestDto.getAmount());
                bill.setUnit(requestDto.getUnit());
                bill.setPeriod(requestDto.getPeriod());
                bill.setPayMethod(requestDto.getPayMethod());
                bill.setUsername(name);
                bill.setTotal((float) money);
                bill.setStatus(BillStatus.PAID);
                bill.setPrepaid(true);
                bill.setTimestamp(Instant.now().toEpochMilli());

                final YearMonth lastmonth = YearMonth.now();

                int monthValue = lastmonth.getMonthValue();
                int year = lastmonth.getYear();
                bill.setYear(year);
                bill.setMonth(monthValue);
                bill.setBalanced(userBalance.getBalanced());

                prePaidBillingRepository.save(bill);
            }

        }, AccountServicePool.taskPool).join();

        return charge;
    }

    private static final String DEFAULT_REGION_KEY = "--default";

    @Override
    public String deletePreOrder(String username) {
        List<PreOrder> preOrders = preOrderRepository.findPreOrderByUserName(username);
        log.error("deletePreOrder size : " + preOrders.size());
        preOrders.forEach(x -> {
            log.error(x.toString());
            preOrderRepository.deleteById(x.getTradeNo());
        });
        return "success";
    }

    @Override
    public IpChargeResultDto preChargeByMoney(String name, double money) {
        IpChargeResultDto precharge = new IpChargeResultDto();
        CompletableFuture.runAsync(() -> {
            precharge.setApprovedAmount(0);
            precharge.setSuccess(false);
            Account account = repository.findByName(name);
            if (account != null) {
                UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
                float balance = 0;
                if (userBalance != null)
                    balance = userBalance.getBalanced();

                if (balance >= money) {
                    precharge.setBalance((int) Math.round(balance - money));
                    precharge.setSuccess(true);
                } else {
                    precharge.setBalance(Math.round(balance));
                }
            }
        }, AccountServicePool.taskPool).join();

        return precharge;
    }

    @Override
    public IpChargeResultDto preChargeIp(String name, int amount, double money) {
        IpChargeResultDto precharge = new IpChargeResultDto();
        CompletableFuture.runAsync(() -> {
            precharge.setApprovedAmount(0);
            precharge.setSuccess(false);
            Account account = repository.findByName(name);
            if (account != null) {
                UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
                float balance = 0;
                if (userBalance != null)
                    balance = userBalance.getBalanced();
                if (balance >= money) {
                    precharge.setApprovedAmount(amount);
                    precharge.setBalance(Math.round(balance - (float) money));
                    precharge.setSuccess(true);
                } else {
                    precharge.setBalance(Math.round(balance));
                }
            }
        }, AccountServicePool.taskPool).join();
        return precharge;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Account findByName(String accountName) {
        Assert.hasLength(accountName);
        Account account = repository.findByName(accountName);
        if (account == null) {
            return null;
        }
        if (StringUtils.isEmpty(account.getParent())) {
            account.setPermission(15);
            account.setAllowedToCreateSubUser(true);
        }
        return account;
    }

    @Override
    public ResultDto getAccountByName(String name) {
        Assert.hasLength(name);
        ResultDto resultDto = new ResultDto();
        try {
            Account account = this.findByName(name);
            AccountDto current = new AccountDto(account);

            UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);

            if (userBalance != null)
                current.setBalance(Math.round(userBalance.getBalanced()));
            else
                current.setBalance(0);

            if (StringUtils.isEmpty(account.getParent())) {
                current.setPermission(15);
                current.setAllowedToCreateSubUser(true);
            }

            List<Account> child = repository.findByParent(name);
            if (child != null && child.size() > 0) {
                List<AccountDto> childDto = child.stream().map(item -> new AccountDto(item)).collect(Collectors.toList());
                current.setChild(childDto);
            }

            ShopSummary shopSummary = shopService.getShopSummary(name);
            if (shopSummary != null)
                current.setShopSummary(shopSummary);

            IpSummary ipSummary = ipResourceService.getIpSummary(name);
            if (ipSummary != null)
                current.setIpSummary(ipSummary);

            List<String> whiteList = whiteSiteRepository.findAll().stream().map(x -> x.getWhite()).collect(Collectors.toList());
            if (current.getWhiteList() != null && current.getWhiteList().size() > 0)
                whiteList.addAll(current.getWhiteList());

            List<String> queryIpUrlList = queryIpUrlListRepository.findAll().stream().map(x -> x.getUrl()).collect(Collectors.toList());
            if (current.getQueryIpUrlList() != null && current.getQueryIpUrlList().size() > 0)
                queryIpUrlList.addAll(current.getQueryIpUrlList());

            current.setQueryIpUrlList(queryIpUrlList);
            current.setWhiteList(whiteList);

            if (account.getPromotion() != null) {
                int invitedUsers = repository.countByPromotionCodeAndParentIsNull(account.getPromotion().getCode());
                String code = account.getPromotion().getCode();
                List<Account> promotes = repository.findByPromotionCode(code);
                double totalCommission = 0;
                totalCommission = 0;
                for (Account promote : promotes) {
                    List<UserPrePaidBilling> userPrePaidBillings = prePaidBillingRepository.findByAdministratorAndYearAndMonthAndPayMethodIn(promote.getName(),
                            YearMonth.now().getYear(),
                            YearMonth.now().getMonthValue(), Arrays.asList(1, 2, 3));
                    totalCommission += userPrePaidBillings.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                }

                account.getPromotion().setCommission((int) totalCommission);
                account.getPromotion().setInvitedUsers(invitedUsers);
                repository.save(account);
            }

            resultDto.setStatus(0);
            resultDto.setData(current);
        } catch (Exception e) {

            log.error("eeee", e);

            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", AccountErrorCode.NAMENOTEXIST);
            statusInfo.put("message", (AuthErrorCode.NAMENOTEXIST.getReason()));
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    public void deleteByName(String name) {
        userAuthService.deleteUser(name);
        repository.deleteById(name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Account create(User user) {
        return create(user, null);
    }

    public void deleteSub(String parent, String child) {
        Assert.hasText(parent);
        Account childAccount = repository.findByName(child);
        if (childAccount == null || !parent.equals(childAccount.getParent())) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }

        userAuthService.deleteUser(child);
        repository.delete(childAccount);
        Account parentAccount = repository.findByName(parent);
        parentAccount.setChildCount(parentAccount.getChildCount() - 1);
        repository.save(parentAccount);
    }

    @Override
    public Account createWithSms(User user) {

        Account existing = repository.findByName(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByPhoneNumber(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByPhoneNumber(user.getPhone());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getPhone());

        existing = repository.findByName(user.getPhone());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getPhone());

        Otp otp = otpRepository.findByPhoneAndCreatedAtGreaterThanEqual(user.getUsername(), Instant.now().minusSeconds(600).toEpochMilli());

        if (otp == null) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        }
        if (!otp.getOtp().equals(user.getAuthCode())) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        } else {
//            otpRepository.delete(otp);
        }

        Account account = new Account();

        account.setName(user.getUsername());
        account.setLastSeen(new Date());
        account.setEmail(user.getEmail());
        account.setPhoneNumber(user.getUsername());
        account.setPermission(15);
        if (user.getPromotionCode() != null) {
            account.setPromotionCode(user.getPromotionCode());
        }

        Account inviter = repository.findByPromotion(user.getPromotionCode());
        if (inviter != null) {
            inviter.getPromotion().setInvitedUsers(inviter.getPromotion().getInvitedUsers() + 1);
            repository.save(inviter);
        }

        Account pExisting = null;
        String random = null;
        do {
            random = String.valueOf((int) ((Math.random() * 9 + 1) * 100000));
            pExisting = repository.findByPromotion(random);
        } while (pExisting != null);

        Promotion promotion = new Promotion();
        promotion.setCode(random);
        promotion.setInvitedUsers(0);
        promotion.setCommission(0);
        account.setPromotion(promotion);
        account.setAllowedToCreateSubUser(true);

        List<String> whiteList = new ArrayList<>();
        if (user.getWhiteList() != null && user.getWhiteList().size() > 0)
            whiteList.addAll(user.getWhiteList());
        account.setWhiteList(whiteList);

        user.setEnabled(true);

        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(user));
        repository.save(account);

        JSONObject param = new JSONObject();
        param.put("newuser", account.getName());
        if (inviter != null)
            param.put("referral", inviter.getName() + "(" + user.getPromotionCode() + ")");
        else
            param.put("referral", "123456");
        com.edgec.browserbackend.common.commons.utils.SmsUtils.sendNewAccountSms("15919921106", com.edgec.browserbackend.common.commons.utils.SmsUtils.SmsTemplateCode.NEWACCOUNT, param);

        if (inviter != null) {
            paymentService.bankTransferInsertion(account.getName(), 12);
        }

        log.info("new account has been created: " + account.getName());

        notifyCustomerRegister(account);
        return account;
    }

    public Account create(User user, String parentName) {

        Account existing = repository.findByName(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByPhoneNumber(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByEmail(user.getEmail());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.EMAILEXIST, "email already exists: " + user.getEmail());
        user.setEnabled(false);
        Account account = new Account();
        account.setName(user.getUsername());
        account.setLastSeen(new Date());
        account.setEmail(user.getEmail());

        if (StringUtils.isEmpty(parentName)) {
            account.setAllowedToCreateSubUser(true);
            account.setPermission(Integer.valueOf("1111", 2));
        } else {
            account.setAllowedToCreateSubUser(user.isAllowedToCreateSubUser());
            account.setPermission(user.getPermission());
        }

        account.setParent(parentName);

        emailService.sendEmailVerification(user.getUsername(), user.getEmail(), user.getVerificationCode());

        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(user));
        repository.save(account);

        log.info("new account has been created: " + account.getName());

//        SmsUtils.notifyNewUserRegistered();
        notifyCustomerRegister(account);
        return account;
    }

    private void notifyCustomerRegister(Account contactUs) {
        StringBuilder sb = new StringBuilder();
        sb.append("Name: " + contactUs.getName() + "<br/>");
        sb.append("Email: " + contactUs.getEmail() + "<br/>");
        sb.append("Phone: " + contactUs.getPhoneNumber() + "<br/>");
        sb.append("Company: " + contactUs.getCompanyName() + "<br/>");
        sb.append("Title: " + contactUs.getJobTitle() + "<br/>");
        try {
            emailService.sendHtmlMail("sales@cloudam.io", "新客户注册:" + contactUs.getName(), sb.toString());
        } catch (Exception e) {
            log.error("sending email fails on customer regisration", e);
        }

    }

    @Override
    public List<UserDto> getAllDesendentUsers(String name, int level) {

        if (level == -1) {
            //find its parent.
            String parent = repository.findByName(name).getParent();

            if (StringUtils.isEmpty(parent))
                return Arrays.asList();
            UserDto dto = new UserDto();
            dto.setUsername(parent);
            return Arrays.asList(dto);
        }

        List<Account> accounts = repository.findByParent(name);

        if (accounts == null || accounts.size() <= 0)
            return new ArrayList<>();


        List<UserDto> collect = accounts.stream().map(a -> {
            UserDto user = new UserDto();
            user.setUsername(a.getName());
            user.setEmail(a.getEmail());
            return user;
        }).collect(Collectors.toList());

        if (level <= 0)
            return collect;
        // 遍历获取所有子员工
        List<UserDto> subUserList = new ArrayList<>();
        for (UserDto userDto : collect) {
            subUserList.addAll(getAllDesendentUsers(userDto.getUsername(), level));
        }
        collect.addAll(subUserList);
        return collect;
    }

    @Override
    public SubUserPageResultDto getSubUserList(String name, int page, int amount) {
        if (amount > 100)
            amount = 100;
        Pageable pageable = PageRequest.of(page, amount);
        Page<Account> accounts = repository.findByParent(name, pageable);
        SubUserPageResultDto subUserPageResultDto = new SubUserPageResultDto();

        if (accounts == null || accounts.getNumberOfElements() <= 0)
            return subUserPageResultDto;

        List<SubUsersDto> subUsersDtoList = new ArrayList<>();

        accounts.getContent().forEach(x -> {
            subUsersDtoList.add(new SubUsersDto(x, x.getPassword()));
        });

        subUserPageResultDto.setUserList(subUsersDtoList);
        PageInfo pageInfo = new PageInfo();
        pageInfo.setTotalItems((int) accounts.getTotalElements());
        pageInfo.setCurrentPage(accounts.getNumber());
        pageInfo.setTotalPages(accounts.getTotalPages());
        subUserPageResultDto.setUserPage(pageInfo);
        return subUserPageResultDto;
    }

    public Account createSub(String name, AccountDto user) {
        Account existing = repository.findByName(name);
        if (existing == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + name);
        if (!existing.isAllowedToCreateSubUser() && !StringUtils.isEmpty(existing.getParent()))
            throw new ClientRequestException(AccountErrorCode.NOTALLOWEDTOCREATESUBUSER, "Not allowed to create sub user");

        if (existing.getChildCount() >= 1000) {
            throw new ClientRequestException(AccountErrorCode.CHILDMAX, "account cannot have more children");
        }

        existing = repository.findByName(user.getName());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getName());

        if (!StringUtils.isEmpty(user.getEmail())) {
            existing = repository.findByEmail(user.getEmail());
            if (existing != null)
                throw new ClientRequestException(AccountErrorCode.EMAILEXIST, "email already exists: " + user.getEmail());
        }

        if (!StringUtils.isEmpty(user.getPhoneNumber())) {
            existing = repository.findByPhoneNumber(user.getPhoneNumber());
            if (existing != null)
                throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getEmail());
        }

        existing = repository.findByPhoneNumber(user.getName());
        if (existing != null) {
            throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getEmail());
        }

        User authUser = new User();
        authUser.setUsername(user.getName());
        authUser.setPassword(user.getPassword());
        authUser.setPhone(user.getPhoneNumber());
        authUser.setEmail(user.getEmail());
        authUser.setEnabled(true);
        authUser.setPermission(user.getPermission());
        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(authUser));

        Account account = new Account(user);
        List<String> whiteList = new ArrayList<>();
        if (user.getWhiteList() != null && user.getWhiteList().size() > 0)
            whiteList.addAll(user.getWhiteList());
        account.setWhiteList(whiteList);
        account.setPhoneNumber(user.getPhoneNumber());
        account.setEmail(user.getEmail());
        account.setParent(name);
        repository.save(account);


//        emailService.sendEmailVerification(user.getUsername(), user.getEmail(), user.getVerificationCode());
        log.info("new account has been created: " + account.getName());
        return account;
    }

    public static String makeRandomPassword(int len) {
        char charr[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890~!@$%^&*.?".toCharArray();
        StringBuilder sb = new StringBuilder();
        Random r = new Random();
        for (int x = 0; x < len; ++x) {
            sb.append(charr[r.nextInt(charr.length)]);
        }
        return sb.toString();
    }

    @Override
    public void createSubUsers(String name, SubUsersRequestDto subUsersRequestDto) {
        Account existing = repository.findByName(name);
        if (existing == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + name);
        if (!existing.isAllowedToCreateSubUser() && !StringUtils.isEmpty(existing.getParent()))
            throw new ClientRequestException(AccountErrorCode.NOTALLOWEDTOCREATESUBUSER, "Not allowed to create sub user");

        if (existing.getChildCount() + subUsersRequestDto.getAmount() >= 1000) {
            throw new ClientRequestException(AccountErrorCode.CHILDMAX, "account cannot have more children");
        }

        int count = existing.getChildCount();
        int time = subUsersRequestDto.getAmount();
        for (int i = 0; i < time; i++) {
            if (count >= 1000)
                throw new ClientRequestException(AccountErrorCode.CHILDMAX, "account cannot have more children");
            AccountDto user = new AccountDto();
            if (i < 10)
                user.setName(existing.getName() + "00" + i);
            else if (i < 100)
                user.setName(existing.getName() + "0" + i);
            else
                user.setName(existing.getName() + i);

            Account child = repository.findByName(user.getName());
            if (child != null) {
                time++;
                continue;
            }

            User authUser = new User();
            authUser.setUsername(user.getName());
            String password;
            if (org.apache.commons.lang3.StringUtils.isNotBlank(subUsersRequestDto.getPassword()))
                password = Aes.aesDecrypt(subUsersRequestDto.getPassword());
            else
                password = makeRandomPassword(8);

            authUser.setPassword(password);
            authUser.setPhone(existing.getPhoneNumber());
            authUser.setEnabled(true);
            if (subUsersRequestDto.getPermission() != -1)
                authUser.setPermission(subUsersRequestDto.getPermission());
            else
                authUser.setPermission(0);
            userAuthService.create(new com.edgec.browserbackend.auth.domain.User(authUser));

            Account account = new Account(user);
            account.setPhoneNumber(existing.getPhoneNumber());
            account.setParent(name);
            if (subUsersRequestDto.getPermission() != -1)
                account.setPermission(subUsersRequestDto.getPermission());
            else
                account.setPermission(0);
            if (existing.getPromotionCode() != null) {
                account.setPromotionCode(existing.getPromotionCode());
            }
            List<String> whiteList = new ArrayList<>();
            if (user.getWhiteList() != null && user.getWhiteList().size() > 0)
                whiteList.addAll(user.getWhiteList());
            account.setWhiteList(whiteList);
            String newpassword = Aes.aesEncrypt(password);
            account.setPassword(newpassword);
            repository.save(account);

//        emailService.sendEmailVerification(user.getUsername(), user.getEmail(), user.getVerificationCode());
            log.info("new account has been created: " + account.getName());
            count++;
        }

        existing.setChildCount(count);
        repository.save(existing);
    }


    public Account saveSub(String name, AccountDto user) {
        Account childAccount = repository.findByName(user.getName());
        if (childAccount == null || !name.equals(childAccount.getParent())) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }
        childAccount.setAllowedToCreateSubUser(user.isAllowedToCreateSubUser());
        childAccount.setPhoneNumber(user.getPhoneNumber());
        childAccount.setJobTitle(user.getJobTitle());
        childAccount.setCompanyName(user.getCompanyName());
        childAccount.setPermission(user.getPermission());
        List<String> whiteList = childAccount.getWhiteList();
        if (user.getWhiteList() != null && user.getWhiteList().size() > 0)
            whiteList.addAll(user.getWhiteList());
        childAccount.setWhiteList(whiteList);
        repository.save(childAccount);
        return childAccount;
    }

    @Override
    public void saveSubUsers(String name, SubUsersRequestDto subUsersRequestDto) {
        for (String username : subUsersRequestDto.getUsernames()) {
            Account childAccount = repository.findByName(username);
            if (childAccount == null || !name.equals(childAccount.getParent())) {
                throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
            }

            if (subUsersRequestDto.getPassword() != null) {
                com.edgec.browserbackend.auth.domain.User user = new com.edgec.browserbackend.auth.domain.User();
                user.setUsername(username);
                if (subUsersRequestDto.getPassword() != null) {
                    user.setPassword(Aes.aesDecrypt(subUsersRequestDto.getPassword()));
                }
                userAuthService.resetUserPassword(user);
            }

            if (subUsersRequestDto.getNickname() != null)
                childAccount.setNickname(subUsersRequestDto.getNickname());
            if (subUsersRequestDto.getPermission() != -1)
                childAccount.setPermission(subUsersRequestDto.getPermission());
            if (subUsersRequestDto.getComment() != null)
                childAccount.setComment(subUsersRequestDto.getComment());
            List<String> whiteList = new ArrayList<>();
            if (subUsersRequestDto.getWhiteList() != null && subUsersRequestDto.getWhiteList().size() > 0)
                whiteList.addAll(subUsersRequestDto.getWhiteList());
            childAccount.setWhiteList(whiteList);
            if (org.apache.commons.lang3.StringUtils.isNotBlank(subUsersRequestDto.getPassword()))
                childAccount.setPassword(subUsersRequestDto.getPassword());
            repository.save(childAccount);
        }

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Account saveChanges(String name, Account update) {

        Account account = repository.findByName(name);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "can't find account with name " + name);
        String preEmail = account.getEmail();
        if (update.getNote() != null)
            account.setNote(update.getNote());
        account.setLastSeen(new Date());
        if (update.getCompanyName() != null)
            account.setCompanyName(update.getCompanyName());
        if (update.getJobTitle() != null)
            account.setJobTitle(update.getJobTitle());
//        account.setPhoneNumber(update.getPhoneNumber());
        if (update.getEmail() != null)
            account.setEmail(update.getEmail());
        if (update.getPermission() != -1)
            account.setPermission(update.getPermission());

        List<String> whiteList = account.getWhiteList();
        if (update.getWhiteList() != null && update.getWhiteList().size() > 0)
            whiteList.addAll(update.getWhiteList());
        account.setWhiteList(whiteList);

        List<String> queryIpUrlList = account.getQueryIpUrlList();
        if (update.getQueryIpUrlList() != null && update.getQueryIpUrlList().size() > 0)
            queryIpUrlList.addAll(update.getQueryIpUrlList());
        account.setQueryIpUrlList(queryIpUrlList);

        if (update.getAuthorized() == 1)
            account.setAuthorized(1);

        log.debug("account {} changes has been saved", name);
        if (!org.apache.commons.lang3.StringUtils.equalsIgnoreCase(preEmail, update.getEmail())) {
            Account account1 = repository.findByEmail(update.getEmail());
            if (account1 != null) {
                throw new ClientRequestException(AccountErrorCode.EMAILEXIST);
            }
            User newuser = new User();
            newuser.setUsername(account.getName());
            newuser.setEmail(update.getEmail());
            userAuthService.updateUser(name, new com.edgec.browserbackend.auth.domain.User(newuser));
        }
        repository.save(account);
        return account;
    }

    @Override
    public void resetPassword(String username, UserDto user) {
        if (user == null)
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + user);

        Account account = repository.findByName(user.getUsername());
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name: " + user);
        }

        User newuser = new User();
        newuser.setUsername(account.getName());
        String code = UUID.randomUUID().toString() + System.currentTimeMillis();
        newuser.setVerificationCode(code);
        userAuthService.reset(new com.edgec.browserbackend.auth.domain.User(newuser));
        emailService.sendResetPassword(account.getName(), account.getEmail(), code);
        log.info("password has been reset for: " + user);

    }

    @Override
    public UserDto determUsernameOrEmail(String text) {
        if (StringUtils.isEmpty(text))
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + text);
        Account account = repository.findByEmail(text);
        if (account == null)
            account = repository.findByName(text);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + text);
        UserDto user = new UserDto();
        user.setEmail(account.getEmail());
        user.setUsername(account.getName());
        return user;
    }

    @Override
    public void resetPasswordWithOtp(UserDto user) {

        Otp otp = otpRepository.findByPhoneAndCreatedAtGreaterThanEqual(user.getUsername(),
                Instant.now().minusSeconds(600).toEpochMilli());
        if (otp == null) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        }
        if (!otp.getOtp().equals(user.getAuthCode())) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        }

        Account account = repository.findById(user.getUsername()).orElse(null);

        if (account == null)
            throw new ClientRequestException(AccountErrorCode.PHONENOTEXIST, "Can't find account with phone " + user.getPhone());

        User client = new User();
        client.setUsername(account.getName());
        client.setPassword(user.getPassword());

        userAuthService.resetUserPassword(new com.edgec.browserbackend.auth.domain.User(client));

    }

    @Override
    public Page<UserPayment> getUserPayment(Pageable pageable, String username) {
        Page<UserPayment> userPayments = userPaymentRepository.findByUsernameAndSucceedOrderByPaymentDateDesc(pageable, username, true);

        return userPayments;
    }

    @Override
    public void sendSmsOtp(String phone) {
        String code = com.edgec.browserbackend.account.service.SmsUtils.sendSmsOTP(phone);
        Otp otp = new Otp();
        otp.setPhone(phone);
        otp.setOtp(code);
        otp.setCreatedAt(Instant.now().toEpochMilli());
        otpRepository.save(otp);
    }

    @Override
    public AccountDto getAccountByCellphone(String cellphone) {
        Account account = repository.findByPhoneNumber(cellphone);
        if (account == null) {
            return null;
        }
        return new AccountDto(account);
    }

    @Override
    public Page<UserPrePaidBilling> listPrepaid(Pageable pageable, String username, String[] chargeType, int year, int month, int day, String zoneId) {

        Page<UserPrePaidBilling> userPrePaidBillings = prePaidBillingRepository.findByUsernameAndChargeTypes(pageable, username, chargeType, year, month, day, zoneId);

        return userPrePaidBillings;
    }

    @Override
    public Page<UserPrePaidBilling> listBills(Pageable pageable, String username, BillQueryCriteriaDto billQueryCriteriaDto, String dateFrom, String dateTo, String zoneId) {
        zoneId = zoneId.replace(" ", "+");
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        List<Integer> dayBegin = listYearMonthDayByStrDate(dateFrom);
        ZonedDateTime zonedDateTimeDayBegin = ZonedDateTime.of(dayBegin.get(0), dayBegin.get(1), dayBegin.get(2), 0, 0, 0, 0, zoneOffset);
        List<Integer> dayEnd = listYearMonthDayByStrDate(dateTo);
        ZonedDateTime zonedDateTimeDayEnd = ZonedDateTime.of(dayEnd.get(0), dayEnd.get(1), dayEnd.get(2), 0, 0, 0, 0, zoneOffset);
        long dayBeginTime = zonedDateTimeDayBegin.withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
        long dayEndTime = zonedDateTimeDayEnd.plusDays(1).withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;

        return prePaidBillingRepository.findBillsByCondition(pageable, username, billQueryCriteriaDto, dayBeginTime, dayEndTime);
    }


    public static void main(String[] args) {
        String zoneId = "+08:00";
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        String dataFrom = "2019-11-20";
        // 获取北京时间 2019年11月20日 0点
        ZonedDateTime zonedDateTime = ZonedDateTime.of(2019, 11, 20, 0, 0, 0, 0, zoneOffset);
        long dayBeginTime = zonedDateTime.withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
        System.out.println(dayBeginTime);
    }


    //根据传入的格式为 yyyy-mm-dd来获取年月日，
    private List<Integer> listYearMonthDayByStrDate(String strDate) {
        List<Integer> nums = new ArrayList<>();
        String[] dates = strDate.split("-");
        int year = 0, month = 0, day = 0;
        if (dates.length > 2) {
            year = Integer.parseInt(dates[0]);
            month = Integer.parseInt(dates[1]);
            day = Integer.parseInt(dates[2]);
        }
        nums.add(year);
        nums.add(month);
        nums.add(day);
        return nums;
    }

    @Override
    public Page<Account> listAccountBySingupDate(Pageable pageable, Date beginDate, Date endDate) {
        return repository.findAllBySignupDateBetween(pageable, beginDate, endDate);
    }

    @Override
    public List<UserBillList> findOverviewByYearAndMonth(String username, int year, int month, String zoneId, Services service) {
        List<UserBillList> userBillLists = new ArrayList<>();
        List<UserPrePaidBilling> userPrePaidBilling = prePaidBillingRepository
                .findOverviewByYearAndMonth(username, year, month, zoneId, service);
        userPrePaidBilling.forEach(x -> {
            UserBillList userBillList = new UserBillList();
            userBillList.setPeriod(year + "-" + (month < 10 ? "0" + month : month));
            userBillList.setBill(x);
            userBillList.setStatus(x.getStatus());
            userBillLists.add(userBillList);
        });
        return userBillLists;
    }

    @Override
    public List<Invoice> findByUsername(String username) {
        List<Invoice> invoices = invoiceRepository.findByUsernameOrderByDateDesc(username);
        invoices.forEach(x -> {
            x.setExpressInformation(x.getUsername() + " " + x.getPhone() + " " + x.getExpressInformation());
        });
        return invoices;
    }

    @Override
    public boolean writeInvoice(Invoice invoice) {
        int num = invoiceSize(invoice.getUsername());
        if (invoice.getAmount() > num || invoice.getAmount() < 100) {
            return false;
        }
        invoiceRepository.save(invoice);
        return true;
    }

    @Override
    public int invoiceSize(String username) {
        List<UserPayment> amounts = userPaymentRepository.findByUsernameAndSucceedAndTradeNoGreaterThan(username, true, "201911290000000000");
        int amount = amounts.stream().map(UserPayment::getAmount).mapToInt((x) -> x).sum();
        List<Invoice> invoices = invoiceRepository.findByUsername(username);
        float invoiceAmount = (float) invoices.stream().map(Invoice::getAmount).mapToDouble((x) -> x).sum();
        return (int) (amount - invoiceAmount);
    }

    @Override
    public void updateUserToken(String username, String token) {
        Account byName = repository.findByName(username);
        byName.setToken(token);
        repository.save(byName);
    }

    @Override
    public void authorizeCompany(String username, CompanyAuthorizeDto companyAuthorizeDto) {
        String companyLicenseId = null;
        String coporationLicenseFront = null;
        String coporationLicenseBack = null;
        String agencyFront = null;
        String agencyBack = null;
        CompanyAuthorize existing = companyAuthorizeRepository.findByUsername(username);
        if (existing != null) {
            throw new ClientRequestException(BrowserErrorCode.COMPANYAUTHORIZEEXIST);
        }
        try {
            Account account = repository.findByName(username);
            if (account == null)
                throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
            companyLicenseId = companyAuthorizeRepository.saveFile(companyAuthorizeDto.getCompanyLicense(), "companyLicense-" + username);

            coporationLicenseFront = companyAuthorizeRepository.saveFile(companyAuthorizeDto.getCoporationLicense_front(), "coporationLicense-front-" + username);
            coporationLicenseBack = companyAuthorizeRepository.saveFile(companyAuthorizeDto.getCoporationLicense_back(), "coporationLicense-back-" + username);

            if (companyAuthorizeDto.getAgency_front() != null && companyAuthorizeDto.getAgency_back() != null) {
                agencyFront = companyAuthorizeRepository.saveFile(companyAuthorizeDto.getAgency_front(), "agency-front-" + username);
                agencyBack = companyAuthorizeRepository.saveFile(companyAuthorizeDto.getAgency_back(), "agency-back-" + username);
            } else if (!(companyAuthorizeDto.getAgency_front() == null && companyAuthorizeDto.getAgency_back() == null))
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            CompanyAuthorize companyAuthorize = new CompanyAuthorize(companyAuthorizeDto);
            companyAuthorize.setUsername(username);
            companyAuthorize.setCompanyLicenseId(companyLicenseId);
            companyAuthorize.setCoporationLicenseFront(coporationLicenseFront);
            companyAuthorize.setCoporationLicenseBack(coporationLicenseBack);
            if (agencyFront != null && agencyBack != null) {
                companyAuthorize.setAgencyFront(agencyFront);
                companyAuthorize.setAgencyBack(agencyBack);
            }
            companyAuthorizeRepository.save(companyAuthorize);
            account.setAuthorized(1);
            repository.save(account);
        } catch (Exception e) {
            if (agencyBack != null)
                companyAuthorizeRepository.deleteFile(agencyBack);
            if (agencyFront != null)
                companyAuthorizeRepository.deleteFile(agencyFront);
            if (coporationLicenseBack != null)
                companyAuthorizeRepository.deleteFile(coporationLicenseBack);
            if (coporationLicenseFront != null)
                companyAuthorizeRepository.deleteFile(coporationLicenseFront);
            if (companyLicenseId != null)
                companyAuthorizeRepository.deleteFile(companyLicenseId);
            log.error(e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }

    }

    @Override
    public CompanyAuthorizeDto getAuthorizeDetails(String username) {
        CompanyAuthorize companyAuthorize = companyAuthorizeRepository.findByUsername(username);
        if (companyAuthorize == null)
            return null;
        CompanyAuthorizeDto companyAuthorizeDto = new CompanyAuthorizeDto(companyAuthorize);
        return companyAuthorizeDto;
    }

    @Override
    public File getAuthorizeFiles(String username) {
        CompanyAuthorize companyAuthorize = companyAuthorizeRepository.findByUsername(username);
        if (companyAuthorize == null)
            return null;
        File companyLicense = companyAuthorizeRepository.getFile(companyAuthorize.getCompanyLicenseId());
        File coporationLicense_Front = companyAuthorizeRepository.getFile(companyAuthorize.getCoporationLicenseFront());
        File coporationLicense_Back = companyAuthorizeRepository.getFile(companyAuthorize.getCoporationLicenseBack());
        File agencyLicense_Front = null;
        File agencyLicense_Back = null;
        if (companyAuthorize.getAgencyFront() != null)
            agencyLicense_Front = companyAuthorizeRepository.getFile(companyAuthorize.getAgencyFront());
        if (companyAuthorize.getAgencyBack() != null)
            agencyLicense_Back = companyAuthorizeRepository.getFile(companyAuthorize.getAgencyBack());
        try {
            File tempFile = File.createTempFile("tempFile", "zip");
            List<File> files = Arrays.asList(companyLicense, coporationLicense_Front, coporationLicense_Back);
            if (agencyLicense_Front != null)
                files.add(agencyLicense_Front);
            if (agencyLicense_Back != null)
                files.add(agencyLicense_Back);
            tempFile = FileUtil.putBatchFilesInZip(files, tempFile);
            companyLicense.delete();
            coporationLicense_Front.delete();
            coporationLicense_Back.delete();
            return tempFile;
        } catch (Exception e) {
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public boolean setAuthorize(String username, boolean isAgree) {
        try {
            Account account = repository.findByName(username);
            if (account == null)
                throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
            if (isAgree) {
                account.setAuthorized(2);
                CompanyAuthorize companyAuthorize = companyAuthorizeRepository.findByUsername(username);
                account.setCompanyName(companyAuthorize.getCompanyName());
                repository.save(account);
            }
            else {
                account.setAuthorized(3);
                account.setCompanyName(null);
                repository.save(account);

                CompanyAuthorize companyAuthorize = companyAuthorizeRepository.findByUsername(username);
                if (companyAuthorize != null) {
                    if (companyAuthorize.getCompanyLicenseId() != null)
                        companyAuthorizeRepository.deleteFile(companyAuthorize.getCompanyLicenseId());
                    if (companyAuthorize.getCoporationLicenseFront() != null)
                        companyAuthorizeRepository.deleteFile(companyAuthorize.getCoporationLicenseFront());
                    if (companyAuthorize.getCoporationLicenseBack() != null)
                        companyAuthorizeRepository.deleteFile(companyAuthorize.getCoporationLicenseBack());
                    if (companyAuthorize.getAgencyFront() != null)
                        companyAuthorizeRepository.deleteFile(companyAuthorize.getAgencyFront());
                    if (companyAuthorize.getAgencyBack() != null)
                        companyAuthorizeRepository.deleteFile(companyAuthorize.getAgencyBack());
                    companyAuthorizeRepository.delete(companyAuthorize);
                }
            }
            return true;
        } catch (Exception e) {
            log.error(e.getMessage());
            return false;
        }
    }
}
