package com.edgec.browserbackend.account.controller;

import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.dto.*;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.account.service.EmailService;
import com.edgec.browserbackend.account.service.PaymentService;
import com.edgec.browserbackend.alipay.AlipayConfig;
import com.edgec.browserbackend.alipay.VpsAlipayConfig;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.common.auth.Securitys;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.utils.DateConverter;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.security.Principal;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

@RestController
@RequestMapping("/user")
public class AccountController {

    public static final String WECHAT_PAY_CALLBACK_URL = "https://cloudam.cn/accounts/0xwxpaycallback/";

    public static final String ALIPAY_PAY_CALLBACK_URL = "https://cloudam.cn/accounts/0xalipaycallback/";

    @Autowired
    private AccountService accountService;

    @Autowired
    private PaymentService paymentService;

    @Autowired
    private EmailService emailService;

    private static final Logger logger = LoggerFactory.getLogger(AccountController.class);

    @RequestMapping(path = "/prechargeIp/{name}/{amount}/{money}", method = RequestMethod.POST)
    public IpChargeResultDto preChargeIp(@PathVariable String name, @PathVariable int amount, @PathVariable double money) {
        return accountService.preChargeIp(name, amount, money);
    }

    @RequestMapping(path = "/precharge/money", method = RequestMethod.POST)
    public IpChargeResultDto preChargeByMoney(@RequestParam("accountId") String name, @RequestParam("money") double money) {
        return accountService.preChargeByMoney(name, money);
    }

    @RequestMapping(path = "/charge/money", method = RequestMethod.PUT)
    public IpChargeResultDto chargeByMoney(@RequestParam("accountId") String name, @RequestParam("money") double money, @RequestBody IpChargeRequestDto requestDto) {
        return accountService.chargeByMoney(name, money, requestDto);
    }


    @RequestMapping(path = "/information", method = RequestMethod.POST)
    public ResultDto getAccount(Principal principal) {
        return accountService.getAccountByName(principal.getName());
    }

    @RequestMapping(path = "/{name}/children/{level}", method = RequestMethod.POST)
    public ResultDto getDesendentUsers(@PathVariable String name, @PathVariable int level) {
        ResultDto resultDto = new ResultDto();
        try {
            List<UserDto> userDtos = accountService.getAllDesendentUsers(name, level);
            resultDto.setStatus(0);
            resultDto.setData(userDtos);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

//    @RequestMapping(path = "/current", method = RequestMethod.DELETE)
//    public void deleteCurrentAccount(Principal principal) {
//         accountService.deleteByName(principal.getName());
//    }

    @RequestMapping(path = "/update", method = RequestMethod.POST)
    public ResultDto saveCurrentAccount(Principal principal, @Valid @RequestBody Account account) {
        ResultDto resultDto = new ResultDto();
        try {
            Account account1 = accountService.saveChanges(principal.getName(), account);
            resultDto.setStatus(0);
            resultDto.setData(account1);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(path = "/authCode", method = RequestMethod.POST)
    public ResultDto requestOTP(@RequestBody MobileDto mobile) {
        ResultDto resultDto = new ResultDto();
        try {
            accountService.sendSmsOtp(mobile);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            logger.error("fail to send sms", e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 用户注册
     */
    @RequestMapping(path = "/signUp", method = RequestMethod.POST)
    public ResultDto createNewAccount(@Valid @RequestBody User user) {
        ResultDto resultDto = new ResultDto();
        try {
            Account account = new Account();
            if (!StringUtils.isEmpty(user.getEmail())) {
                account = accountService.create(user);
            } else {
                account = accountService.createWithSms(user);
            }

            resultDto.setStatus(0);
            resultDto.setData(account);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(path = "/contactus", method = RequestMethod.POST)
    public void contactus(Principal principal, @Valid @RequestBody ContactUs contactUs) {

        StringBuilder sb = new StringBuilder();
        sb.append("Name: " + contactUs.getName() + "<br/>");
        sb.append("Email: " + contactUs.getEmail() + "<br/>");
        sb.append("Phone: " + contactUs.getPhone() + "<br/>");
        sb.append("Company: " + contactUs.getCompany() + "<br/>");
        sb.append("Title: " + contactUs.getTitle() + "<br/>");
        sb.append("Details: " + contactUs.getDetail() + "<br/>");
        emailService.sendHtmlMail("sales@cloudam.io", "客户联系:" + contactUs.getName(), sb.toString());
    }


    /**
     * 费用明细
     */
    @RequestMapping(path = "/userbilling", method = RequestMethod.POST)
    public ResultDto getUserBills0(Principal principal, @RequestBody UserPrePaidBillingRequestDto userPrePaidBillingRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            resultDto.setData(accountService.getUserPrePaidBills(principal.getName(), userPrePaidBillingRequestDto));
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    // 消费总览
    @RequestMapping(path = "/v2/current/userbilling", method = RequestMethod.GET)
    public List<UserBillList> getUserBills(@RequestParam(value = "strDate", defaultValue = "") String strDate,
                                           @RequestParam(value = "services", defaultValue = "") String services,
                                           @RequestParam(value = "zone", defaultValue = "+08:00") String zoneId,
                                           Principal principal) {
        Services service = null;
        if (!StringUtils.isEmpty(services)) {
            service = Services.valueOf(services);
        }
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        if (StringUtils.isEmpty(strDate)) {
            strDate = DateTimeFormatter.ofPattern("yyyy-MM").format(ZonedDateTime.now(zoneOffset));
        }
        String[] datas = strDate.split("-");
        int year = Integer.parseInt(datas[0]);
        int month = 0;
        if (datas.length > 1) {
            month = Integer.parseInt(datas[1]);
        }
        return accountService.findOverviewByYearAndMonth(principal.getName(), year, month, zoneId, service);
    }

    @RequestMapping(path = "/current/updatetoken", method = RequestMethod.POST)
    void updateUserToken(@RequestParam("username") String username, @RequestParam("token") String token) {
        accountService.updateUserToken(username, token);
    }

    @RequestMapping(path = "/resetpassword", method = RequestMethod.POST)
    public ResultDto forgetPassword(Principal principal, @RequestBody UserDto user) {
        ResultDto resultDto = new ResultDto();
        try {
            accountService.resetPassword(principal.getName(), user);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(path = "/deter/{text}", method = RequestMethod.GET)
    public UserDto determinateUser(@PathVariable String text) {
        return accountService.determUsernameOrEmail(text);
    }

    @RequestMapping(path = "/forgotPassword", method = RequestMethod.POST)
    public ResultDto resetWithOtp(@Valid @RequestBody UserDto user) {
        ResultDto resultDto = new ResultDto();
        try {
            accountService.resetPasswordWithOtp(user);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 使用微信进行账户充值
     */
    @RequestMapping(path = "/0xwxpayputorder/{amount}", method = RequestMethod.GET)
    public UserPaymentDto wechatPayCallback(Principal principal, @PathVariable int amount) {
        return paymentService.wxPutPayOrder(principal.getName(), amount);
    }

    /**
     * 使用微信进行账户充值成功后，微信会调用该接口
     */
    @RequestMapping(path = "/0xwxpaycallback/{tradno}", method = RequestMethod.GET)
    public String wechatPayCallback(@PathVariable String tradno) {
        return paymentService.wechatPayCallback(tradno);
    }

    /**
     * 使用微信进行账户充值成功后，微信会调用该接口
     */
    @RequestMapping(path = "/0xwxcheckorderstatus/{tradno}/{chargeType}", method = RequestMethod.GET)
    public UserPaymentDto wechatCheckOrderStatus(Principal principal, @PathVariable String tradno, @PathVariable int chargeType) {
        return paymentService.wxCheckOrderStatus(tradno, chargeType);
    }

    @RequestMapping(path = "/wxpay/checkstatus/{tradno}/{chargeType}", method = RequestMethod.GET)
    public UserPaymentDto wechatPayCheckStatus(@PathVariable String tradno, @PathVariable int chargeType) {
        return paymentService.wxCheckOrderStatus(tradno, chargeType);
    }

    /**
     * 使用 支付宝 充值 成功后回调
     */
    @RequestMapping(path = "/0xalipaycallback/{tradno}", method = RequestMethod.GET)
    public void alipayPayCallback(@PathVariable String tradno) {
        paymentService.alipaCallback(tradno);
    }

    /**
     * 使用 支付宝 充值 成功后回调
     */
    @RequestMapping(path = "/0xalicheckorderstatus/{tradno}/{chargeType}", method = RequestMethod.GET)
    public UserPaymentDto alipayCheckOrderStatus(Principal principal, @PathVariable(required = false) String tradno, @PathVariable int chargeType) {
        return paymentService.aliCheckOrderStatus(tradno, chargeType);
    }


    /**
     * 使用 支付宝 充值
     */
    @RequestMapping(path = "/0xalipay/{amount}", method = RequestMethod.GET)
    public void alipayPutOrder(HttpServletRequest request, HttpServletResponse response, @PathVariable int amount) throws Exception {
        String by = request.getParameter("by");
        String form = paymentService.alipayPutPayOrder(request.getUserPrincipal().getName(), amount, by);
        response.setContentType("text/html;charset=utf-8");
        response.getWriter().println(form);

    }

    @RequestMapping(path = "/0xh5wxpayputorder/{amount}", method = RequestMethod.GET)
    public UserPaymentDto h5wechatPayPutOrder(Principal principal, @PathVariable(required = false) int amount, HttpServletRequest request) {
        Enumeration<String> names = request.getHeaderNames();
        while (names.hasMoreElements()) {
            String name = names.nextElement();
            logger.error("微信支付key = " + name + ", value = " + request.getHeader(name));
        }
        String xff = request.getHeader("X-Forwarded-For");
        if (StringUtils.isEmpty(xff)) {
            xff = request.getHeader("X-Real-IP");
        }
        if (StringUtils.isEmpty(xff)) {
            xff = request.getRemoteAddr();
        }
        return paymentService.h5wxPayOrder(xff, principal.getName(), amount);
    }


    //查询该用户所有的充值记录 tested
    @RequestMapping(path = "/fee/userpayment", method = RequestMethod.GET)
    public Page<UserPayment> getUserPaymentInfo(@RequestParam(value = "page", defaultValue = "0") int page,
                                                @RequestParam(value = "size", defaultValue = "10") int size,
                                                Principal principal) {
        Pageable pageable = PageRequest.of(page, size);
        return accountService.getUserPayment(pageable, principal.getName());
    }


    @RequestMapping(path = "/wirteinvoice", method = RequestMethod.POST)
    public boolean writeInvoice(Principal principal, @RequestBody Invoice invoice) {
        invoice.setUsername(principal.getName());
        return accountService.writeInvoice(invoice);
    }

    @RequestMapping(path = "/invoice", method = RequestMethod.GET)
    public List<Invoice> queryInvoice(Principal principal) {
        return accountService.findByUsername(principal.getName());
    }

    @RequestMapping(path = "/invoicesize", method = RequestMethod.GET)
    public int invoicesize(Principal principal) {
        return accountService.invoiceSize(principal.getName());
    }

    @RequestMapping(path = "/cellphone/{cellphone}", method = RequestMethod.GET)
    public AccountDto getAccountByCellphone(@PathVariable String cellphone) {
        return accountService.getAccountByCellphone(cellphone);
    }

    @RequestMapping(path = "/fee/userprepaidbilling", method = RequestMethod.GET)
    public Page<UserPrePaidBilling> queryPerPaid(@RequestParam(value = "page", defaultValue = "0") int page,
                                                 @RequestParam(value = "size", defaultValue = "10") int size,
                                                 @RequestParam(value = "strDate") String strDate,
                                                 @RequestParam(value = "chargeType", defaultValue = "-") String chargeType,
                                                 @RequestParam(value = "zone") String zoneId,
                                                 Principal principal) {

        Pageable pageable = PageRequest.of(page, size);

        String[] datas = strDate.split("-");
        int year = Integer.parseInt(datas[0]);
        int month = Integer.parseInt(datas[1]);
        int day = 0;
        if (datas.length > 2) {
            day = Integer.valueOf(datas[2]);
        }
        String[] chargeTypes = chargeType.split("-");

        return accountService.listPrepaid(pageable, principal.getName(), chargeTypes, year, month, day, zoneId);

    }

    @RequestMapping(path = "/fee/userbilling", method = RequestMethod.POST)
    public Page<UserPrePaidBilling> queryDetailed(@RequestParam(value = "page", defaultValue = "0") int page,
                                                  @RequestParam(value = "size", defaultValue = "10") int size,
                                                  @RequestBody BillQueryCriteriaDto billQueryCriteriaDto,
                                                  @RequestParam(value = "dateFrom") String dateFrom,
                                                  @RequestParam(value = "dateTo") String dateTo,
                                                  @RequestParam(value = "zone") String zoneId,
                                                  Principal principal) {
        Pageable pageable = PageRequest.of(page, size);
        return accountService.listBills(pageable, principal.getName(), billQueryCriteriaDto, dateFrom, dateTo, zoneId);
    }

    @RequestMapping(path = "/0xadministrator/vpsusers", method = RequestMethod.GET)
    @PreAuthorize(Securitys.ADMIN_EL)
    public Page<Account> listAccountBySignupDate(@RequestParam(value = "page", defaultValue = "0") int page,
                                                 @RequestParam(value = "size", defaultValue = "10") int size,
                                                 @RequestParam(value = "beginDate", defaultValue = "0") String beginDateStr,
                                                 @RequestParam(value = "endDate", defaultValue = "0") String endDateStr) {
        Date endDate = new Date();
        if (!endDateStr.equals("0")) {
            endDate = DateConverter.stringToDate(endDateStr);
        }
        Date beginDate = DateUtils.addDays(endDate, -1);
        if (!beginDateStr.equals("0")) {
            beginDate = DateConverter.stringToDate(beginDateStr);
        }
        Pageable pageable = PageRequest.of(page, size);
        return accountService.listAccountBySingupDate(pageable, beginDate, endDate);
    }

    @RequestMapping(path = "/preorder/delete/{username}", method = RequestMethod.POST)
    public String deletePreOrder(@PathVariable("username") String username) {
        return accountService.deletePreOrder(username);
    }

    @RequestMapping(path = "/authorize/add", method = RequestMethod.POST)
    public ResultDto companyAuthorize(Principal principal,
                                      @RequestParam(value = "type", defaultValue = "0") int type,
                                      @RequestParam(value = "companyName") String companyName,
                                      @RequestParam(value = "registerNumber") String registerNumber,
                                      @RequestParam(value = "companyLicense") MultipartFile companyLicense,
                                      @RequestParam(value = "coporationPlace") String coporationPlace,
                                      @RequestParam(value = "coporationLicense_front") MultipartFile coporationLicense_front,
                                      @RequestParam(value = "coporationLicense_back") MultipartFile coporationLicense_back,
                                      @RequestParam(value = "writePerson", defaultValue = "0") int writePerson,
                                      @RequestParam(value = "agency_front", required = false) MultipartFile agency_front,
                                      @RequestParam(value = "agency_back", required = false) MultipartFile agency_back) {
        ResultDto resultDto = new ResultDto();
        try {
            CompanyAuthorizeDto companyAuthorizeDto = new CompanyAuthorizeDto();
            companyAuthorizeDto.setUsername(principal.getName());
            companyAuthorizeDto.setType(type);
            companyAuthorizeDto.setCompanyName(companyName);
            companyAuthorizeDto.setRegisterNumber(registerNumber);
            if (companyLicense == null)
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            companyAuthorizeDto.setCompanyLicense(companyLicense);
            companyAuthorizeDto.setCoporationPlace(coporationPlace);
            if (coporationLicense_front == null || coporationLicense_back == null)
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            companyAuthorizeDto.setCoporationLicense_front(coporationLicense_front);
            companyAuthorizeDto.setCoporationLicense_back(coporationLicense_back);
            companyAuthorizeDto.setWritePerson(writePerson);
            if (writePerson == 1 && (agency_back == null || agency_back == null))
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            if (agency_front != null && agency_back != null) {
                companyAuthorizeDto.setAgency_front(agency_front);
                companyAuthorizeDto.setAgency_back(agency_back);
            }
            accountService.authorizeCompany(principal.getName(), companyAuthorizeDto);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(path = "/ali/login", method = RequestMethod.GET)
    public String save(HttpServletRequest request, HttpServletResponse response) {
        //页面回调地址 必须与应用中的设置一样
        String return_url = "http://https://www.fangguanlian.cn/ali/withdraw";
        //回调地址必须经encode
        return_url = java.net.URLEncoder.encode(return_url);
        //重定向到授权页面
        AlipayConfig alipayConfig = new VpsAlipayConfig();
        return "redirect:" + VpsAlipayConfig.ALIPAY_AUTH_URL + "?app_id=" + alipayConfig.getAPPID() + "&scope=auth_user&redirect_uri=" + return_url;
    }

    @RequestMapping(path = "/ali/withdraw", method = RequestMethod.PUT)
    public ResultDto alipayWithdraw(Principal principal, @RequestParam("account") String account, @RequestParam("realName") String realName, @RequestParam("amount") double amount) {
//        //获取支付宝GET过来反馈信息
//        Map<String,String> params = new HashMap<String,String>();
//        Map requestParams = request.getParameterMap();
//        for (Iterator iter = requestParams.keySet().iterator(); iter.hasNext();) {
//            String name = (String) iter.next();
//            String[] values = (String[]) requestParams.get(name);
//            String valueStr = "";
//            for (int i = 0; i < values.length; i++) {
//                valueStr = (i == (values.length-1)) ? valueStr + values[i]:valueStr + values[i] + ",";
//            }
//            params.put(name, valueStr);
//        }
//
//        String accessToken= AlipaySubmit.buildRequest(params);
//        if(accessToken!=null && accessToken!=""){
//            String uid =  AlipaySubmit.get(accessToken);
//            logger.error(uid);
//            paymentService.alipayWithdraw(request.getUserPrincipal().getName(), uid, )
//        }

        ResultDto resultDto = new ResultDto();
        // 关闭礼金提现相关服务
        /*try {
            if (paymentService.alipayWithdraw(principal.getName(), account, realName, amount)) {
                resultDto.setStatus(0);
            } else {
                resultDto.setStatus(-1);
            }

        } catch (ClientRequestException e) {
            dealClientRequestException(resultDto, e);
        }*/
        resultDto.setStatus(0);
        resultDto.setData("该功能已关闭，如有疑问，请联系客服。");
        return resultDto;
    }


    private void dealClientRequestException(ResultDto resultDto, ClientRequestException e) {
        resultDto.setStatus(-1);
        Map<String, Object> statusInfo = new HashMap<>();
        statusInfo.put("code", e.getErrorCode());
        statusInfo.put("message", e.getMessage());
        resultDto.setStatusInfo(statusInfo);
    }
}
