package com.edgec.browserbackend.account.controller;

import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.dto.MobileDto;
import com.edgec.browserbackend.account.dto.ResultDto;
import com.edgec.browserbackend.account.dto.BillQueryCriteriaDto;
import com.edgec.browserbackend.common.auth.Securitys;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.utils.DateConverter;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.account.service.EmailService;
import com.edgec.browserbackend.account.service.PaymentService;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.security.Principal;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

@RestController
@RequestMapping("user")
public class AccountController {

    public static final String WECHAT_PAY_CALLBACK_URL = "https://cloudam.cn/accounts/0xwxpaycallback/";

    public static final String ALIPAY_PAY_CALLBACK_URL = "https://cloudam.cn/accounts/0xalipaycallback/";

    @Autowired
    private AccountService accountService;

    @Autowired
    private PaymentService paymentService;

    @Autowired
    private EmailService emailService;

    private static final Logger logger = LoggerFactory.getLogger(AccountController.class);

    @RequestMapping(path = "/prechargeIp/{name}/{amount}/{money}", method = RequestMethod.POST)
    public IpChargeResultDto preChargeIp(@PathVariable String name, @PathVariable int amount, @PathVariable double money) {
        return accountService.preChargeIp(name, amount, money);
    }

    @RequestMapping(path = "/precharge/money", method = RequestMethod.POST)
    public IpChargeResultDto preChargeByMoney(@RequestParam("accountId") String name, @RequestParam("money") double money) {
        return accountService.preChargeByMoney(name, money);
    }

    @RequestMapping(path = "/charge/money", method = RequestMethod.PUT)
    public IpChargeResultDto chargeByMoney(@RequestParam("accountId") String name, @RequestParam("money") double money, @RequestBody IpChargeRequestDto requestDto) {
        return accountService.chargeByMoney(name, money, requestDto);
    }


    @RequestMapping(path = "/information", method = RequestMethod.POST)
    public ResultDto getAccount(Principal principal) {
        return accountService.getAccountByName(principal.getName());
    }

    @RequestMapping(path = "/{name}/children/{level}", method = RequestMethod.POST)
    public ResultDto getDesendentUsers(@PathVariable String name, @PathVariable int level) {
        ResultDto resultDto = new ResultDto();
        try {
            List<UserDto> userDtos = accountService.getAllDesendentUsers(name, level);
            resultDto.setStatus(0);
            resultDto.setData(userDtos);
        } catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

//    @RequestMapping(path = "/current", method = RequestMethod.DELETE)
//    public void deleteCurrentAccount(Principal principal) {
//         accountService.deleteByName(principal.getName());
//    }

    @RequestMapping(path = "/updatet", method = RequestMethod.POST)
    public ResultDto saveCurrentAccount(Principal principal, @Valid @RequestBody Account account) {
        ResultDto resultDto = new ResultDto();
        try {
            Account account1 = accountService.saveChanges(principal.getName(), account);
            resultDto.setStatus(0);
            resultDto.setData(account1);
        } catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(path = "/authCode", method = RequestMethod.POST)
    public ResultDto requestOTP(@RequestBody MobileDto mobile) {
        ResultDto resultDto = new ResultDto();
        try {
            accountService.sendSmsOtp(mobile.getMobile());
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(path = "/signUp", method = RequestMethod.POST)
    public ResultDto createNewAccount(@Valid @RequestBody User user) {
        ResultDto resultDto = new ResultDto();
        try {
            Account account = new Account();
            if (!StringUtils.isEmpty(user.getEmail()))
                account = accountService.create(user);
            else
                account =  accountService.createWithSms(user);
            resultDto.setStatus(0);
            resultDto.setData(account);
        } catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(path = "/contactus", method = RequestMethod.POST)
    public void contactus(Principal principal, @Valid @RequestBody ContactUs contactUs) {

        StringBuilder sb = new StringBuilder();
        sb.append("Name: " + contactUs.getName() + "<br/>");
        sb.append("Email: " + contactUs.getEmail() + "<br/>");
        sb.append("Phone: " + contactUs.getPhone() + "<br/>");
        sb.append("Company: " + contactUs.getCompany() + "<br/>");
        sb.append("Title: " + contactUs.getTitle() + "<br/>");
        sb.append("Details: " + contactUs.getDetail() + "<br/>");
        emailService.sendHtmlMail("sales@cloudam.io", "客户联系:" + contactUs.getName(), sb.toString());
    }

    // 消费总览
    @RequestMapping(path = "/current/userbilling", method = RequestMethod.GET)
    @Deprecated
    public List<UserBillList> getUserBills0(Principal principal) {
        return accountService.getUserBills0(principal.getName());
    }

    // 消费总览
    @RequestMapping(path = "/v2/current/userbilling", method = RequestMethod.GET)
    public List<UserBillList> getUserBills(@RequestParam(value = "strDate", defaultValue = "") String strDate,
                                           @RequestParam(value = "services", defaultValue = "") String services,
                                           @RequestParam(value = "zone", defaultValue = "+08:00") String zoneId,
                                           Principal principal) {
        Services service = null;
        if (!StringUtils.isEmpty(services)) {
            service = Services.valueOf(services);
        }
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        if (StringUtils.isEmpty(strDate)) {
            strDate = DateTimeFormatter.ofPattern("yyyy-MM").format(ZonedDateTime.now(zoneOffset));
        }
        String[] datas = strDate.split("-");
        int year = Integer.parseInt(datas[0]);
        int month = 0;
        if (datas.length > 1) {
            month = Integer.parseInt(datas[1]);
        }
        return accountService.findOverviewByYearAndMonth(principal.getName(), year, month, zoneId, service);
    }


    public static void main(String[] args) {
        String intStr = "08";
        int month = Integer.parseInt(intStr);
        System.out.println(month);
    }


    @RequestMapping(path = "/current/updatetoken", method = RequestMethod.POST)
    void updateUserToken(@RequestParam("username") String username, @RequestParam("token") String token) {
        accountService.updateUserToken(username, token);
    }

    @RequestMapping(path = "/resetpassword", method = RequestMethod.POST)
    public ResultDto forgetPassword(Principal principal, @RequestBody UserDto user) {
        ResultDto resultDto = new ResultDto();
        try {
            accountService.resetPassword(principal.getName(), user);
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(path = "/deter/{text}", method = RequestMethod.GET)
    public UserDto determinateUser(@PathVariable String text) {
        return accountService.determUsernameOrEmail(text);
    }

    @RequestMapping(path = "/forgotPassword", method = RequestMethod.POST)
    public ResultDto resetWithOtp(@Valid @RequestBody UserDto user) {
        ResultDto resultDto = new ResultDto();
        try {
            accountService.resetPasswordWithOtp(user);
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(path = "/0xwxpayputorder/{amount}", method = RequestMethod.GET)
    public UserPaymentDto wechatPayCallback(Principal principal, @PathVariable int amount) {
        return paymentService.wxPutPayOrder(principal.getName(), amount);
    }

    @RequestMapping(path = "/0xwxpaycallback/{tradno}", method = RequestMethod.GET)
    public String wechatPayCallback(@PathVariable String tradno) {
        return paymentService.wechatPayCallback(tradno);
    }

    @RequestMapping(path = "/0xwxcheckorderstatus/{tradno}/{more}", method = RequestMethod.GET)
    public UserPaymentDto wechatCheckOrderStatus(Principal principal, @PathVariable String tradno, @PathVariable int more) {
        return paymentService.wxCheckOrderStatus(tradno, more);
    }

    @RequestMapping(path = "/wxpay/checkstatus/{tradno}/{more}", method = RequestMethod.GET)
    public UserPaymentDto wechatPayCheckStatus(@PathVariable String tradno, @PathVariable int more) {
        return paymentService.wxCheckOrderStatus(tradno, more);
    }

    @RequestMapping(path = "/0xalipaycallback/{tradno}", method = RequestMethod.GET)
    public void alipayPayCallback(@PathVariable String tradno) {
        paymentService.alipaCallback(tradno);
    }

    @RequestMapping(path = "/0xalicheckorderstatus/{tradno}/{more}", method = RequestMethod.GET)
    public UserPaymentDto alipayCheckOrderStatus(Principal principal, @PathVariable String tradno, @PathVariable int more) {
        return paymentService.aliCheckOrderStatus(tradno, more);
    }

    @RequestMapping(path = "/alipay/checkstatus/{tradno}/{more}", method = RequestMethod.GET)
    @PreAuthorize("#oauth2.hasScope('server')")
    public UserPaymentDto alipayCheckStatus(@PathVariable String tradno, @PathVariable int more) {
        return paymentService.aliCheckOrderStatus(tradno, more);
    }

    @RequestMapping(path = "/0xalipay/{amount}", method = RequestMethod.GET)
    public void alipayPutOrder(HttpServletRequest request, HttpServletResponse response, @PathVariable int amount) throws Exception {

        String by = request.getParameter("by");
        String form = paymentService.alipayPutPayOrder(request.getUserPrincipal().getName(), amount, by);
        response.setContentType("text/html;charset=utf-8");
        response.getWriter().println(form);

    }

    @RequestMapping(path = "/0xh5wxpayputorder/{amount}", method = RequestMethod.GET)
    public UserPaymentDto h5wechatPayPutOrder(Principal principal, @PathVariable int amount, HttpServletRequest request) {
        Enumeration<String> names = request.getHeaderNames();
        while (names.hasMoreElements()) {
            String name = names.nextElement();
            logger.error("微信支付key = " + name + ", value = " + request.getHeader(name));
        }
        String xff = request.getHeader("X-Forwarded-For");
        if (StringUtils.isEmpty(xff)) {
            xff = request.getHeader("X-Real-IP");
        }
        if (StringUtils.isEmpty(xff)) {
            xff = request.getRemoteAddr();
        }
        return paymentService.h5wxPayOrder(xff, principal.getName(), amount);
    }


    //查询该用户所有的充值记录 tested
    @RequestMapping(path = "/fee/userpayment", method = RequestMethod.GET)
    public Page<UserPayment> getUserPaymentInfo(@RequestParam(value = "page", defaultValue = "0") int page,
                                                @RequestParam(value = "size", defaultValue = "10") int size,
                                                Principal principal) {
        Pageable pageable = PageRequest.of(page, size);
        return accountService.getUserPayment(pageable, principal.getName());
    }


    @RequestMapping(path = "/wirteinvoice", method = RequestMethod.POST)
    public boolean writeInvoice(Principal principal, @RequestBody Invoice invoice) {
        invoice.setUsername(principal.getName());
        return accountService.writeInvoice(invoice);
    }

    @RequestMapping(path = "/invoice", method = RequestMethod.GET)
    public List<Invoice> queryInvoice(Principal principal) {
        return accountService.findByUsername(principal.getName());
    }

    @RequestMapping(path = "/invoicesize", method = RequestMethod.GET)
    public int invoicesize(Principal principal) {
        return accountService.invoiceSize(principal.getName());
    }

    @RequestMapping(path = "/cellphone/{cellphone}", method = RequestMethod.GET)
    public AccountDto getAccountByCellphone(@PathVariable String cellphone) {
        return accountService.getAccountByCellphone(cellphone);
    }

    @RequestMapping(path = "/fee/userprepaidbilling", method = RequestMethod.GET)
    public Page<UserPrePaidBilling> queryPerPaid(@RequestParam(value = "page", defaultValue = "0") int page,
                                                 @RequestParam(value = "size", defaultValue = "10") int size,
                                                 @RequestParam(value = "strDate") String strDate,
                                                 @RequestParam(value = "chargeType", defaultValue = "-") String chargeType,
                                                 @RequestParam(value = "zone") String zoneId,
                                                 Principal principal) {

        Pageable pageable = PageRequest.of(page, size);

        String[] datas = strDate.split("-");
        int year = Integer.valueOf(datas[0]);
        int month = Integer.valueOf(datas[1]);
        int day = 0;
        if (datas.length > 2) {
            day = Integer.valueOf(datas[2]);
        }
        String[] chargeTypes = chargeType.split("-");

        return accountService.listPrepaid(pageable, principal.getName(), chargeTypes, year, month, day, zoneId);

    }

    @RequestMapping(path = "/fee/userbilling", method = RequestMethod.POST)
    public Page<UserPrePaidBilling> queryDetailed(@RequestParam(value = "page", defaultValue = "0") int page,
                                                  @RequestParam(value = "size", defaultValue = "10") int size,
                                                  @RequestBody BillQueryCriteriaDto billQueryCriteriaDto,
                                                  @RequestParam(value = "dateFrom") String dateFrom,
                                                  @RequestParam(value = "dateTo") String dateTo,
                                                  @RequestParam(value = "zone") String zoneId,
                                                  Principal principal) {
        Pageable pageable = PageRequest.of(page, size);
        return accountService.listBills(pageable, principal.getName(), billQueryCriteriaDto, dateFrom, dateTo, zoneId);
    }

    @RequestMapping(path = "/0xadministrator/vpsusers", method = RequestMethod.GET)
    @PreAuthorize(Securitys.ADMIN_EL)
    public Page<Account> listAccountBySignupDate(@RequestParam(value = "page", defaultValue = "0") int page,
                                                 @RequestParam(value = "size", defaultValue = "10") int size,
                                                 @RequestParam(value = "beginDate", defaultValue = "0") String beginDateStr,
                                                 @RequestParam(value = "endDate", defaultValue = "0") String endDateStr) {
        Date endDate = new Date();
        if (!endDateStr.equals("0")) {
            endDate = DateConverter.stringToDate(endDateStr);
        }
        Date beginDate = DateUtils.addDays(endDate, -1);
        if (!beginDateStr.equals("0")) {
            beginDate = DateConverter.stringToDate(beginDateStr);
        }
        Pageable pageable = PageRequest.of(page, size);
        return accountService.listAccountBySingupDate(pageable, beginDate, endDate);
    }

    @RequestMapping(path = "/preorder/delete/{username}", method = RequestMethod.POST)
    public String deletePreOrder( @PathVariable("username") String username) {
        return accountService.deletePreOrder(username);
    }

}
