package com.edgec.browserbackend.account.service.impl;

import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.*;
import com.edgec.browserbackend.account.service.AdministratorService;
import com.edgec.browserbackend.auth.service.UserAuthService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.StringTokenizer;


@Service
@ComponentScan("com.edgec.browserbackend.account.repository")
public class AdministratorServiceImpl implements AdministratorService {

    private final Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    private AdministratorRepository administratorRepository;

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private UserPrePaidBillingRepository userPrePaidBillingRepository;

    @Autowired
    private UserBalanceRepository userBalanceRepository;

    @Autowired
    private UserBillingRepository userBillingRepository;

    @Autowired
    private UserAuthService userAuthService;

    @Override
    public Administrator createAdministrator(Administrator administrator) {
        Administrator administrator1 = new Administrator();
        if(StringUtils.isEmpty(administrator.getName())){
            throw new ClientRequestException(AccountErrorCode.NAMEEMPTY, "userName connot be empty " );
        }

        administrator1.setName(administrator.getName());
        User newUser = new User();
        newUser.setUsername(administrator.getName());
        newUser.setPassword(administrator.getPassword());
        newUser.setEnabled(true);

        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(newUser));

        administrator1.setPassword("");
        administratorRepository.save(administrator1);

        return administrator1;
    }

    @Override
    public List<Administrator> getAllAdministrator() {
        return administratorRepository.findAll();
    }

    @Override
    public Account getAccountByName(String name) {
        Account account = accountRepository.findByName(name);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);
        return account;
    }

    @Override
    public Account getAccountByPhoneNumber(String phoneNumber) {
        Account account = accountRepository.findByPhoneNumber(phoneNumber);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.PHONENOTEXIST, "Can not find account with phone number" + phoneNumber);
        return account;
    }

    @Override
    public Account getAccountByEmail(String email) {
        Account account = accountRepository.findByEmail(email);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.EMAILNOTEXIST, "Email does not exist: " + email);
        return account;
    }


    @Override
    public List<UserPrePaidBilling> getUserBillingByName(String name) {
        List<UserPrePaidBilling> userBillingList = userPrePaidBillingRepository.findByUsername(name);
        if (userBillingList == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);
        return userBillingList;
    }

    @Override
    public Account unLockLockedAccount(String name,Account account) {
        Account newAccount = accountRepository.findByName(name);
        if (newAccount == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);

        User newUser = new User();
        newUser.setUsername(name);
        userAuthService.unlock(new com.edgec.browserbackend.auth.domain.User(newUser),"unlock");

        return newAccount;
    }

    @Override
    public Account lockAbnormalAccount(String name, Account account) {
        Account abnormalAccount = accountRepository.findByName(name);
        if (abnormalAccount == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name);

        abnormalAccount.setLockReason(account.getLockReason());

        User abnormalUser = new User();
        abnormalUser.setUsername(name);
        userAuthService.lock(new com.edgec.browserbackend.auth.domain.User(abnormalUser),"lock");

        accountRepository.save(abnormalAccount);

        return abnormalAccount;
    }

    @Override
    public Page<Account> searchAllUserPage(Pageable pageable) {
        Page<Account> accounts = accountRepository.findAll(pageable);
        return accounts;
    }

    @Override
    public Page<UserPrePaidBilling> searchAllUserBillingPage(int page, int year1, int year2, int month1, int month2) {
//        Pageable pageable = new PageRequest(page, 15);
//        Page<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findAllByYearBetweenAndMonthBetween(pageable, year1, year2, month1, month2);
//        return userPrePaidBillings;
        return null;
    }

    @Override
    public Page<Account> searchCreateAccountBetween(Pageable pageable, String  strDate1, String strDate2 ) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        Page<Account> accounts = null;
        try {
            Date dateTime1 = formatter.parse(strDate1);
            Date dateTime2 = formatter.parse(strDate2);
            accounts = accountRepository.findAllBySignupDateBetween(pageable, dateTime1, dateTime2);

        }catch (ParseException e){
            e.printStackTrace();
        }
        return accounts;

    }
    @Override
    public void deleteAdministrator( String name) {
        Administrator administrator = administratorRepository.findByName(name);
        if (administrator == null) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }
        userAuthService.deleteUser(name);
        administratorRepository.delete(administrator);

    }

    @Override
    public Administrator updateAdministrator(String name, String roles) {
        Administrator updating = administratorRepository.findByName(name);
        if (updating == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "can't find adminstrator with name " + name);
        }
        updating.setRole(roles);
        userAuthService.updateRoles(name, roles);
        administratorRepository.save(updating);

        return updating;
    }

    @Override
    public List<CompanyEarningsDto> getCompanyEarningDetail(String stringdate) {
        StringTokenizer stringToInt = new StringTokenizer(stringdate, "-");
        int year = Integer.parseInt(stringToInt.nextToken());
        int month = Integer.parseInt(stringToInt.nextToken());

        List<CompanyEarningsDto> result = new ArrayList<>();

        List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByYearAndMonth(year, month);
        if (userPrePaidBillings != null && userPrePaidBillings.size() > 0) {
            for (UserPrePaidBilling userPrePaidBilling : userPrePaidBillings) {
                CompanyEarningsDto companyEarningsDto = new CompanyEarningsDto();

                companyEarningsDto.setUsername(userPrePaidBilling.getUsername());
                companyEarningsDto.setUserBillAmount(userPrePaidBilling.getOwnActualCost());
                companyEarningsDto.setTotalBillingCost(userPrePaidBilling.getTotal());
                companyEarningsDto.setPayStatus(userPrePaidBilling.getStatus());

                Account account = accountRepository.findByName(userPrePaidBilling.getUsername());

                companyEarningsDto.setCompanyShouldEarning(userPrePaidBilling.getTotal());

                if (userPrePaidBilling.getStatus().equals(BillStatus.PAID)) {
                    companyEarningsDto.setCompanyActualEarning(userPrePaidBilling.getTotal());
                } else {
                    companyEarningsDto.setCompanyActualEarning(0);
                }

                result.add(companyEarningsDto);

            }
        }

        return result;
    }

    @Override
    public List<UserPrePaidBilling> getServiceFeeOwedUserInfo() {

        List<UserPrePaidBilling> result = userPrePaidBillingRepository.findByStatus(BillStatus.UNPAID);

        return result;
    }

    @Override
    public boolean getUserLockState(String name) {
        return userAuthService.lockState(name);
    }


    @Override
    public void deleteUser(String username) {
        userAuthService.deleteUser(username);
        accountRepository.deleteById(username);
    }

    @Override
    public UserBalance findUserBalance(String username) {
        return userBalanceRepository.findById(username).orElse(null);
    }

    @Override
    public boolean userbillingTransfer() {
        List<UserBilling> userBillings = userBillingRepository.findByUsername("51tou");
        userBillings.forEach(x ->{
             generateUserPrePaidBillingByUserBilling(x);
        });
        return true;
    }

    private void generateUserPrePaidBillingByUserBilling(UserBilling userBilling) {
        UserPrePaidBilling userPrePaidBilling = new UserPrePaidBilling();
        userPrePaidBilling.setYear(userBilling.getYear());
        userPrePaidBilling.setMonth(userBilling.getMonth());
        userPrePaidBilling.setServices(Services.INTELLIGROUP);
        if (userBilling.getStatus() != BillStatus.PAID) {
            userPrePaidBilling.setStatus(BillStatus.UNPAID);
        } else {
            userPrePaidBilling.setStatus(BillStatus.PAID);
        }
        userPrePaidBilling.setDeductionRecords(userBilling.getDeductionRecords());
        userPrePaidBilling.setOwnPotentialCost(userBilling.getOwnPotentialCost());
        userPrePaidBilling.setOwnActualCost(userBilling.getOwnActualCost());
        userPrePaidBilling.setOwnBillingCost(userBilling.getOwnBillingCost());
        userPrePaidBilling.setTotal(userBilling.getTotalBillingCost());
        userPrePaidBilling.setTimestamp(ZonedDateTime.of(userBilling.getYear(), userBilling.getMonth(), 1, 0, 0, 0, 0, ZoneOffset.systemDefault()).toEpochSecond() * 1000);
        userPrePaidBilling.setPrepaid(false);
        userPrePaidBilling.setChargeType(9);
        userPrePaidBilling.setUnit("month");
        userPrePaidBilling.setUser(userBilling.getUsername());
        userPrePaidBilling.setUsername(userBilling.getUsername());
        userPrePaidBilling.setTarget(userBilling.getUsername() + "创建的云端伸缩组");
        userPrePaidBillingRepository.save(userPrePaidBilling);

    }
}









