package com.edgec.browserbackend.auth.config;

import com.edgec.browserbackend.auth.exception.AuthErrorCode;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.error.IntelligroupOauthException;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.oauth2.common.DefaultThrowableAnalyzer;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.common.exceptions.InsufficientScopeException;
import org.springframework.security.oauth2.common.exceptions.OAuth2Exception;
import org.springframework.security.oauth2.provider.error.DefaultWebResponseExceptionTranslator;
import org.springframework.security.web.util.ThrowableAnalyzer;
import org.springframework.stereotype.Component;
import org.springframework.web.HttpRequestMethodNotSupportedException;

import java.io.IOException;

@Component("oAuthResponseExceptionTranslator")
public class OAuthResponseExceptionTranslator extends DefaultWebResponseExceptionTranslator {

    private ThrowableAnalyzer throwableAnalyzer = new DefaultThrowableAnalyzer();


    private ResponseEntity<OAuth2Exception> handleOAuth2Exception(OAuth2Exception e, AuthErrorCode error) throws IOException {

        IntelligroupOauthException ex = new IntelligroupOauthException(error.getReason(), error.getReason(), error.getCode());
        int status = e.getHttpErrorCode();
        HttpHeaders headers = new HttpHeaders();
        headers.set("Cache-Control", "no-store");
        headers.set("Pragma", "no-cache");
        if (status == HttpStatus.UNAUTHORIZED.value() || (e instanceof InsufficientScopeException)) {
            headers.set("WWW-Authenticate", String.format("%s %s", OAuth2AccessToken.BEARER_TYPE, e.getSummary()));
        }

        ResponseEntity<OAuth2Exception> response = new ResponseEntity<OAuth2Exception>(ex, headers, HttpStatus.valueOf(status));
        return response;
    }

    @Override
    public ResponseEntity<OAuth2Exception> translate(Exception e) throws Exception {

        // Try to extract a SpringSecurityException from the stacktrace
        Throwable[] causeChain = throwableAnalyzer.determineCauseChain(e);

        Exception ase = (AuthenticationException) throwableAnalyzer.getFirstThrowableOfType(AuthenticationException.class,
                causeChain);
        if (ase != null) {
            AuthErrorCode code = AuthErrorCode.AUTHENTICATION_ERROR;
            return handleOAuth2Exception((OAuth2Exception) ase, code);
        }

        ase = (AccessDeniedException) throwableAnalyzer
                .getFirstThrowableOfType(AccessDeniedException.class, causeChain);
        if (ase instanceof AccessDeniedException) {
            AuthErrorCode code = AuthErrorCode.AUTHORIZATION_ERROR;
            return handleOAuth2Exception((OAuth2Exception) ase, code);
        }

        ase = (HttpRequestMethodNotSupportedException) throwableAnalyzer
                .getFirstThrowableOfType(HttpRequestMethodNotSupportedException.class, causeChain);
        if (ase instanceof HttpRequestMethodNotSupportedException) {
            AuthErrorCode code = AuthErrorCode.OTHERS;
            return handleOAuth2Exception((OAuth2Exception) ase, code);

        }

        ase = (OAuth2Exception) throwableAnalyzer.getFirstThrowableOfType(
                OAuth2Exception.class, causeChain);

        if (ase != null) {
            AuthErrorCode code = AuthErrorCode.AUTHENTICATION_ERROR;
            return handleOAuth2Exception((OAuth2Exception) ase, code);
        }

        ase = (ClientRequestException) throwableAnalyzer.getFirstThrowableOfType(
                ClientRequestException.class, causeChain);
        if (ase != null) {
            ClientRequestException clientRequestException = (ClientRequestException) ase;
            OAuth2Exception exception = OAuth2Exception.create(OAuth2Exception.UNAUTHORIZED_CLIENT, "xxxx");
            return handleOAuth2Exception(exception, (AuthErrorCode) clientRequestException.getErrorCode());
        }
        AuthErrorCode code = AuthErrorCode.INTERNALSERVERERROR;
        return handleOAuth2Exception((OAuth2Exception) ase, code);
    }

}
