package com.edgec.browserbackend.browser.repository;

import com.edgec.browserbackend.browser.domain.BindHistory;
import com.edgec.browserbackend.browser.domain.IpResource;
import com.edgec.browserbackend.browser.dto.IpResourceUnwindResultDto;
import com.mongodb.client.result.UpdateResult;
import org.bson.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.*;
import org.springframework.data.mongodb.core.query.BasicQuery;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Update;

import java.time.Instant;
import java.util.*;

import static org.springframework.data.mongodb.core.query.Criteria.where;

public class IpResourceRepositoryCustomImpl implements IpResourceRepositoryCustom {

    @Autowired
    MongoTemplate mongoTemplate;


    @Override
    public boolean lockTask(IpResource ipResource) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        Criteria criteria = new Criteria();
        criteria.orOperator(where("id").is(ipResource.getId()).and("isLocked").is(false).and("status").is(ipResource.getStatus()).and("isDeleted").is(false),
                where("lockTimestamp").lte(Instant.now().minusSeconds(300).toEpochMilli()).and("status").is(ipResource.getStatus()).and("isDeleted").is(false));
        basicQuery.addCriteria(criteria);
        Update update = new Update();
        update.set("isLocked", true).set("lockTimestamp", Instant.now().toEpochMilli());
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, IpResource.class);

        return result.getModifiedCount() >= 1;
    }

    @Override
    public boolean unLockTask(String id) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("id").is(id));
        Update update = new Update();
        update.set("isLocked", false).set("lockTimestamp", Instant.now().toEpochMilli());
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, IpResource.class);

        if (result.getModifiedCount() < 1)
            return false;
        else
            return true;
    }

    @Override
    public boolean healthLock(IpResource ipResource) {
        return true;
    }

    @Override
    public boolean unLockHealth(String id) {
        return true;
    }

    @Override
    public List<IpResource> sampleTasks(int status, long timestamp) {
        Criteria matchCriteria = new Criteria();
        matchCriteria.orOperator(where("status").is(status).and("isLocked").is(false).and("isDeleted").is(false),
                where("status").is(status).and("isLocked").is(true).and("lockTimestamp").lte(timestamp).and("isDeleted").is(false));

        MatchOperation match = Aggregation.match(matchCriteria);

        SampleOperation sample = Aggregation.sample(100);
        AggregationResults<IpResource> results = mongoTemplate.aggregate(Aggregation.newAggregation(match, sample), IpResource.class, IpResource.class);
        List<IpResource> mappedResults = results.getMappedResults();

        return mappedResults;
    }

    @Override
    public List<IpResource> sampleTasks(List<Integer> status) {
        Criteria matchCriteria = new Criteria();
        matchCriteria.orOperator(where("status").in(status).and("isLocked").is(false).and("validTime").gt(Instant.now().toEpochMilli()).and("isDeleted").is(false).and("healthLockTimestamp").lte(Instant.now().minusSeconds(60 * 30).toEpochMilli()),
                where("status").in(status).and("isLocked").is(true).and("healthLockTimestamp").lte(Instant.now().minusSeconds(600).toEpochMilli()).and("isDeleted").is(false));

        MatchOperation match = Aggregation.match(matchCriteria);

        SampleOperation sample = Aggregation.sample(20);
        AggregationResults<IpResource> results = mongoTemplate.aggregate(Aggregation.newAggregation(match, sample), IpResource.class, IpResource.class);
        List<IpResource> mappedResults = results.getMappedResults();

        return mappedResults;
    }

    @Override
    public boolean addShopId(String ipId, String shopId) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("id").is(ipId).and("isDeleted").is(false));
        Update update = new Update();
        update.push("shopIds", shopId);
        update.set("bind", true);
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, IpResource.class);

        if (result.getModifiedCount() < 1)
            return false;
        else
            return true;
    }

    @Override
    public boolean deleteShopId(String id, String shopId, BindHistory bindHistory) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("id").is(id).and("isDeleted").is(false));
        Update update = new Update();
        update.pull("shopIds", shopId).push("bindhistory", bindHistory);
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, IpResource.class);

        if (result.getModifiedCount() < 1)
            return false;
        else
            return true;
    }

    @Override
    public boolean updateStatus(String id, int status) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("id").is(id).and("isDeleted").is(false));
        Update update = new Update();
        update.set("status", status);
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, IpResource.class);

        if (result.getModifiedCount() < 1)
            return false;
        else
            return true;
    }

    @Override
    public boolean updateBind(String id, boolean isbind) {
        Document doc = new Document();
        BasicQuery basicQuery = new BasicQuery(doc);
        basicQuery.addCriteria(where("id").is(id).and("isDeleted").is(false));
        Update update = new Update();
        update.set("bind", isbind);
        UpdateResult result = mongoTemplate.updateFirst(basicQuery, update, IpResource.class);

        if (result.getModifiedCount() < 1)
            return false;
        else
            return true;
    }

    @Override
    public List<IpResource> findShopIdInList(List<String> shopIds, boolean isDeleted) {
        MatchOperation matchOperation = Aggregation.match(where("isDeleted").is(isDeleted));
        UnwindOperation unwind = Aggregation.unwind("shopIds");
        MatchOperation matchshopId = Aggregation.match(where("shopIds").in(shopIds));
        List<IpResourceUnwindResultDto> ipResourceUnwindResultDtos = mongoTemplate.aggregate(
                Aggregation.newAggregation(matchOperation, unwind, matchshopId), IpResource.class, IpResourceUnwindResultDto.class
        ).getMappedResults();
        if (ipResourceUnwindResultDtos.isEmpty()) {
            return new ArrayList<>();
        }

        HashMap<String, IpResource> ipResourceHashMap = new HashMap<>();
        ipResourceUnwindResultDtos.forEach(
                x -> {
                    if (ipResourceHashMap.containsKey(x.getId())) {
                        ipResourceHashMap.get(x.getId()).getShopIds().add(x.getShopId());
                    } else {
                        ipResourceHashMap.put(x.getId(), x.toResource());
                    }
                }
        );

        List<IpResource> result = new ArrayList<>();
        Set<Map.Entry<String, IpResource>> entry = ipResourceHashMap.entrySet();
        for (Map.Entry<String, IpResource> e : entry) {
            result.add(e.getValue());
        }
        return result;
    }

    @Override
    public List<IpResource> findShopIdInListAndStatus(List<String> shopIds, boolean isDeleted, int status) {
        MatchOperation matchOperation = Aggregation.match(where("isDeleted").is(isDeleted).and("status").is(status));
        UnwindOperation unwind = Aggregation.unwind("shopIds");
        MatchOperation matchshopId = Aggregation.match(where("shopIds").in(shopIds));
        List<IpResourceUnwindResultDto> ipResourceUnwindResultDtos = mongoTemplate.aggregate(
                Aggregation.newAggregation(matchOperation, unwind, matchshopId), IpResource.class, IpResourceUnwindResultDto.class).getMappedResults();
        if (ipResourceUnwindResultDtos.isEmpty()) {
            return new ArrayList<>();
        }
        HashMap<String, IpResource> ipResourceHashMap = new HashMap<>();
        ipResourceUnwindResultDtos.forEach(x -> {
            if (ipResourceHashMap.containsKey(x.getId())) {
                ipResourceHashMap.get(x.getId()).getShopIds().add(x.getShopId());
            } else {
                ipResourceHashMap.put(x.getId(), x.toResource());
            }
        });
        List<IpResource> result = new ArrayList<>();
        Set<Map.Entry<String, IpResource>> entry = ipResourceHashMap.entrySet();
        for (Map.Entry<String, IpResource> e : entry) {
            result.add(e.getValue());
        }
        return result;
    }

    @Override
    public List<IpResource> findShopIdInListAndRegionLike(List<String> shopIds, boolean isDeleted, String region) {
        MatchOperation matchOperation = Aggregation.match(where("isDeleted").is(isDeleted).and("regionCn").regex(".*?\\" + region + ".*"));
        UnwindOperation unwind = Aggregation.unwind("shopIds");
        MatchOperation matchshopId = Aggregation.match(where("shopIds").in(shopIds));
        List<IpResourceUnwindResultDto> ipResourceUnwindResultDtos = mongoTemplate.aggregate(
                Aggregation.newAggregation(matchOperation, unwind, matchshopId), IpResource.class, IpResourceUnwindResultDto.class).getMappedResults();
        if (ipResourceUnwindResultDtos.isEmpty()) {
            return new ArrayList<>();
        }
        HashMap<String, IpResource> ipResourceHashMap = new HashMap<>();
        ipResourceUnwindResultDtos.forEach(x -> {
            if (ipResourceHashMap.containsKey(x.getId())) {
                ipResourceHashMap.get(x.getId()).getShopIds().add(x.getShopId());
            } else {
                ipResourceHashMap.put(x.getId(), x.toResource());
            }
        });
        List<IpResource> result = new ArrayList<>();
        Set<Map.Entry<String, IpResource>> entry = ipResourceHashMap.entrySet();
        for (Map.Entry<String, IpResource> e : entry) {
            result.add(e.getValue());
        }
        return result;
    }

    @Override
    public List<IpResource> findIds() {
        BasicQuery basicQuery = new BasicQuery(new Document());
        basicQuery.fields().include("id");
        return mongoTemplate.find(basicQuery, IpResource.class);
    }


}
