package com.edgec.browserbackend.browser.controller;

import com.edgec.browserbackend.account.dto.ResultDto;
import com.edgec.browserbackend.browser.dto.IpListRequestDto;
import com.edgec.browserbackend.browser.dto.IpResourceRequestDto;
import com.edgec.browserbackend.browser.dto.IpResourceUpdateDto;
import com.edgec.browserbackend.browser.service.Impl.IpResourceServiceImpl;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.common.auth.Securitys;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.lock.MongoDistributedLock;
import com.edgec.browserbackend.common.utils.ResponseUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.security.Principal;
import java.util.HashMap;
import java.util.Map;

import static com.edgec.browserbackend.common.lock.MongoDistributedLockKey.RENEWIP;

/**
 * @author JMW
 */
@RestController
@RequestMapping("/ip")
public class IpController {

    @Autowired
    private IpResourceService ipResourceService;

    private final Logger log = LoggerFactory.getLogger(IpResourceServiceImpl.class);

    /**
     * 购买IP资源
     *
     * @param principal            principal
     * @param ipResourceRequestDto ipResourceRequestDto
     * @return ResultDto
     */
    @RequestMapping(value = "/buy", method = RequestMethod.POST)
    public ResultDto buyIp(Principal principal, @RequestBody IpResourceRequestDto ipResourceRequestDto) {
        String logs = "【buyIp】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            return ResponseUtil.success(ipResourceService.buyIp(principal.getName(), ipResourceRequestDto));
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    @RequestMapping(value = "/queryinventory", method = RequestMethod.POST)
    public ResultDto queryinventory(Principal principal, @RequestBody IpResourceRequestDto ipResourceRequestDto) {
        String logs = "【buyIp】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            return ResponseUtil.success(ipResourceService.queryInventory(principal.getName(), ipResourceRequestDto));
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * 续费IP资源
     *
     * @param principal            principal
     * @param ipResourceRequestDto ipResourceRequestDto
     * @return ResultDto
     */
    @RequestMapping(value = "/renew", method = RequestMethod.POST)
    public ResultDto renewIp(Principal principal, @RequestBody IpResourceRequestDto ipResourceRequestDto) {
        String logs = "【renewIp】 ";
        log.info("{}, params : {}", logs, principal.getName());
        boolean lock = MongoDistributedLock.getLock(RENEWIP, principal.getName(), 2000);
        if (!lock) {
            return ResponseUtil.error("");
        }
        try {
            return ResponseUtil.success(ipResourceService.renewIp(principal.getName(), ipResourceRequestDto));
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        } finally {
            MongoDistributedLock.releaseLock(RENEWIP, principal.getName());
        }
    }

    /**
     * 删除指定IP资源
     *
     * @param principal            principal
     * @param ipResourceRequestDto ipResourceRequestDto
     * @return ResultDto
     */
    @RequestMapping(value = "/del", method = RequestMethod.POST)
    public ResultDto deleteIp(Principal principal, @RequestBody IpResourceRequestDto ipResourceRequestDto) {
        String logs = "【deleteIp】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            return ResponseUtil.success(ipResourceService.deleteIp(principal.getName(), ipResourceRequestDto));
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * 获取IP列表
     *
     * @param principal        principal
     * @param ipListRequestDto ipListRequestDto
     * @return ResultDto
     */
    @RequestMapping(value = "/list", method = RequestMethod.POST)
    public ResultDto getIpList(Principal principal, @RequestBody IpListRequestDto ipListRequestDto) {
        String logs = "【getIpList】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            return ResponseUtil.success(
                    ipResourceService.getIpList(principal.getName(),
                            ipListRequestDto.getGroupType(),
                            ipListRequestDto.getPage(),
                            ipListRequestDto.getAmount(),
                            ipListRequestDto.getFilter()));
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * （暂不知）
     *
     * @param principal           principal
     * @param ipResourceUpdateDto ipResourceUpdateDto
     * @return ResultDto
     */
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResultDto updateIp(Principal principal, @RequestBody IpResourceUpdateDto ipResourceUpdateDto) {
        String logs = "【updateIp】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ipResourceService.updateIp(principal.getName(), ipResourceUpdateDto);
            return ResponseUtil.success();

        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * 添加自有ip的检测
     *
     * @param principal           principal
     * @param ipResourceUpdateDto ipResourceUpdateDto
     * @return ResultDto
     */
    @RequestMapping(value = "/query", method = RequestMethod.POST)
    public ResultDto queryIp(Principal principal, @RequestBody IpResourceUpdateDto ipResourceUpdateDto) {
        String logs = "【queryIp】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            Map<String, Boolean> map = new HashMap<>(16);
            map.put("found", ipResourceService.queryIpExist(principal.getName(), ipResourceUpdateDto));
            return ResponseUtil.success(map);
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * @param principal            principal
     * @param ipResourceRequestDto ipResourceRequestDto
     * @return ResultDto
     */
    @RequestMapping(value = "/detail", method = RequestMethod.POST)
    public ResultDto queryIpDetail(Principal principal, @RequestBody IpResourceRequestDto ipResourceRequestDto) {
        String logs = "【queryIpDetail】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            return ResponseUtil.success(ipResourceService.queryIp(principal.getName(), ipResourceRequestDto));
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * @param principal principal
     * @return ResultDto
     */
    @RequestMapping(value = "/special/set", method = RequestMethod.PUT)
    public ResultDto setSpecialLine(Principal principal) {
        String logs = "【setSpecialLine】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ipResourceService.setSpecialLine();
            return ResponseUtil.success();
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * @param principal principal
     * @return ResultDto
     */
    @RequestMapping(value = "/releasedeleted", method = RequestMethod.PUT)
    public ResultDto deleteUseless(Principal principal) {
        String logs = "【deleteUseless】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ipResourceService.releaseDeletedIp();
            return ResponseUtil.success();
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }

    /**
     * @param principal principal
     * @return ResultDto
     */
    @RequestMapping(value = "/transferbindshops", method = RequestMethod.PUT)
    @PreAuthorize(Securitys.ADMIN_EL)
    public ResultDto transferBindShops(Principal principal) {
        String logs = "【transferBindShops】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ipResourceService.transferBindShops();
            return ResponseUtil.success();
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
            return ResponseUtil.error(e.getErrorCode());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            return ResponseUtil.error(e.getMessage());
        }
    }
}
