package com.edgec.browserbackend.account.service.impl;

import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.dto.*;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.*;
import com.edgec.browserbackend.account.service.AdministratorService;
import com.edgec.browserbackend.auth.service.UserService;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.repository.IpCountRecordRepository;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.browser.repository.ProxyConfigRepository;
import com.edgec.browserbackend.browser.repository.ShopRepository;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.utils.DateConverter;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.io.File;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import static com.edgec.browserbackend.browser.domain.IpType.VENDOR;


@Service
@ComponentScan("com.edgec.browserbackend.account.repository")
public class AdministratorServiceImpl implements AdministratorService {

    private final Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    private AdministratorRepository administratorRepository;

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private UserPrePaidBillingRepository userPrePaidBillingRepository;

    @Autowired
    private UserBalanceRepository userBalanceRepository;

    @Autowired
    private UserBillingRepository userBillingRepository;

    @Autowired
    private UserService userService;

    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Autowired
    private WhiteSiteRepository whiteSiteRepository;

    @Autowired
    private ProxyConfigRepository proxyConfigRepository;

    @Autowired
    private IpCountRecordRepository ipCountRecordRepository;

    @Autowired
    private ShopRepository shopRepository;

    @Autowired
    private NoticeRepository noticeRepository;

    @Override
    public Administrator createAdministrator(Administrator administrator) {
        Administrator administrator1 = new Administrator();
        if (StringUtils.isEmpty(administrator.getName())) {
            throw new ClientRequestException(AccountErrorCode.NAMEEMPTY, "userName connot be empty ");
        }

        administrator1.setName(administrator.getName());
        User newUser = new User();
        newUser.setUsername(administrator.getName());
        newUser.setPassword(administrator.getPassword());
        newUser.setEnabled(true);

        userService.create(new com.edgec.browserbackend.auth.domain.User(newUser));

        administrator1.setPassword("");
        administratorRepository.save(administrator1);

        return administrator1;
    }

    @Override
    public List<Administrator> getAllAdministrator() {
        return administratorRepository.findAll();
    }

    @Override
    public Account getAccountByName(String name) {
        Account account = accountRepository.findByName(name).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "Username does not exist: " + name));
        return account;
    }

    @Override
    public Account getAccountByPhoneNumber(String phoneNumber) {
        Account account = accountRepository.findByPhoneNumberAndParentIsNull(phoneNumber);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.PHONENOTEXIST, "Can not find account with phone number" + phoneNumber);
        return account;
    }

    @Override
    public Account getAccountByEmail(String email) {
        Account account = accountRepository.findByEmail(email);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.EMAILNOTEXIST, "Email does not exist: " + email);
        return account;
    }


    @Override
    public BillQueryResultDto getUserBillingByName(String name) {
        Account account = accountRepository.findByName(name).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByUsernameAndPayMethodIn(name, Arrays.asList(1, 2));
        double parentexpense = 0;
        if (userPrePaidBillings != null)
            parentexpense = userPrePaidBillings.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
        double parentwithdrawn = account.getPromotion().getWithdrawn();
        UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
        double parentbalanced = 0;
        double parentbalanceused = 0;
        if (userBalance != null) {
            parentbalanced = userBalance.getBalanced();
            parentbalanceused = userBalance.getUsed();
        }
        List<UserPrePaidBilling> userPrePaidBillings1 = userPrePaidBillingRepository.findByUsernameAndPayMethod(name, 3);
        double parentbanktransfer = 0;
        if (userPrePaidBillings1 != null)
            parentbanktransfer = userPrePaidBillings1.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();

        double childexpense = 0;
        double childwithdrawn = 0;
        double childbalanced = 0;
        double childbanktransfer = 0;
        double childbalanceused = 0;
        if (account.getParent() == null) {
            List<Account> children = accountRepository.findByParent(account.getName());
            if (children != null && children.size() > 0) {
                for (Account child : children) {
                    List<UserPrePaidBilling> userPrePaidBillings_child = userPrePaidBillingRepository.findByUsernameAndPayMethodIn(child.getName(), Arrays.asList(1, 2));
                    if (userPrePaidBillings_child != null)
                        childexpense += userPrePaidBillings_child.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                    UserBalance userBalance_child = userBalanceRepository.findById(child.getName()).orElse(null);
                    if (userBalance_child != null) {
                        childbalanced += userBalance_child.getBalanced();
                        childbalanceused += userBalance_child.getUsed();
                    }
                    List<UserPrePaidBilling> userPrePaidBillings1_child = userPrePaidBillingRepository.findByUsernameAndPayMethod(child.getName(), 3);
                    if (userPrePaidBillings1_child != null)
                        childbanktransfer += userPrePaidBillings1_child.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                }
            }
        }

        BillQueryResultDto billQueryResultDto = new BillQueryResultDto();
        billQueryResultDto.setChildExpense(childexpense);
        billQueryResultDto.setChildBankTransfer(childbanktransfer);
        billQueryResultDto.setChildBalanced(childbalanced);
        billQueryResultDto.setChildWithdraw(childwithdrawn);
        billQueryResultDto.setChildBalanceUsed(childbalanceused);

        billQueryResultDto.setParentExpense(parentexpense);
        billQueryResultDto.setParentBalanced(parentbalanced);
        billQueryResultDto.setParentBankTransfer(parentbanktransfer);
        billQueryResultDto.setParentWithdrawn(parentwithdrawn);
        billQueryResultDto.setParentBalanceUsed(parentbalanceused);

        Account promoter = null;
        if (account.getPromotionCode() != null)
            promoter = accountRepository.findByPromotion(account.getPromotionCode());
        if (promoter != null)
            billQueryResultDto.setPromoter(promoter.getName());

        billQueryResultDto.setPromotion(account.getPromotion());

        return billQueryResultDto;
    }

    @Override
    public Account unLockLockedAccount(String name, Account account) {
        Account newAccount = accountRepository.findByName(name).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));


        User newUser = new User();
        newUser.setUsername(name);
        userService.unlock(new com.edgec.browserbackend.auth.domain.User(newUser), "unlock");

        return newAccount;
    }

    @Override
    public Account lockAbnormalAccount(String name, Account account) {
        Account abnormalAccount = accountRepository.findByName(name).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));


        abnormalAccount.setLockReason(account.getLockReason());

        User abnormalUser = new User();
        abnormalUser.setUsername(name);
        userService.lock(new com.edgec.browserbackend.auth.domain.User(abnormalUser), "lock");

        accountRepository.save(abnormalAccount);

        return abnormalAccount;
    }

    @Override
    public Page<Account> searchAllUserPage(Pageable pageable) {
        Page<Account> accounts = accountRepository.findAll(pageable);
        return accounts;
    }

    @Override
    public Page<UserPrePaidBilling> searchAllUserBillingPage(int page, int year1, int year2, int month1, int month2) {
        Pageable pageable = PageRequest.of(page, 15);
        Page<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findAllByYearBetweenAndMonthBetween(pageable, year1, year2, month1, month2);
        return userPrePaidBillings;
    }

    @Override
    public Page<Account> searchCreateAccountBetween(Pageable pageable, String strDate1, String strDate2, int isAuthorized) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        Page<Account> accounts = null;
        try {
            Date dateTime1 = formatter.parse(strDate1);
            Date dateTime2 = formatter.parse(strDate2);
            if (isAuthorized == 5)
                accounts = accountRepository.findAllBySignupDateBetweenAndParentIsNull(pageable, dateTime1, dateTime2);
            else
                accounts = accountRepository.findAllBySignupDateBetweenAndParentIsNullAndAuthorized(pageable, dateTime1, dateTime2, isAuthorized);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return accounts;

    }

    @Override
    public void deleteAdministrator(String name) {
        Administrator administrator = administratorRepository.findByName(name);
        if (administrator == null) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }
        userService.deleteUser(name);
        administratorRepository.delete(administrator);

    }

    @Override
    public Administrator updateAdministrator(String name, String roles) {
        Administrator updating = administratorRepository.findByName(name);
        if (updating == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "can't find adminstrator with name " + name);
        }
        updating.setRole(roles);
        userService.updateRoles(name, roles);
        administratorRepository.save(updating);

        return updating;
    }

    @Override
    public List<CompanyEarningsDto> getCompanyEarningDetail(String stringdate) {
        StringTokenizer stringToInt = new StringTokenizer(stringdate, "-");
        int year = Integer.parseInt(stringToInt.nextToken());
        int month = Integer.parseInt(stringToInt.nextToken());

        List<CompanyEarningsDto> result = new ArrayList<>();

        List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByYearAndMonth(year, month);
        if (userPrePaidBillings != null && userPrePaidBillings.size() > 0) {
            for (UserPrePaidBilling userPrePaidBilling : userPrePaidBillings) {
                CompanyEarningsDto companyEarningsDto = new CompanyEarningsDto();

                companyEarningsDto.setUsername(userPrePaidBilling.getUsername());
                companyEarningsDto.setTotalBillingCost(userPrePaidBilling.getTotal());
                companyEarningsDto.setPayStatus(userPrePaidBilling.getStatus());

                Account account = accountRepository.findByName(userPrePaidBilling.getUsername())
                        .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

                companyEarningsDto.setCompanyShouldEarning(userPrePaidBilling.getTotal());

                if (userPrePaidBilling.getStatus().equals(BillStatus.PAID)) {
                    companyEarningsDto.setCompanyActualEarning(userPrePaidBilling.getTotal());
                } else {
                    companyEarningsDto.setCompanyActualEarning(0);
                }

                result.add(companyEarningsDto);

            }
        }

        return result;
    }

    @Override
    public List<UserPrePaidBilling> getServiceFeeOwedUserInfo() {

        List<UserPrePaidBilling> result = userPrePaidBillingRepository.findByStatus(BillStatus.UNPAID);

        return result;
    }

    @Override
    public boolean getUserLockState(String name) {
        return userService.lockState(name);
    }


    @Override
    public void deleteUser(String username) {
        userService.deleteUser(username);
        accountRepository.deleteById(username);
    }

    @Override
    public UserBalance findUserBalance(String username) {
        return userBalanceRepository.findById(username).orElse(null);
    }

    @Override
    public boolean userbillingTransfer() {
        List<UserBilling> userBillings = userBillingRepository.findByUsername("51tou");
        userBillings.forEach(x -> {
            generateUserPrePaidBillingByUserBilling(x);
        });
        return true;
    }

    private void generateUserPrePaidBillingByUserBilling(UserBilling userBilling) {
        UserPrePaidBilling userPrePaidBilling = new UserPrePaidBilling();
        userPrePaidBilling.setYear(userBilling.getYear());
        userPrePaidBilling.setMonth(userBilling.getMonth());
        userPrePaidBilling.setServices(Services.INTELLIGROUP);
        if (userBilling.getStatus() != BillStatus.PAID) {
            userPrePaidBilling.setStatus(BillStatus.UNPAID);
        } else {
            userPrePaidBilling.setStatus(BillStatus.PAID);
        }
        userPrePaidBilling.setDeductionRecords(userBilling.getDeductionRecords());
        userPrePaidBilling.setTotal(userBilling.getTotalBillingCost());
        userPrePaidBilling.setTimestamp(ZonedDateTime.of(userBilling.getYear(), userBilling.getMonth(), 1, 0, 0, 0, 0, ZoneOffset.systemDefault()).toEpochSecond() * 1000);
        userPrePaidBilling.setPrepaid(false);
        userPrePaidBilling.setChargeType(9);
        userPrePaidBilling.setUnit("month");
        userPrePaidBilling.setUsername(userBilling.getUsername());
        userPrePaidBillingRepository.save(userPrePaidBilling);

    }

    @Override
    public void addPromotionCode(String username, String promotionCode) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        account.setPromotionCode(promotionCode);
        accountRepository.save(account);
    }

    @Override
    public IpCountQueryResultDto queyrIpCount(String username) {
        IpCountQueryResultDto ipCountQueryResultDto = new IpCountQueryResultDto();
        if (StringUtils.isBlank(username)) {
            long ipcount_using = ipResourceRepository.countAllByIsDeletedAndValidTimeGreaterThan(false, Instant.now().toEpochMilli());
            long ipcount_all = ipResourceRepository.count();
            ipCountQueryResultDto.setIpCount_using(ipcount_using);
            ipCountQueryResultDto.setIpCount_all(ipcount_all);
            IpCountRecord ipCountRecord = ipCountRecordRepository.findFirstByUsernameAndTimestampBetween("all",
                    Instant.now().minus(1, ChronoUnit.DAYS).toEpochMilli(),
                    Instant.now().toEpochMilli());
            if (ipCountRecord != null) {
                ipCountQueryResultDto.setIpCount_yesterday_all(ipCountRecord.getIp_all());
                ipCountQueryResultDto.setIpCount_yesterday_using(ipCountRecord.getIp_using());
            }
        } else {
            long ipcount_using = ipResourceRepository.countAllByOwnerAndIsDeletedAndValidTimeGreaterThan(username, false, Instant.now().toEpochMilli());
            long ipcount_all = ipResourceRepository.countAllByOwner(username);
            ipCountQueryResultDto.setIpCount_using(ipcount_using);
            ipCountQueryResultDto.setIpCount_all(ipcount_all);
            IpCountRecord ipCountRecord = ipCountRecordRepository.findFirstByUsernameAndTimestampBetween(username,
                    Instant.now().minus(1, ChronoUnit.DAYS).toEpochMilli(),
                    Instant.now().toEpochMilli());
            if (ipCountRecord != null) {
                ipCountQueryResultDto.setIpCount_yesterday_all(ipCountRecord.getIp_all());
                ipCountQueryResultDto.setIpCount_yesterday_using(ipCountRecord.getIp_using());
            }
        }

        return ipCountQueryResultDto;
    }

    @Override
    public PromotionQueryResultDto queryPromotion(Pageable pageable, String username, String promotionCode, String strDate1, String strDate2) {
        if (StringUtils.isBlank(username) && StringUtils.isBlank(promotionCode)) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        Account account = null;
        if (!StringUtils.isBlank(username)) {
            account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        } else {
            account = accountRepository.findByPromotion(promotionCode);
        }
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (account.getParent() != null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }


        PromotionQueryResultDto promotionQueryResultDto = new PromotionQueryResultDto(account);
        promotionQueryResultDto.setUsername(username);
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
            Date dateTime1 = formatter.parse(strDate1);
            Date dateTime2 = formatter.parse(strDate2);


            Page<Account> accountPage = accountRepository.findAllBySignupDateBetweenAndPromotionCodeAndParentIsNull(pageable, dateTime1, dateTime2, account.getPromotion().getCode());
            List<AccountPromotionDto> accountPromotionDtos = new ArrayList<>();
            if (accountPage.getNumberOfElements() > 0) {
                accountPage.get().forEach(account1 -> {
                    accountPromotionDtos.add(new AccountPromotionDto(account1));
                });
            }
            Page<AccountPromotionDto> accountPromotionDtoPage = new PageImpl<AccountPromotionDto>(accountPromotionDtos);
            promotionQueryResultDto.setUsers(accountPromotionDtoPage);

            // 查询在指定的日期通过邀请码注册的账户
            // List<Account> accounts = accountRepository.findAllBySignupDateBetweenAndPromotionCodeAndParentIsNull(dateTime1, dateTime2, account.getPromotion().getCode());
            // 从上面的  改为查询 某用户邀请的所有账户
            List<Account> accounts = accountRepository.findAllByPromotionCodeAndParentIsNull(account.getPromotion().getCode());

            Promotion promotion = new Promotion();
            Account finalAccount = account;
            AtomicLong ipCount = new AtomicLong();
            if (accounts != null && accounts.size() > 0) {
                accounts.forEach(x -> {
                    promotion.setInvitedUsers(promotion.getInvitedUsers() + 1);
                    // 查询账户在指定日期充值记录
                    List<UserPrePaidBilling> userPrePaidBillings = userPrePaidBillingRepository.findByAdministratorAndPayMethodInAndTimestampBetween(x.getName(), Arrays.asList(1, 2, 3), dateTime1.getTime(), dateTime2.getTime());
                    double totalCommission = 0;
                    if (userPrePaidBillings != null && userPrePaidBillings.size() > 0) {
                        totalCommission = userPrePaidBillings.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                    }

                    promotion.setTotalCommission(promotion.getTotalCommission() + (int) totalCommission);
                    promotion.setCommission(promotion.getCommission() + x.getPromotion().getCommission());

                    ipCount.addAndGet(ipResourceRepository.countAllByOwnerAndIsDeletedAndValidTimeGreaterThan(x.getName(), false, Instant.now().toEpochMilli()));
                    List<Account> children = accountRepository.findByParent(x.getName());
                    if (children != null && children.size() > 0) {
                        ipCount.addAndGet(ipResourceRepository.countAllByOwnerInAndIsDeletedAndValidTimeGreaterThan(children.stream().map(Account::getName).collect(Collectors.toList()), false, Instant.now().toEpochMilli()));
                    }

                    double secondCommission = 0;
                    if (finalAccount.getPromotion().isSale() && x.getParent() == null) {
                        List<Account> secondPromotes = accountRepository.findByPromotionCodeAndParentIsNull(x.getPromotion().getCode());
                        if (secondPromotes != null && secondPromotes.size() > 0) {
                            for (Account secondPromote : secondPromotes) {
                                if (finalAccount.getPromotion().isSale()) {
                                    ipCount.addAndGet(ipResourceRepository.countAllByOwnerAndIsDeletedAndValidTimeGreaterThan(secondPromote.getName(), false, Instant.now().toEpochMilli()));
                                    List<Account> secondChildren = accountRepository.findByParent(secondPromote.getName());
                                    if (children != null && children.size() > 0) {
                                        ipCount.addAndGet(ipResourceRepository.countAllByOwnerInAndIsDeletedAndValidTimeGreaterThan(secondChildren.stream().map(Account::getName).collect(Collectors.toList()), false, Instant.now().toEpochMilli()));
                                    }
                                }
                                List<UserPrePaidBilling> userPrePaidBillings1 = userPrePaidBillingRepository.findByAdministratorAndPayMethodInAndTimestampBetween(secondPromote.getName(), Arrays.asList(1, 2, 3), dateTime1.getTime(), dateTime2.getTime());
                                if (userPrePaidBillings1 != null && userPrePaidBillings1.size() > 0)
                                    secondCommission += userPrePaidBillings1.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
                            }
                        }
                    }
                    if (finalAccount.getPromotion().isSale()) {
                        promotion.setAllGift(promotion.getAllGift() + totalCommission * 0.1 + secondCommission * 0.02);
                    } else {
                        promotion.setAllGift(promotion.getAllGift() + totalCommission * 0.08);
                    }
                    promotion.setWithdrawn(promotion.getWithdrawn() + secondCommission);
                    promotion.setGift(promotion.getGift() + x.getPromotion().getGift());
                    promotion.setCommissionLastMonth(promotion.getCommissionLastMonth() + x.getPromotion().getCommissionLastMonth());
                });
            }
            promotionQueryResultDto.setPromotion(promotion);
            promotionQueryResultDto.setIpCount(ipCount.get());
            return promotionQueryResultDto;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }


    @Override
    public PromotionInfoDto countPromotionInfos(String username, String promotionCode, String beginDate, String endDate) {
        long start = System.currentTimeMillis();
        // 1. 根据传入的 username 或者 推荐码 获取用户信息
        Account account = getAccount(username, promotionCode);

        // 2. 获取 该推荐码下所有的用户
        List<Account> accounts = accountRepository.findAllByPromotionCodeAndParentIsNull(account.getPromotion().getCode());

        // promotion 用来存储所有用户的消费信息
        PromotionInfoDto promotionInfoDto = new PromotionInfoDto(0, 0, 0, 0, 0.0, 0, 0);
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
            Date beginTime = formatter.parse(beginDate);
            Date endTime = formatter.parse(endDate);

            if (accounts != null && accounts.size() > 0) {
                // 邀请的一级用户数
                promotionInfoDto.setFirstLevelInvitedUsers(accounts.size());

                int[] arr1 = getCommissionAndIps(beginTime, endTime, accounts.stream().map(Account::getName).collect(Collectors.toList()));
                promotionInfoDto.setFirstLevelCommission(arr1[0]);
                promotionInfoDto.setFirstLevelIps(arr1[1]);

                List<String> codeList = accounts.stream().filter(x -> x.getParent() == null).map(x -> x.getPromotion().getCode()).collect(Collectors.toList());
                if (account.getPromotion().isSale()) {
                    List<Account> secondPromotes = accountRepository.findByPromotionCodeInAndParentIsNull(codeList);
                    if (secondPromotes != null && secondPromotes.size() > 0) {
                        int[] arr2 = getCommissionAndIps(beginTime, endTime, secondPromotes.stream().map(Account::getName).collect(Collectors.toList()));
                        promotionInfoDto.setSecondLevelCommission(arr2[0]);
                        promotionInfoDto.setSecondLevelIps(arr2[1]);
                    }
                }
            }

            // 设置总ip
            promotionInfoDto.setAllLevelIps(promotionInfoDto.getFirstLevelIps() + promotionInfoDto.getSecondLevelIps());
            // 设置应当提现的礼物金额
            if (account.getPromotion().isSale()) {
                promotionInfoDto.setGift(promotionInfoDto.getFirstLevelCommission() * 0.1 + promotionInfoDto.getSecondLevelCommission() * 0.02);
            } else {
                promotionInfoDto.setGift(promotionInfoDto.getFirstLevelCommission() * 0.08);
            }

            log.info("===>countPromotionInfos consume time {}", System.currentTimeMillis() - start);
            return promotionInfoDto;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public Page<AccountPromotionDto> pagePromotionInfos(Pageable pageable, String username, String promotionCode, String beginDate, String endDate) {
        // 1. 根据传入的 username 或者 推荐码 获取用户信息
        Account account = getAccount(username, promotionCode);

        //PromotionQueryResultDto promotionQueryResultDto = new PromotionQueryResultDto(account);
        try {
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
            Date beginTime = formatter.parse(beginDate);
            Date endTime = formatter.parse(endDate);

            Page<Account> accountPage = accountRepository.findAllByPromotionCodeAndParentIsNull(pageable, account.getPromotion().getCode());
            List<AccountPromotionDto> list = new ArrayList<>();
            if (accountPage.getNumberOfElements() > 0) {
                accountPage.get().forEach(x -> {
                    AccountPromotionDto accountPromotionDto = new AccountPromotionDto(x);
                    accountPromotionDto.setFirstLevelCommission(0);
                    accountPromotionDto.setFirstLevelIps(0);
                    accountPromotionDto.setSecondLevelCommission(0);
                    accountPromotionDto.setSecondLevelIps(0);
                    int[] arr1 = getCommissionAndIps(beginTime, endTime, x);
                    accountPromotionDto.setFirstLevelCommission(arr1[0]);
                    accountPromotionDto.setFirstLevelIps(arr1[1]);
                    if (account.getPromotion().isSale() && x.getParent() == null) {
                        // 获取当前账户的下级账户
                        List<Account> secondPromotes = accountRepository.findByPromotionCodeAndParentIsNull(x.getPromotion().getCode());
                        if (secondPromotes != null && secondPromotes.size() > 0) {
                            for (Account secondPromote : secondPromotes) {
                                // 将下级账户及下级账户的子账户中未过期的ip统计到 ipCount中
                                int[] arr2 = getCommissionAndIps(beginTime, endTime, secondPromote);
                                accountPromotionDto.setSecondLevelCommission(accountPromotionDto.getSecondLevelCommission() + arr2[0]);
                                accountPromotionDto.setSecondLevelIps(accountPromotionDto.getSecondLevelIps() + arr2[1]);
                            }
                        }
                    }
                    list.add(accountPromotionDto);
                });
            }
            Page<AccountPromotionDto> page = new PageImpl<AccountPromotionDto>(list, pageable, accountPage.getTotalElements());
            return page;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @NotNull
    private Account getAccount(String username, String promotionCode) {
        if (StringUtils.isBlank(username) && StringUtils.isBlank(promotionCode)) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        Account account = null;
        if (!StringUtils.isBlank(username)) {
            account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        } else {
            account = accountRepository.findByPromotion(promotionCode);
        }
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (account.getParent() != null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        return account;
    }

    private int[] getCommissionAndIps(Date beginTime, Date endTime, Account currentAccount) {
        Integer ownIps = (int) ipResourceRepository.countAllByOwnerAndIsDeletedAndValidTimeGreaterThan(currentAccount.getName(), false, Instant.now().toEpochMilli());
        List<Account> children = accountRepository.findByParent(currentAccount.getName());
        if (children != null && children.size() > 0) {
            ownIps += (int) ipResourceRepository.countAllByOwnerInAndIsDeletedAndValidTimeGreaterThan(children.stream().map(Account::getName).collect(Collectors.toList()), false, Instant.now().toEpochMilli());
        }

        // 计算出账户的所有消费
        double commission = 0;
        List<UserPrePaidBilling> userPrePaidBillings1 = userPrePaidBillingRepository.findByAdministratorAndPayMethodInAndTimestampBetween(currentAccount.getName(), Arrays.asList(1, 2, 3), beginTime.getTime(), endTime.getTime());
        if (userPrePaidBillings1 != null && userPrePaidBillings1.size() > 0) {
            commission = userPrePaidBillings1.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
        }

        int[] arr = {(int) commission, ownIps};
        return arr;
    }

    // 计算当前用户的购买的 所有ip 与 消费
    private int[] getCommissionAndIps(Date beginTime, Date endTime, List<String> nameList) {
        Integer ownIps = (int) ipResourceRepository.countAllByOwnerInAndIsDeletedAndValidTimeGreaterThan(nameList, false, Instant.now().toEpochMilli());
        List<Account> children = accountRepository.findByParentIn(nameList);
        if (children != null && children.size() > 0) {
            ownIps += (int) ipResourceRepository.countAllByOwnerInAndIsDeletedAndValidTimeGreaterThan(children.stream().map(Account::getName).collect(Collectors.toList()), false, Instant.now().toEpochMilli());
        }

        // 计算出账户的所有消费
        double commission = 0;
        List<UserPrePaidBilling> userPrePaidBillings1 = userPrePaidBillingRepository.findByAdministratorInAndPayMethodInAndTimestampBetween(nameList, Arrays.asList(1, 2, 3), beginTime.getTime(), endTime.getTime());
        if (userPrePaidBillings1 != null && userPrePaidBillings1.size() > 0) {
            commission = userPrePaidBillings1.stream().mapToDouble(UserPrePaidBilling::getTotal).sum();
        }

        int[] arr = {(int) commission, ownIps};
        return arr;
    }

    @Override
    public void addWhiteList(String website) {
        WhiteSite whiteSite = new WhiteSite();
        whiteSite.setWhite(website);
        whiteSiteRepository.save(whiteSite);
    }

    @Override
    public void addUserWhiteList(String username, String website) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        account.getWhiteList().add(website);
        accountRepository.save(account);
    }

    @Override
    public File getProxyConfig() {
        ProxyConfig proxyConfig = null;
        if (proxyConfigRepository.count() > 0)
            proxyConfig = proxyConfigRepository.findAll().get(0);
        if (proxyConfig == null)
            return null;
        File file = proxyConfigRepository.getFile(proxyConfig.getConfigFileId(), ".cfg");
        if (file != null)
            return file;
        else
            return null;
    }

    @Override
    public Page<UserUsedDto> getAllUserUsed(Pageable pageable) {
        Page<UserBalance> userBalances = userBalanceRepository.findAllOrderByUsedDesc(pageable);
        List<UserBalance> userBalanceList = userBalances.toList();
        List<UserUsedDto> userUsedDtos = new ArrayList<>();
        if (userBalanceList != null && userBalanceList.size() > 0) {
            userBalanceList.forEach(x -> {
                UserUsedDto userUsedDto = new UserUsedDto();
                Account account = accountRepository.findByName(x.getUsername())
                        .orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

                userUsedDto.setPromotionCode(account.getPromotionCode());
                userUsedDto.setUserBalance(x);
                userUsedDtos.add(userUsedDto);
            });
        }
        Page<UserUsedDto> userUsedDtopage = new PageImpl<>(userUsedDtos, pageable, userBalances.getTotalElements());
        return userUsedDtopage;
    }

    @Override
    public Long convertToDistributor(String id, boolean tag) {
        return accountRepository.convertToDistributor(id, tag);
    }

    @Override
    public List<String> getUsernameIpList(String owner) {
        return ipResourceRepository.findByOwnerAndIsDeletedAndIpType(owner, false, VENDOR).stream().map(IpResource::getAddr).collect(Collectors.toList());
    }

    @Override
    public List<String> ipMigration(String owner, String newOwner, String ips) {
        accountRepository.findByName(newOwner).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        Arrays.asList(ips.split(";")).forEach(ip -> unbindAndMigration(owner, newOwner, ip));
        return getUsernameIpList(newOwner);
    }

    private void unbindAndMigration(String owner, String newOwner, String ip) {
        String logs = "【unbindAndMigration】 ";
        try {
            IpResource ipResource = ipResourceRepository.findByAddrAndIsDeleted(ip, false);
            List<String> shopIds = ipResource.getShopIds();
            String id = ipResource.getId();
            shopIds.forEach(shopId -> unbindAndSetHistory(ip, id, shopId));
            ipResourceRepository.updateBind(id, false);
            ipResourceRepository.ipMigration(owner, newOwner, ip);
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
    }

    private void unbindAndSetHistory(String ip, String id, String shopId) {
        String logs = "【unbindAndSetHistory】 ";
        try {
            Shop shop = shopRepository.findById(shopId).orElseThrow(() -> new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST));
            BindHistory bindHistory = new BindHistory();
            bindHistory.setIp(ip);
            bindHistory.setPlatform(shop.getShopPlatform());
            bindHistory.setShopName(shop.getShopName());
            bindHistory.setUnbindTime(ZonedDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            ipResourceRepository.deleteShopId(id, shopId, bindHistory);
        } catch (ClientRequestException e) {
            log.warn("{}, ClientRequestException : {}", logs, e.getErrorCode().getReason());
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
    }

    @Override
    public void setNotice(Notice notice) {
        if (notice == null
                || StringUtils.isEmpty(notice.getStarDate())
                || StringUtils.isEmpty(notice.getEndDate())
                || StringUtils.isEmpty(notice.getContent())
                || DateConverter.stringToDate(notice.getStarDate()).
                compareTo(DateConverter.stringToDate(notice.getEndDate())) > 0) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        Notice only = noticeRepository.findOneByNameIsNotNull();
        if (only != null) {
            only.setStarDate(notice.getStarDate());
            only.setEndDate(notice.getEndDate());
            only.setContent(notice.getContent());
            noticeRepository.save(only);
        } else {
            noticeRepository.save(notice);
        }
    }

    @Override
    public NoticeDto getNotice() {
        Notice notice = noticeRepository.findOneByNameIsNotNull();
        if (notice != null) {
            return new NoticeDto(notice.getStarDate(), notice.getEndDate(), notice.getContent());
        }
        return null;
    }

    @Override
    public void delNotice() {
        noticeRepository.deleteAll();
    }
}