package com.edgec.browserbackend.browser.service.Impl;

import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.BindHistory;
import com.edgec.browserbackend.browser.domain.IpResource;
import com.edgec.browserbackend.browser.domain.Shop;
import com.edgec.browserbackend.browser.domain.UserShop;
import com.edgec.browserbackend.browser.dto.ShopRequestDto;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.browser.repository.ShopRepository;
import com.edgec.browserbackend.browser.repository.UserShopRepository;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

@Service
public class IpAndShopServiceImpl implements IpAndShopService {

    private final Logger logger = LoggerFactory.getLogger(IpAndShopServiceImpl.class);

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private UserShopRepository userShopRepository;

    @Autowired
    private ShopRepository shopRepository;

    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Override
    public void bindShop(String username, ShopRequestDto shopRequestDto) {
        String shopId = shopRequestDto.getShopId();
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        IpResource ipResource = null;
        if (StringUtils.isNotBlank(shopRequestDto.getIpAddr())) {
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(shopRequestDto.getIpAddr(), false);
        } else if (StringUtils.isNotBlank(shopRequestDto.getIpId())) {
            ipResource = ipResourceRepository.findByIdAndIsDeleted(shopRequestDto.getIpId(), false);
        }

        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);

        IpResource bind = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shopId, false);
        if (bind != null) {
            return;
        }
        try {
            ipResourceRepository.addShopId(ipResource.getId(), shopId);
        } catch (Exception e) {
            logger.error("fail to bind shop and ip", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void unBindShop(String username, ShopRequestDto shopRequestDto) {
        String shopId = null;
        if (StringUtils.isNotBlank(shopRequestDto.getShopId()))
            shopId = shopRequestDto.getShopId();
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        IpResource ipResource = null;
        if (StringUtils.isNotBlank(shopRequestDto.getIpAddr())) {
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(shopRequestDto.getIpAddr(), false);
        } else if (StringUtils.isNotBlank(shopRequestDto.getIpId())) {
            ipResource = ipResourceRepository.findByIdAndIsDeleted(shopRequestDto.getIpId(), false);
        }
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        if (ipResource.getShopIds() == null && !ipResource.getShopIds().contains(shop.getShopId()))
            throw new ClientRequestException(BrowserErrorCode.IPNOTBINDTOSHOP);
        try {
            BindHistory bindHistory = new BindHistory();
            bindHistory.setIp(ipResource.getAddr());
            bindHistory.setPlatform(shop.getShopPlatform());
            bindHistory.setShopName(shop.getShopName());
            bindHistory.setUnbindTime(ZonedDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
            ipResourceRepository.deleteShopId(ipResource.getId(), shopId, bindHistory);
            IpResource newIp = ipResourceRepository.findById(ipResource.getId()).orElse(null);
            if (newIp.getShopIds() == null || newIp.getShopIds().size() == 0)
                ipResourceRepository.updateBind(ipResource.getId(), false);
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void unBindShops(String username, ShopRequestDto shopRequestDto) {
        List<String> shopIds = null;
        if (shopRequestDto.getShopIds() != null && shopRequestDto.getShopIds().size() > 0)
            shopIds = shopRequestDto.getShopIds();
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        IpResource ipResource = null;
        if (StringUtils.isNotBlank(shopRequestDto.getIpAddr())) {
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(shopRequestDto.getIpAddr(), false);
        } else if (StringUtils.isNotBlank(shopRequestDto.getIpId())) {
            ipResource = ipResourceRepository.findByIdAndIsDeleted(shopRequestDto.getIpId(), false);
        }
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        for (String shopId : shopIds) {
            UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
            if (account.getPermission() < 4 || userShop == null) {
                continue;
            }
            Shop shop = shopRepository.findById(shopId).orElse(null);
            if (shop == null)
                continue;
            if (ipResource.getShopIds() == null && !ipResource.getShopIds().contains(shop.getShopId()))
                continue;
            try {
                BindHistory bindHistory = new BindHistory();
                bindHistory.setIp(ipResource.getAddr());
                bindHistory.setPlatform(shop.getShopPlatform());
                bindHistory.setShopName(shop.getShopName());
                bindHistory.setUnbindTime(ZonedDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss")));
                ipResourceRepository.deleteShopId(ipResource.getId(), shopId, bindHistory);
                IpResource newIp = ipResourceRepository.findById(ipResource.getId()).orElse(null);
                if (newIp.getShopIds() == null || newIp.getShopIds().size() == 0)
                    ipResourceRepository.updateBind(ipResource.getId(), false);
            } catch (Exception e) {
                logger.error("fail to unbind", e.getMessage());
                throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
            }
        }
    }
}
