package com.edgec.browserbackend.browser.service.Impl;

import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.dto.*;
import com.edgec.browserbackend.browser.repository.*;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.utils.FileUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

@Service
@Transactional
public class ShopServiceImpl implements ShopService {

    private final Logger logger = LoggerFactory.getLogger(ShopServiceImpl.class);

    @Autowired
    ShopRepository shopRepository;

    @Autowired
    AccountRepository accountRepository;

    @Autowired
    UserShopRepository userShopRepository;

    @Autowired
    GroupRepository groupRepository;

    @Autowired
    IpResourceRepository ipResourceRepository;

    @Autowired
    IpAndShopService ipAndShopService;

    @Autowired
    IpResourceService ipResourceService;

    @Autowired
    SpecialLineRepository specialLineRepository;

    @Override
    public String addShop(String username, ShopResultDto shopResultDto) {
        Account account = accountRepository.findByName(username);
        String id = null;
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        if (account.getShopCount() >= 10000) {
            throw new ClientRequestException(AccountErrorCode.SHOPMAX);
        }
        UserShop us = null;
        if (shopResultDto.getShopId() != null)
            us = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
        if (shopResultDto.getGroup() == null)
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        Group group = groupRepository.findById(shopResultDto.getGroup()).orElse(null);
        if (group == null) {
            throw new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST);
        }
        try {
            shopResultDto.setOwner(username);
            Shop shop = new Shop();
            shop.of(shopResultDto);
            shop.setCreateTime(Instant.now().toEpochMilli());
            id = shopRepository.save(shop).getShopId();
            UserShop userShop = new UserShop();
            userShop.setUsername(username);
            userShop.setShopId(id);
            if (shopResultDto.getGroup() != null && us == null) {
                userShop.setGroupId(shopResultDto.getGroup());
            }
            userShopRepository.save(userShop);
            if (StringUtils.isNotBlank(account.getParent())) {
                UserShop userShop1 = new UserShop();
                userShop1.setUsername(account.getParent());
                userShop1.setShopId(id);
                userShop1.setGroupId("-1");
                userShopRepository.save(userShop1);
            }
            //可以优化
            account.setShopCount(account.getShopCount() + 1);
            accountRepository.save(account);
        }catch (Exception e) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        return id;
    }

    @Override
    public List<String> addShops(String username, MultipartFile file) throws IOException {
        List<String> ids = new ArrayList<>();
        List<List<Object>> list = FileUtil.readExcel(file.getInputStream());
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        if (account.getShopCount() >= 10000) {
            throw new ClientRequestException(AccountErrorCode.SHOPMAX);
        }
        for (List<Object> l : list) {
            ShopResultDto shopResultDto = new ShopResultDto();
            shopResultDto.setOwner(username);
            shopResultDto.setShopName(l.get(0).toString());
            shopResultDto.setShopPlatform(l.get(1).toString());
            shopResultDto.setGroup("-1");
            if (StringUtils.isNotBlank(l.get(2).toString()))
                shopResultDto.setShopAccount(l.get(2).toString());
            if (StringUtils.isNotBlank(l.get(3).toString()))
                shopResultDto.setShopPassword(l.get(3).toString());
            String id = null;
            UserShop us = null;
            if (shopResultDto.getShopId() != null)
                us = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
            if (shopResultDto.getGroup() == null)
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            Group group = groupRepository.findById(shopResultDto.getGroup()).orElse(null);
            if (group == null) {
                throw new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST);
            }
            try {
                shopResultDto.setOwner(username);
                Shop shop = new Shop();
                shop.of(shopResultDto);
                shop.setCreateTime(Instant.now().toEpochMilli());
                id = shopRepository.save(shop).getShopId();
                UserShop userShop = new UserShop();
                userShop.setUsername(username);
                userShop.setShopId(id);
                if (shopResultDto.getGroup() != null && us == null) {
                    userShop.setGroupId(shopResultDto.getGroup());
                }
                userShopRepository.save(userShop);
                if (StringUtils.isNotBlank(account.getParent())) {
                    UserShop userShop1 = new UserShop();
                    userShop1.setUsername(account.getParent());
                    userShop1.setShopId(id);
                    userShop1.setGroupId("-1");
                    userShopRepository.save(userShop1);
                }
                //可以优化
                account.setShopCount(account.getShopCount() + 1);
                accountRepository.save(account);
                ids.add(id);
            }catch (Exception e) {
                logger.error("fail to add shops", e.getMessage());
                throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
            }
        }
        return ids;
    }

    @Override
    public String updateShop(String username, ShopResultDto shopResultDto) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (shopResultDto == null || StringUtils.isBlank(shopResultDto.getShopId())) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop_old = shopRepository.findById(shopResultDto.getShopId()).orElseGet(null);
        if (shop_old == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        try {
            shop_old = shop_old.of(shopResultDto);
            shopRepository.save(shop_old);
        }catch (Exception e) {
            logger.error("fail to update", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        return shop_old.getShopId();
    }

    @Override
    public void deleteShop(String username, String shopId) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (shopId == null) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shop.getShopId(), false);
        if (ipResource != null) {
            ShopRequestDto shopRequestDto = new ShopRequestDto();
            shopRequestDto.setIpId(ipResource.getId());
            shopRequestDto.setShopId(shopId);
            ipAndShopService.unBindShop(username, shopRequestDto);
        }
        boolean result = userShopRepository.deleteByShopId(shopId);
        if (result) {
            shopRepository.deleteById(shopId);
            account.setShopCount(account.getShopCount() - 1);
            accountRepository.save(account);
        } else throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
    }

    @Override
    public void transferShop(String username, String shopId, String groupId) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        Group group = groupRepository.findById(groupId).orElse(null);
        if (userShop == null || (!group.getId().equals("-1") && group.getOwner() != null && !group.getOwner().equals(username))) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        try {
            userShop.setGroupId(groupId);
            userShopRepository.save(userShop);
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }

    }

    @Override
    public void assignShops(String username, List<String> shopIds, List<String> users) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        List<UserShop> userShops = userShopRepository.findByUsernameAndShopIdIn(username, shopIds);
        if (account.getPermission() < 8 || userShops == null || userShops.size() < 1) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Pageable pageable = PageRequest.of(0, 100);
        List<Shop> shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(shopIds, pageable).getContent();
        if (shops == null || shops.size() < 1)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        List<Account> accounts = accountRepository.findByNameIn(users);
        if (accounts == null || accounts.size() != users.size())
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        for (Account ac : accounts) {
            if (ac.getParent() == null || !ac.getParent().equals(username))
            {
                throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
            }
        }
        if (shops.size() == 1) {
            try {
                userShopRepository.deleteByShopIdExceptOwner(shops.get(0).getShopId(), shops.get(0).getOwner());
                for (Account account1 : accounts) {
                    UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shops.get(0).getShopId());
                    if (userShop1 != null)
                        continue;
                    userShop1 = new UserShop(account1.getName(), shops.get(0).getShopId(), "-1");
                    userShopRepository.save(userShop1);
                }
            } catch (Exception e) {
                logger.error("fail to assign", e.getMessage());
                throw  new ClientRequestException(BrowserErrorCode.UNKNOWN);
            }
        } else {
            shops.stream().forEach(shop -> {
                try {
                    for (Account account1 : accounts) {
                        UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shop.getShopId());
                        if (userShop1 != null)
                            continue;
                        userShop1 = new UserShop(account1.getName(), shop.getShopId(), "-1");
                        userShopRepository.save(userShop1);
                    }
                } catch (Exception e) {
                    logger.error("fail to assign", e.getMessage());
                    throw  new ClientRequestException(BrowserErrorCode.UNKNOWN);
                }
            });
        }
    }

    @Override
    public ShopResultDto queryShop(String username, String shopId) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null) {
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        }
        IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(shopId, false);
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        String group = userShopRepository.findByUsernameAndShopId(username, shop.getShopId()).getGroupId();
        ShopResultDto shopResultDto = null;
        if (ipResource.isSpecialLine()) {
            SpecialLine specialLine = specialLineRepository.findAll().get(0);
            shopResultDto = ShopResultDto.of(shop, group, new IpResourceDto(ipResource, null, false, specialLine));
        }
        else
            shopResultDto = ShopResultDto.of(shop, group, new IpResourceDto(ipResource, null, false));
        return shopResultDto;
    }

    @Override
    public ShopPageResultDto getShopList(String username, String groupId, int page, int amount, ShopFilterDto shopFilterDto) {
        if (amount > 100)
            amount = 100;
        Pageable pageable = PageRequest.of(page, amount);
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        Group group = null;
        if (groupId != null)
            group = groupRepository.findById(groupId).orElse(null);
        if (group == null)
            throw new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST);
        if (group != null && group.getOwner() != null && !group.getOwner().equals(username)) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        List<String> shopIds = new ArrayList<>();
        if (groupId.equals("-1")) {
            if (shopFilterDto.getBindIp() == 0)
                shopIds = userShopRepository.findByUsername(username).stream().
                        map(x -> x.getShopId()).collect(Collectors.toList());
            else if (shopFilterDto.getBindIp() == 1) {
                List<String> allIds = userShopRepository.findByUsername(username).stream()
                        .map(x -> x.getShopId()).collect(Collectors.toList());
                shopIds = ipResourceRepository.findShopIdInList(allIds, false)
                        .stream().flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
            }
            else {
                List<String> allIds = userShopRepository.findByUsername(username).stream()
                        .map(x -> x.getShopId()).collect(Collectors.toList());
                for (String id:allIds) {
                    IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, false);
                    if (ipResource == null) {
                        shopIds.add(id);
                    }
                }
            }
        } else {
            if (shopFilterDto.getBindIp() == 0)
                shopIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream().
                        map(x -> x.getShopId()).collect(Collectors.toList());
            else if (shopFilterDto.getBindIp() == 1) {
                List<String> allIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream().
                        map(x -> x.getShopId()).collect(Collectors.toList());
                shopIds = ipResourceRepository.findShopIdInList(allIds, false)
                        .stream().flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
            }
            else {
                List<String> allIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream()
                        .map(x -> x.getShopId()).collect(Collectors.toList());
                for (String id:allIds) {
                    IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, false);
                    if (ipResource == null)
                        shopIds.add(id);
                }
            }
        }
        Page<Shop> shops;
        if (shopFilterDto != null && StringUtils.isNotBlank(shopFilterDto.getIpRegion())) {
            List<String> filter = ipResourceRepository.findShopIdInListAndRegionLike(shopIds, false, shopFilterDto.getIpRegion())
                    .stream().flatMap((x -> x.getShopIds().stream())).collect(Collectors.toList());
            shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(filter, pageable);
        }
        else if (shopFilterDto != null && StringUtils.isNotBlank(shopFilterDto.getShopAccount()))
            shops = shopRepository.findByShopIdInAndShopAccountLikeOrderByCreateTimeDesc(shopIds, shopFilterDto.getShopAccount(), pageable);
        else if (shopFilterDto != null && StringUtils.isNotBlank(shopFilterDto.getShopName()))
            shops = shopRepository.findByShopIdInAndShopNameLikeOrderByCreateTimeDesc(shopIds, shopFilterDto.getShopName(), pageable);
        else
            shops = shopRepository.findByShopIdInOrderByCreateTimeDesc(shopIds, pageable);
        if (shops == null || shops.getNumberOfElements() < 1)
            return new ShopPageResultDto();
        List<ShopResultDto> shopResultDtos = new ArrayList<>();
        shops.getContent().stream().forEach(x -> {
            IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(x.getShopId(), false);
                    if (ipResource != null) {
                        if (StringUtils.isNotBlank(ipResource.getAddr())) {
                            if (ipResource.getValidTime() <= Instant.now().plusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getValidTime() > Instant.now().toEpochMilli()) {
                                if (ipResource.getStatus() != 5 && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                            ipResource.setStatus(2);
                            ipResourceRepository.save(ipResource);
                                }
                            } else if (ipResource.getValidTime() <= Instant.now().minusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                                if (ipResource.getIpType() == IpType.VENDOR) {
                                    IpResourceRequestDto ipResourceRequestDto1 = new IpResourceRequestDto();
                                    ipResourceRequestDto1.setAddr(Arrays.asList(ipResource.getAddr()));
                                    try {
                                        ipResourceService.deleteIp(username, ipResourceRequestDto1);
                                    } catch (Exception e) {
                                        logger.error(e.getMessage(), e);
                                    }
                                } else {
                                    IpResourceRequestDto ipResourceRequestDto1 = new IpResourceRequestDto();
                                    ipResourceRequestDto1.setIpId(Arrays.asList(ipResource.getId()));
                                    try {
                                        ipResourceService.deleteIp(username, ipResourceRequestDto1);
                                    } catch (Exception e) {
                                        logger.error(e.getMessage(), e);
                                    }
                                }
                                return;
                            } else if (ipResource.getValidTime() <= Instant.now().toEpochMilli() && ipResource.getStatus() != 3 && ipResource.getStatus() != 6) {
                        ipResource.setStatus(1);
                        ipResourceRepository.save(ipResource);
                            } else {
                                if ((ipResource.getStatus() == 0 || ipResource.getStatus() == 1 || ipResource.getStatus() == 2) && ipResource.getVendor() != Vendor.local) {
                            ipResource.setStatus(0);
                            ipResourceRepository.save(ipResource);
                                } else if (ipResource.getIpType().equals(IpType.LOCAL) && (ipResource.getStatus() == 1 || ipResource.getStatus() == 2) && ipResource.getAddr().equals("本地Ip未使用")) {
                            ipResource.setStatus(4);
                            ipResourceRepository.save(ipResource);
                                }
                            }
                        }
                    }
                    if (ipResource == null)
                        ipResource = new IpResource();
            String group1 = userShopRepository.findByUsernameAndShopId(username, x.getShopId()).getGroupId();
            ShopResultDto shopResultDto = null;
                    if (ipResource.isSpecialLine()) {
                        SpecialLine specialLine = specialLineRepository.findAll().get(0);
                shopResultDto = ShopResultDto.of(x, group1, new IpResourceDto(ipResource, null, false, specialLine));
                    }
                    else
                shopResultDto = ShopResultDto.of(x, group1, new IpResourceDto(ipResource, null, false));
            shopResultDtos.add(shopResultDto);
        });
        Page<ShopResultDto> shopDtoPage = new PageImpl<>(shopResultDtos, pageable, shopIds.size());
        ShopPageResultDto shopPageResultDto = new ShopPageResultDto();
        shopPageResultDto.setShopList(shopDtoPage.getContent());
        PageInfo pageInfo = new PageInfo();
        pageInfo.setCurrentPage(shopDtoPage.getPageable().getPageNumber());
        pageInfo.setTotalPages(shopDtoPage.getTotalPages());
        pageInfo.setTotalItems(shopIds.size());
        shopPageResultDto.setShopPage(pageInfo);
        return shopPageResultDto;
    }

    @Override
    public ShopSummary getShopSummary(String username) {
        Account account = accountRepository.findByName(username);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        ShopSummary shopSummary = new ShopSummary();
        List<String> allShopIds = userShopRepository.findByUsername(username).stream().map(x -> x.getShopId()).collect(Collectors.toList());
        List<String> unbind = new ArrayList<>();
        if (allShopIds != null && allShopIds.size() > 0) {
            for (String id:allShopIds) {
                IpResource ipResource = ipResourceRepository.findFirstByShopIdsIsAndIsDeleted(id, false);
                if (ipResource == null) {
                    unbind.add(id);
                }
            }
        }
        shopSummary.setUnbind(unbind.size());
        List<String> shopIds = userShopRepository.findByUsername(username).stream().map(x -> x.getShopId()).collect(Collectors.toList());
        List<String> bind = ipResourceRepository.findShopIdInList(shopIds, false).stream().map(x -> x.getId()).collect(Collectors.toList());
        int expired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(1, bind, false);
        int willexpired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(2, bind, false);
        shopSummary.setExpired(expired);
        shopSummary.setWillExpire(willexpired);
        shopSummary.setTotal(shopIds.size());
        return shopSummary;
    }

    @Override
    public List<String> getShopUsers(String username, String shopId) {
        Account account = accountRepository.findByName(username);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getParent() != null)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        List<String> shopUsers = userShopRepository.findByShopId(shopId).stream().map(x -> x.getUsername()).filter(x -> !x.equals(username)).collect(Collectors.toList());
        return shopUsers;
    }

    @Override
    public List<String> getBatchShopUsers(String username, List<String> shopIds) {
        Account account = accountRepository.findByName(username);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        if (account.getParent() != null)
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        List<String> shopUsers = new ArrayList<>();
        if (shopIds != null && shopIds.size() > 0) {
            String maxShopId = null;
            int max = 0;
            for (String shopId:shopIds) {
                int userCount = userShopRepository.countByShopId(shopId);
                if (userCount > max) {
                    max = userCount;
                    maxShopId = shopId;
                }
            }
            if (maxShopId == null)
                return new ArrayList<>();
            List<String> users = userShopRepository.findByShopId(maxShopId).stream().map(x -> x.getUsername()).filter(x -> !x.equals(username)).collect(Collectors.toList());
            for (String user : users) {
                int shopCount = userShopRepository.countByUsernameAndShopIdIn(user, shopIds);
                if (shopCount < shopIds.size())
                    continue;
                else
                    shopUsers.add(user);
            }
        }
        return shopUsers;
    }
}
