package com.edgec.browserbackend.browser.controller;

import com.alibaba.fastjson.JSONObject;
import com.edgec.browserbackend.account.dto.ResultDto;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.dto.ShopPageResultDto;
import com.edgec.browserbackend.browser.dto.ShopRequestDto;
import com.edgec.browserbackend.browser.dto.ShopResultDto;
import com.edgec.browserbackend.browser.dto.ShopStringResultDto;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.security.Principal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/shop")
public class ShopController {

    private final Logger log = LoggerFactory.getLogger(ShopController.class);

    @Autowired
    private ShopService shopService;

    @Autowired
    private IpAndShopService ipAndShopService;


    @PostMapping("/add")
    public ResultDto addShop(Principal principal, @RequestBody ShopResultDto shopResultDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【addShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ShopStringResultDto shopStringResultDto = new ShopStringResultDto();
            shopStringResultDto.setId(shopService.addShop(principal.getName(), shopResultDto));
            resultDto.setData(shopStringResultDto);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @PostMapping("/multiadd")
    public ResultDto addShops(Principal principal, @RequestParam("file") MultipartFile file) {
        ResultDto resultDto = new ResultDto();
        String logs = "【addShops】 ";
        log.info("{}, params : {}", logs, principal.getName());

        String name = file.getOriginalFilename();
        if (name.length() < 6 || !name.substring(name.length() - 5).equals(".xlsx")) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", BrowserErrorCode.INFORMATIONNOTCOMPELETE.getCode());
            statusInfo.put("message", "文件格式错误");
            resultDto.setStatusInfo(statusInfo);
        }
        try {
            shopService.addShops(principal.getName(), file);
            resultDto.setStatus(0);
        } catch (ClientRequestException | IOException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", 40102);
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResultDto updateShop(Principal principal, @RequestBody ShopResultDto shopResultDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【updateShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ShopStringResultDto shopStringResultDto = new ShopStringResultDto();
            shopStringResultDto.setId(shopService.updateShop(principal.getName(), shopResultDto));
            resultDto.setData(shopStringResultDto);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            shopResultDto.setShopCookie("");
            log.error("{} step-2, Exception : {},{},{}", logs, principal.getName(), JSONObject.toJSONString(shopResultDto), e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    public ResultDto deleteShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【deleteShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            shopService.deleteShop(principal.getName(), shopRequestDto.getShopId());
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(value = "/bind", method = RequestMethod.POST)
    public ResultDto bindShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【bindShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ipAndShopService.bindShop(principal.getName(), shopRequestDto);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            log.error("fail to bind shop", e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(value = "/unbind", method = RequestMethod.POST)
    public ResultDto unBindShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【unBindShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ipAndShopService.unBindShop(principal.getName(), shopRequestDto);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 移动店铺 到 分组
     */
    @RequestMapping(value = "/transfer", method = RequestMethod.POST)
    public ResultDto transferShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【transfer】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            shopService.transferShop(principal.getName(), shopRequestDto.getShopId(), shopRequestDto.getToGroup());
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 店铺分配
     */
    @RequestMapping(value = "/assign", method = RequestMethod.POST)
    public ResultDto assignShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【assignShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            shopService.assignShops(principal.getName(), shopRequestDto.getShopIds(), shopRequestDto.getToUsers());
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 获取带有 ip 资源信息的 shop
     */
    @RequestMapping(value = "/list", method = RequestMethod.POST)
    public ResultDto getShopList(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        long start = System.currentTimeMillis();
        ResultDto resultDto = new ResultDto();
        String logs = "【getShopList】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ShopPageResultDto shopDtos = shopService.getShopList(principal.getName(), shopRequestDto.getGroup(),
                    shopRequestDto.getPage(), shopRequestDto.getAmount(), shopRequestDto.getFilter(), null);
            resultDto.setData(shopDtos);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            //log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
        return resultDto;
    }

    /**
     * 针对鹏海的需求
     * 当 groupId 传 -1 的时候，查询出来的不应该是所有的店铺，而是所有未分配的店铺
     */
    @RequestMapping(value = "/penghai/list", method = RequestMethod.POST)
    public ResultDto getPenghaiShopList(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        long start = System.currentTimeMillis();
        ResultDto resultDto = new ResultDto();
        String logs = "【getPenghaiShopList】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {

            ShopPageResultDto shopDtos = shopService.getShopList(principal.getName(), shopRequestDto.getGroup(),
                    shopRequestDto.getPage(), shopRequestDto.getAmount(), shopRequestDto.getFilter(), "penghai");
            resultDto.setData(shopDtos);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            //log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
        return resultDto;
    }

    /**
     * 获取当前登录用户已经绑定了当前商铺的 所有子用户名
     */
    @RequestMapping(value = "/subusers", method = RequestMethod.POST)
    public ResultDto getShopSubUsers(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【getShopSubUsers】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            List<String> subUsers = shopService.getShopUsers(principal.getName(), shopRequestDto.getShopId());
            resultDto.setData(subUsers);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    @RequestMapping(value = "/batch/subusers", method = RequestMethod.POST)
    public ResultDto getBatchShopSubUsers(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【getBatchShopSubUsers】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            List<String> subUsers = shopService.getBatchShopUsers(principal.getName(), shopRequestDto.getShopIds());
            resultDto.setData(subUsers);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 查询店铺信息
     */
    @RequestMapping(value = "/query", method = RequestMethod.POST)
    public ResultDto queryShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        String logs = "【queryShop】 ";
        log.info("{}, params : {}", logs, principal.getName());
        try {
            ShopResultDto shopDto = shopService.queryShop(principal.getName(), shopRequestDto.getShopId());
            resultDto.setData(shopDto);
            resultDto.setStatus(0);
        } catch (ClientRequestException e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
            dealClientRequestException(resultDto, e);
        }
        return resultDto;
    }

    /**
     * 恢复部分缺失数据，仅仅调用一次就好了
     */
    @GetMapping(value = "/deal_dirtydata")
    public ResultDto dealDirtydata() {
        ResultDto resultDto = new ResultDto();
        String logs = "【dealDirtydata】 ";
        log.info("{}, params :", logs);
        try {
            Integer num = shopService.dealDirtyData();
            resultDto.setStatus(0);
            resultDto.setData("修复脏数据" + num + "条");
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
        return resultDto;
    }


    /**
     * 获取cookie
     */
    @GetMapping("/cookie/{shopId}")
    public ResultDto getShopCookie(@PathVariable String shopId) {
        ResultDto resultDto = new ResultDto();
        String logs = "【getShopCookie】 ";
        log.info("{}, params : {}", logs, shopId);
        try {
            String cookie = shopService.queryShopCookieById(shopId);
            resultDto.setStatus(0);
            resultDto.setData(cookie);
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
        return resultDto;
    }

    /**
     * 删除cookie
     */
    @DeleteMapping("/cookie/{shopId}")
    public ResultDto delShopCookie(@PathVariable String shopId) {
        ResultDto resultDto = new ResultDto();
        String logs = "【getVpsList】 ";
        log.info("{}, params : {}", logs, shopId);
        try {
            Integer result = shopService.deleteShopCookieById(shopId);
            resultDto.setStatus(result);
        } catch (Exception e) {
            log.error("{}, Exception : {}", logs, e.getMessage(), e);
        }
        return resultDto;

    }

    private void dealClientRequestException(ResultDto resultDto, ClientRequestException e) {
        resultDto.setStatus(-1);
        Map<String, Object> statusInfo = new HashMap<>();
        statusInfo.put("code", e.getErrorCode());
        statusInfo.put("message", e.getMessage());
        resultDto.setStatusInfo(statusInfo);
    }
}
