package com.edgec.browserbackend.browser.domain;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

@Getter
@Setter
@NoArgsConstructor
@Document(collection = "ipresource")
@JsonIgnoreProperties(ignoreUnknown = true)
public class IpResource implements Serializable {

    @Id
    private String id;

    /**
     * ip 地址
     */
    private String addr;

    /**
     * ip 类型，默认是 ip代理 模式
     */
    private IpType ipType = IpType.VENDOR;

    /**
     * ip 的具体代理使用
     */
    private Vendor vendor;
    private String vendorCn;

    /**
     * ip 地域
     */
    private String region;
    private String regionCn;

    /**
     * ip 的状态
     * 0:正常, 1:已过期, 2:即将过期, 3:正在分配, 4:未使用, 5:已失效, 6:未分配, 7:未缴费, 8:分配成功
     */
    private int status;

    /**
     * ip 端口号
     */
    private List<String> port;

    /**
     * ip的购买时间点
     */
    private long purchasedTime;

    /**
     * ip的过期的时间点
     */
    private long validTime;

    private String details;

    /**
     * ip的绑定历史
     */
    private List<BindHistory> bindHistory = new ArrayList<>();

    /**
     * ip的状态
     * 当 ip 资源过期的时候，系统不会直接删除 ip 资源，而是将 isDeleted 设置为 true
     * 过期7天后再删除这个ip资源
     */
    private boolean isDeleted;

    /**
     * 如果 使用的 ip 资源 是在我们系统申请的，则默认为 fangguanlianbrowser
     */
    private String username;

    /**
     * ip 资源的购买者（拥有者）
     */
    private String owner;

    private String userParent;

    /**
     * ip访问协议
     */
    private List<String> protocol;
    private String password;
    private double price;

    /**
     * 是否使用专线
     */
    private boolean specialLine;

    /**
     * 是否正在使用专线
     */
    private boolean usingSpecialLine;

    /**
     * 专线的代理用户名
     */
    private String proxyUsername;

    /**
     * 专线的代理密码
     */
    private String proxyPassword;

    /**
     * ip 资源绑定的 商铺号
     */
    private String shopId;
    private List<String> shopIds = new ArrayList<>();

    /**
     * 绑定状态
     */
    private boolean bind;

    private boolean isLocked;
    private long lockTimestamp;
    private long healthLockTimestamp;

    /**
     * 购买方式（按月、按周）
     */
    private String unit;

    /**
     * 购买数量
     * 例如 购买了 2 个月 的 ip资源，则
     * period：2
     * unit：month
     */
    private int period;

    /**
     * KCP端口号
     */
    private String secondaryProxyPort = "";

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof IpResource)) {
            return false;
        }
        IpResource that = (IpResource) o;
        return Objects.equals(getId(), that.getId());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getId());
    }

    @Override
    public String toString() {
        return "IpResource{" +
                "id='" + id + '\'' +
                ", addr='" + addr + '\'' +
                ", ipType=" + ipType +
                ", vendor=" + vendor +
                ", vendorCn='" + vendorCn + '\'' +
                ", region='" + region + '\'' +
                ", regionCn='" + regionCn + '\'' +
                ", status=" + status +
                ", port=" + port +
                ", purchasedTime=" + purchasedTime +
                ", validTime=" + validTime +
                ", details='" + details + '\'' +
                ", bindHistory=" + bindHistory +
                ", isDeleted=" + isDeleted +
                ", username='" + username + '\'' +
                ", owner='" + owner + '\'' +
                ", userParent='" + userParent + '\'' +
                ", protocol=" + protocol +
                ", password='" + password + '\'' +
                ", price=" + price +
                ", specialLine=" + specialLine +
                ", usingSpecialLine=" + usingSpecialLine +
                ", proxyUsername='" + proxyUsername + '\'' +
                ", proxyPassword='" + proxyPassword + '\'' +
                ", shopId='" + shopId + '\'' +
                ", shopIds=" + shopIds +
                ", bind=" + bind +
                ", isLocked=" + isLocked +
                ", lockTimestamp=" + lockTimestamp +
                ", healthLockTimestamp=" + healthLockTimestamp +
                ", unit='" + unit + '\'' +
                ", period=" + period +
                ", secondaryProxyPort='" + secondaryProxyPort + '\'' +
                '}';
    }
}



