package com.edgec.browserbackend.account.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.edgec.browserbackend.account.dto.*;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.account.service.PaymentService;
import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.*;
import com.edgec.browserbackend.account.service.EmailService;
import com.edgec.browserbackend.account.utils.AccountServicePool;
import com.edgec.browserbackend.auth.exception.AuthErrorCode;
import com.edgec.browserbackend.auth.repository.UserRepository;
import com.edgec.browserbackend.auth.service.UserAuthService;
import com.edgec.browserbackend.browser.domain.IpSummary;
import com.edgec.browserbackend.browser.domain.ShopSummary;
import com.edgec.browserbackend.browser.dto.PageInfo;
import com.edgec.browserbackend.browser.dto.ShopPageResultDto;
import com.edgec.browserbackend.browser.dto.ShopResultDto;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.utils.CommonStringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.thymeleaf.util.StringUtils;

import java.time.Instant;
import java.time.YearMonth;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Service
@Transactional
@ComponentScan("com.edgec.browserbackend.account.repository")
public class AccountServiceImpl implements AccountService {

    private static final int TRIVAL_MONTHS = 1;

    private final Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    private OtpRepository otpRepository;

    @Autowired
    private UserBalanceRepository userBalanceRepository;

    @Autowired
    private PreOrderRepository preOrderRepository;

    @Autowired
    private AccountRepository repository;

    @Autowired
    private EmailService emailService;

    @Autowired
    private UserBillingRepository billingRepository;

    @Autowired
    private UserRateRepository rateRepository;

    @Autowired
    private UserPrePaidBillingRepository prePaidBillingRepository;

    @Autowired
    private UserPaymentRepository userPaymentRepository;

    @Autowired
    private InvoiceRepository invoiceRepository;

    @Autowired
    private UserAuthService userAuthService;

    @Autowired
    private UserRepository userRepository;

    @Autowired
    private ShopService shopService;

    @Autowired
    private IpResourceService ipResourceService;

    @Override
    public List<UserBillList> getUserBills0(String name) {

        List<UserBillList> result = new ArrayList<>();

        List<UserBilling> byUsernameOrderByYearDescMonthDesc = billingRepository.findByUsernameOrderByYearDescMonthDesc(name);

        if (byUsernameOrderByYearDescMonthDesc != null && byUsernameOrderByYearDescMonthDesc.size() > 0) {
            for (UserBilling userBilling : byUsernameOrderByYearDescMonthDesc) {

                UserBillList list = new UserBillList();
                list.setStatus(userBilling.getStatus());
                list.setPeriod(userBilling.getYear() + "-" + (userBilling.getMonth() < 10 ? "0" + userBilling.getMonth() : userBilling.getMonth()));
                list.addUserBill(userBilling);
                result.add(list);

                List<Account> children = repository.findByParent(name);
                if (children != null && children.size() > 0) {
                    List<String> collect = children.stream().map(x -> x.getName()).collect(Collectors.toList());
                    List<UserBilling> childrenBills = billingRepository.findByUsernameInAndYearAndMonthOrderByYearDescMonthDesc(collect, userBilling.getYear(), userBilling.getMonth());

                    if (childrenBills != null && childrenBills.size() > 0)
                        childrenBills.forEach(x -> list.addUserBill(x));
                }
            }
        }
        return result;
    }

    @Override
    public List<UserBillList> getUserBills(String name, Services service) {
        List<UserBillList> result = new ArrayList<>();
        List<UserPrePaidBilling> userPrePaidBillings = prePaidBillingRepository.findBillStatisticsByUsernameAndService(name, service);
        if (userPrePaidBillings != null && userPrePaidBillings.size() > 0) {
            for (UserPrePaidBilling userBilling : userPrePaidBillings) {
                UserBillList list = new UserBillList();
                list.setStatus(userBilling.getStatus());
                list.setPeriod(userBilling.getYear() + "-" + (userBilling.getMonth() < 10 ? "0" + userBilling.getMonth() : userBilling.getMonth()));
                list.setBill(userBilling);
                result.add(list);
            }
        }
        return result;
    }

    @Override
    public IpChargeResultDto chargeByMoney(String name, double money, IpChargeRequestDto requestDto) {
        IpChargeResultDto charge = new IpChargeResultDto();
        CompletableFuture.runAsync(() -> {
            charge.setApprovedAmount(0);
            charge.setSuccess(true);

            Account account = repository.findByName(name);
            if (account != null) {
                UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
                if (userBalance == null) {
                    userBalance = new UserBalance();
                    userBalance.setBalanced(0);
                    userBalance.setUsed(0);
                    userBalance.setUsername(name);
                    userBalanceRepository.save(userBalance);
                }

                userBalanceRepository.incrementBalance(userBalance, -(float) money, (float) money);
                userBalance = userBalanceRepository.findById(name).orElse(null);

                charge.setBalance(Math.round(userBalance.getBalanced()));
                charge.setSuccess(true);

                UserPrePaidBilling bill = new UserPrePaidBilling();
                bill.setChargeType(requestDto.getChargeType());
                bill.setAmount(requestDto.getAmount());
                bill.setUnit(requestDto.getUnit());
                bill.setPeriod(requestDto.getPeriod());
                bill.setPayMethod(requestDto.getPayMethod());
                bill.setUsername(name);
                bill.setTotal((float) money);
                bill.setStatus(BillStatus.PAID);
                bill.setPrepaid(true);
                bill.setTimestamp(Instant.now().toEpochMilli());


                final YearMonth lastmonth = YearMonth.now();

                int monthValue = lastmonth.getMonthValue();
                int year = lastmonth.getYear();
                bill.setYear(year);
                bill.setMonth(monthValue);

                prePaidBillingRepository.save(bill);
            }

        }, AccountServicePool.taskPool).join();

        return charge;
    }


    private float findCvmRate(String name, String region, int usedChargeType) {
        CvmChargeRegion cvmChargeRegion = JSONObject.parseObject(region, CvmChargeRegion.class);
        float defaulRate = cvmChargeRegion.getDefaultRate();
        String chargeRegion = cvmChargeRegion.getRegion();
        List<UserRate> byUsername = rateRepository.findByUsernameAndChargeType(name, usedChargeType);
        float defaultRatePerMonth = byUsername.stream().filter(x -> x.getRegion().equalsIgnoreCase(DEFAULT_REGION_KEY)).findFirst().map(x -> x.getRate()).orElse(40.0F);
        if (StringUtils.isEmpty(region))
            return defaultRatePerMonth;
        UserRate userRate = byUsername.stream().filter(x -> Pattern.matches(x.getRegion(), chargeRegion)).findFirst().orElse(null);
        if (userRate == null) {
            return defaulRate;
        }
        return userRate.getRate();
    }

    private static final String DEFAULT_REGION_KEY = "--default";

    private float tryToFindClosedRate(String name, String region, int usedChargeType) {
        if (region != null && region.startsWith("vps:")) {
            name = "vpsclient";
        }

        List<UserRate> byUsername = rateRepository.findByUsernameAndChargeType(name, usedChargeType);

        float defaultRatePerMonth = byUsername.stream().filter(x -> x.getRegion().equalsIgnoreCase(DEFAULT_REGION_KEY)).findFirst().map(x -> x.getRate()).orElse(40.0F);

        if (StringUtils.isEmpty(region))
            return defaultRatePerMonth;
        else {
            float ratePerMonth = byUsername.stream().sorted((x1, x2) -> {
                int match1 = CommonStringUtils.getLongestCommonSubstring(region, x1.getRegion());
                int match2 = CommonStringUtils.getLongestCommonSubstring(region, x2.getRegion());
                return match2 - match1;
            }).findFirst().map(x -> {
                int match1 = CommonStringUtils.getLongestCommonSubstring(region, x.getRegion());
                if (match1 <= 3)
                    return defaultRatePerMonth;
                else
                    return x.getRate();
            }).orElse(defaultRatePerMonth);
            return ratePerMonth;
        }
    }

    @Override
    public String deletePreOrder(String username) {
        List<PreOrder> preOrders = preOrderRepository.findPreOrderByUserName(username);
        log.error("deletePreOrder size : " + preOrders.size());
        preOrders.forEach(x -> {
            log.error(x.toString());
            preOrderRepository.deleteById(x.getTradeNo());
        });
        return "success";
    }

    @Override
    public IpChargeResultDto preChargeByMoney(String name, double money) {
        IpChargeResultDto precharge = new IpChargeResultDto();
        CompletableFuture.runAsync(() -> {
            precharge.setApprovedAmount(0);
            precharge.setSuccess(false);
            Account account = repository.findByName(name);
            if (account != null) {
                UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
                float balance = 0;
                if (userBalance != null)
                    balance = userBalance.getBalanced();

                if (balance >= money) {
                    precharge.setBalance((int) Math.round(balance - money));
                    precharge.setSuccess(true);
                } else {
                    precharge.setBalance(Math.round(balance));
                }
            }
        }, AccountServicePool.taskPool).join();

        return precharge;
    }

    @Override
    public IpChargeResultDto preChargeIp(String name, int amount, double money) {
        IpChargeResultDto precharge = new IpChargeResultDto();
        CompletableFuture.runAsync(() -> {
            precharge.setApprovedAmount(0);
            precharge.setSuccess(false);
            Account account = repository.findByName(name);
            if (account != null) {
                UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);
                float balance = 0;
                if (userBalance != null)
                    balance = userBalance.getBalanced();
                if (balance >= money) {
                    precharge.setApprovedAmount(amount);
                    precharge.setBalance(Math.round(balance - (float) money));
                    precharge.setSuccess(true);
                } else {
                    precharge.setBalance(Math.round(balance));
                }
            }
        }, AccountServicePool.taskPool).join();
        return precharge;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Account findByName(String accountName) {
        Assert.hasLength(accountName);
        Account account = repository.findByName(accountName);
        if (account == null) {
            return null;
        }
        if (StringUtils.isEmpty(account.getParent())) {
            account.setPermission(15);
            account.setAllowedToCreateSubUser(true);
        }
        return account;
    }

    @Override
    public ResultDto getAccountByName(String name) {
        Assert.hasLength(name);
        ResultDto resultDto = new ResultDto();
        try {
            Account account = this.findByName(name);
            AccountDto current = new AccountDto(account);

            UserBalance userBalance = userBalanceRepository.findById(name).orElse(null);

            if (userBalance != null)
                current.setBalance(Math.round(userBalance.getBalanced()));
            else
                current.setBalance(0);

            if (StringUtils.isEmpty(account.getParent())) {
                current.setPermission(15);
                current.setAllowedToCreateSubUser(true);
            }

            List<AccountDto> child = repository.findByParent(name).stream().map(item -> new AccountDto(item)).collect(Collectors.toList());
            current.setChild(child);

            ShopSummary shopSummary = shopService.getShopSummary(name);
            if (shopSummary != null)
                current.setShopSummary(shopSummary);

            IpSummary ipSummary = ipResourceService.getIpSummary(name);
            if (ipSummary != null)
                current.setIpSummary(ipSummary);

            resultDto.setStatus(0);
            resultDto.setData(current);
        } catch (Exception e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", AccountErrorCode.NAMENOTEXIST);
            statusInfo.put("message", (AuthErrorCode.NAMENOTEXIST.getReason()));
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    public void deleteByName(String name) {
        userAuthService.deleteUser(name);
        repository.deleteById(name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Account create(User user) {
        return create(user, null);
    }

    public void deleteSub(String parent, String child) {
        Assert.hasText(parent);
        Account childAccount = repository.findByName(child);
        if (childAccount == null || !parent.equals(childAccount.getParent())) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }

        userAuthService.deleteUser(child);
        repository.delete(childAccount);
        Account parentAccount = repository.findByName(parent);
        parentAccount.setChildCount(parentAccount.getChildCount() - 1);
        repository.save(parentAccount);
    }

    @Override
    public Account createWithSms(User user) {

        Account existing = repository.findByName(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByPhoneNumber(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByPhoneNumber(user.getPhone());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getPhone());

        existing = repository.findByName(user.getPhone());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getPhone());

        Otp otp = otpRepository.findByPhoneAndCreatedAtGreaterThanEqual(user.getUsername(), Instant.now().minusSeconds(600).toEpochMilli());

        if (otp == null) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        }
        if (!otp.getOtp().equals(user.getAuthCode())) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        } else {
//            otpRepository.delete(otp);
        }

        Account account = new Account();

        account.setName(user.getUsername());
        account.setLastSeen(new Date());
        account.setEmail(user.getEmail());
        account.setPhoneNumber(user.getUsername());
        account.setPermission(15);
        account.setPromotion(new Promotion());
        account.setAllowedToCreateSubUser(true);

        user.setEnabled(true);

        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(user));
        repository.save(account);

        log.info("new account has been created: " + account.getName());

        notifyCustomerRegister(account);
        return account;
    }

    public Account create(User user, String parentName) {

        Account existing = repository.findByName(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByPhoneNumber(user.getUsername());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getUsername());

        existing = repository.findByEmail(user.getEmail());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.EMAILEXIST, "email already exists: " + user.getEmail());
        user.setEnabled(false);
        Account account = new Account();
        account.setName(user.getUsername());
        account.setLastSeen(new Date());
        account.setEmail(user.getEmail());

        if (StringUtils.isEmpty(parentName)) {
            account.setAllowedToCreateSubUser(true);
            account.setPermission(Integer.valueOf("1111",2));
        } else {
            account.setAllowedToCreateSubUser(user.isAllowedToCreateSubUser());
            account.setPermission(user.getPermission());
        }

        account.setParent(parentName);

        emailService.sendEmailVerification(user.getUsername(), user.getEmail(), user.getVerificationCode());

        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(user));
        repository.save(account);

        log.info("new account has been created: " + account.getName());

//        SmsUtils.notifyNewUserRegistered();
        notifyCustomerRegister(account);
        return account;
    }

    private void notifyCustomerRegister(Account contactUs) {
        StringBuilder sb = new StringBuilder();
        sb.append("Name: " + contactUs.getName() + "<br/>");
        sb.append("Email: " + contactUs.getEmail() + "<br/>");
        sb.append("Phone: " + contactUs.getPhoneNumber() + "<br/>");
        sb.append("Company: " + contactUs.getCompanyName() + "<br/>");
        sb.append("Title: " + contactUs.getJobTitle() + "<br/>");
        try {
            emailService.sendHtmlMail("sales@cloudam.io", "新客户注册:" + contactUs.getName(), sb.toString());
        } catch (Exception e) {
            log.error("sending email fails on customer regisration", e);
        }

    }

    @Override
    public List<UserDto> getAllDesendentUsers(String name, int level) {

        if (level == -1) {
            //find its parent.
            String parent = repository.findByName(name).getParent();

            if (StringUtils.isEmpty(parent))
                return Arrays.asList();
            UserDto dto = new UserDto();
            dto.setUsername(parent);
            return Arrays.asList(dto);
        }

        List<Account> accounts = repository.findByParent(name);

        if (accounts == null || accounts.size() <= 0)
            return new ArrayList<>();


        List<UserDto> collect = accounts.stream().map(a -> {
            UserDto user = new UserDto();
            user.setUsername(a.getName());
            user.setEmail(a.getEmail());
            return user;
        }).collect(Collectors.toList());

        if (level <= 0)
            return collect;
        // 遍历获取所有子员工
        List<UserDto> subUserList = new ArrayList<>();
        for (UserDto userDto : collect) {
            subUserList.addAll(getAllDesendentUsers(userDto.getUsername(), level));
        }
        collect.addAll(subUserList);
        return collect;
    }

    @Override
    public SubUserPageResultDto getSubUserList(String name, int page, int amount) {
        if (amount > 100)
            amount = 100;
        Pageable pageable = PageRequest.of(page, amount);
        Page<Account> accounts = repository.findByParent(name, pageable);
        SubUserPageResultDto subUserPageResultDto = new SubUserPageResultDto();

        if (accounts == null || accounts.getNumberOfElements() <= 0)
            return subUserPageResultDto;

        List<SubUsersDto> subUsersDtoList = new ArrayList<>();

        accounts.getContent().forEach(x -> {
            com.edgec.browserbackend.auth.domain.User user = userRepository.findById(x.getName()).orElse(null);
            subUsersDtoList.add(new SubUsersDto(x, user.getPassword()));
        });

        subUserPageResultDto.setUserList(subUsersDtoList);
        PageInfo pageInfo = new PageInfo();
        pageInfo.setTotalItems((int)accounts.getTotalElements());
        pageInfo.setCurrentPage(accounts.getNumber());
        pageInfo.setTotalPages(accounts.getTotalPages());
        subUserPageResultDto.setUserPage(pageInfo);
        return subUserPageResultDto;
    }

    public Account createSub(String name, AccountDto user) {
        Account existing = repository.findByName(name);
        if (existing == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + name);
        if (!existing.isAllowedToCreateSubUser() && !StringUtils.isEmpty(existing.getParent()))
            throw new ClientRequestException(AccountErrorCode.NOTALLOWEDTOCREATESUBUSER, "Not allowed to create sub user");

        if (existing.getChildCount() >= 1000) {
            throw new ClientRequestException(AccountErrorCode.CHILDMAX, "account cannot have more children");
        }

        existing = repository.findByName(user.getName());
        if (existing != null)
            throw new ClientRequestException(AccountErrorCode.NAMEEXIST, "account already exists: " + user.getName());

        if (!StringUtils.isEmpty(user.getEmail())) {
            existing = repository.findByEmail(user.getEmail());
            if (existing != null)
                throw new ClientRequestException(AccountErrorCode.EMAILEXIST, "email already exists: " + user.getEmail());
        }

        if (!StringUtils.isEmpty(user.getPhoneNumber())) {
            existing = repository.findByPhoneNumber(user.getPhoneNumber());
            if (existing != null)
                throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getEmail());
        }

        existing = repository.findByPhoneNumber(user.getName());
        if (existing != null) {
            throw new ClientRequestException(AccountErrorCode.PHONEEXIST, "phone number already exists: " + user.getEmail());
        }

        User authUser = new User();
        authUser.setUsername(user.getName());
        authUser.setPassword(user.getPassword());
        authUser.setPhone(user.getPhoneNumber());
        authUser.setEmail(user.getEmail());
        authUser.setEnabled(true);
        authUser.setPermission(user.getPermission());
        userAuthService.create(new com.edgec.browserbackend.auth.domain.User(authUser));

        Account account = new Account(user);
        account.setPhoneNumber(user.getPhoneNumber());
        account.setEmail(user.getEmail());
        account.setParent(name);
        repository.save(account);


//        emailService.sendEmailVerification(user.getUsername(), user.getEmail(), user.getVerificationCode());
        log.info("new account has been created: " + account.getName());
        return account;
    }

    public static String makeRandomPassword(int len){
        char charr[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890~!@$%^&*.?".toCharArray();
        StringBuilder sb = new StringBuilder();
        Random r = new Random();
        for (int x = 0; x < len; ++x) {
            sb.append(charr[r.nextInt(charr.length)]);
        }
        return sb.toString();
    }

    @Override
    public void createSubUsers(String name, SubUsersRequestDto subUsersRequestDto) {
        Account existing = repository.findByName(name);
        if (existing == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + name);
        if (!existing.isAllowedToCreateSubUser() && !StringUtils.isEmpty(existing.getParent()))
            throw new ClientRequestException(AccountErrorCode.NOTALLOWEDTOCREATESUBUSER, "Not allowed to create sub user");

        if (existing.getChildCount() + subUsersRequestDto.getAmount() >= 1000) {
            throw new ClientRequestException(AccountErrorCode.CHILDMAX, "account cannot have more children");
        }

        int nameNumber = existing.getChildCount();
        int time = subUsersRequestDto.getAmount();
        for (int i = 0; i < time; i++) {
            if (nameNumber >= 1000)
                throw new ClientRequestException(AccountErrorCode.CHILDMAX, "account cannot have more children");
            AccountDto user = new AccountDto();
            nameNumber++;
            if (nameNumber < 10)
                user.setName(existing.getName() + "00" + nameNumber);
            else if (nameNumber < 100)
                user.setName(existing.getName() + "0" + nameNumber);
            else
                user.setName(existing.getName() + nameNumber);

            Account child = repository.findByName(user.getName());
            if (child != null) {
                time++;
                continue;
            }

            User authUser = new User();
            authUser.setUsername(user.getName());
            if (org.apache.commons.lang3.StringUtils.isNotBlank(subUsersRequestDto.getPassword()))
                authUser.setPassword(subUsersRequestDto.getPassword());
            else
                authUser.setPassword(makeRandomPassword(8));
            authUser.setPhone(existing.getPhoneNumber());
            authUser.setEnabled(true);
            authUser.setPermission(subUsersRequestDto.getPermission());
            userAuthService.create(new com.edgec.browserbackend.auth.domain.User(authUser));

            Account account = new Account(user);
            account.setPhoneNumber(existing.getPhoneNumber());
            account.setParent(name);
            account.setPermission(subUsersRequestDto.getPermission());
            repository.save(account);

//        emailService.sendEmailVerification(user.getUsername(), user.getEmail(), user.getVerificationCode());
            log.info("new account has been created: " + account.getName());
        }

        existing.setChildCount(nameNumber);
        repository.save(existing);
    }


    public Account saveSub(String name, AccountDto user) {
        Account childAccount = repository.findByName(user.getName());
        if (childAccount == null || !name.equals(childAccount.getParent())) {
            throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
        }
        childAccount.setAllowedToCreateSubUser(user.isAllowedToCreateSubUser());
        childAccount.setPhoneNumber(user.getPhoneNumber());
        childAccount.setJobTitle(user.getJobTitle());
        childAccount.setCompanyName(user.getCompanyName());
        childAccount.setPermission(user.getPermission());
        repository.save(childAccount);
        return childAccount;
    }

    @Override
    public void saveSubUsers(String name, SubUsersRequestDto subUsersRequestDto) {
        for (String username: subUsersRequestDto.getUsernames()) {
            Account childAccount = repository.findByName(username);
            if (childAccount == null || !name.equals(childAccount.getParent())) {
                throw new ClientRequestException(AccountErrorCode.UNKNOWN, "Invalid Request");
            }

            if (subUsersRequestDto.getPassword() != null) {
                com.edgec.browserbackend.auth.domain.User user = new com.edgec.browserbackend.auth.domain.User();
                user.setUsername(username);
//                user.setPassword(subUsersRequestDto.getPassword());
                userAuthService.resetUserPassword(user);
            }
            if (subUsersRequestDto.getNickname() != null)
                childAccount.setNickname(subUsersRequestDto.getNickname());
            if (subUsersRequestDto.getPermission() != -1)
                childAccount.setPermission(subUsersRequestDto.getPermission());
            if (subUsersRequestDto.getComment() != null)
                childAccount.setComment(subUsersRequestDto.getComment());
            repository.save(childAccount);
        }

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Account saveChanges(String name, Account update) {

        Account account = repository.findByName(name);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "can't find account with name " + name);
        String preEmail = account.getEmail();
        account.setNote(update.getNote());
        account.setLastSeen(new Date());
        account.setCompanyName(update.getCompanyName());
        account.setJobTitle(update.getJobTitle());
//        account.setPhoneNumber(update.getPhoneNumber());
        account.setEmail(update.getEmail());
        account.setPermission(update.getPermission());
        account.setWhiteList(update.getWhiteList());
        log.debug("account {} changes has been saved", name);
        if (!org.apache.commons.lang3.StringUtils.equalsIgnoreCase(preEmail, update.getEmail())) {
            Account account1 = repository.findByEmail(update.getEmail());
            if (account1 != null) {
                throw new ClientRequestException(AccountErrorCode.EMAILEXIST);
            }
            User newuser = new User();
            newuser.setUsername(account.getName());
            newuser.setEmail(update.getEmail());
            userAuthService.updateUser(name, new com.edgec.browserbackend.auth.domain.User(newuser));
        }
        repository.save(account);
        return account;
    }

    @Override
    public void resetPassword(UserDto user) {
        if (user == null)
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + user);

        Account account = repository.findByEmail(user.getEmail());
        if (account == null) {
            Account accountname = repository.findByName(user.getUsername());
            if (accountname == null)
                throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + user);
            account = accountname;
        }

        User newuser = new User();
        newuser.setUsername(account.getName());
        String code = UUID.randomUUID().toString() + System.currentTimeMillis();
        newuser.setVerificationCode(code);
        userAuthService.reset(new com.edgec.browserbackend.auth.domain.User(newuser));
        emailService.sendResetPassword(account.getName(), account.getEmail(), code);
        log.info("password has been reset for: " + user);

    }

    @Override
    public UserDto determUsernameOrEmail(String text) {
        if (StringUtils.isEmpty(text))
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + text);
        Account account = repository.findByEmail(text);
        if (account == null)
            account = repository.findByName(text);
        if (account == null)
            throw new ClientRequestException(AccountErrorCode.NAMEOREMAILNOTEXIST, "Can't find user with name or mail: " + text);
        UserDto user = new UserDto();
        user.setEmail(account.getEmail());
        user.setUsername(account.getName());
        return user;
    }

    @Override
    public void resetPasswordWithOtp(UserDto user) {

        Otp otp = otpRepository.findByPhoneAndCreatedAtGreaterThanEqual(user.getUsername(),
                Instant.now().minusSeconds(600).toEpochMilli());
        if (otp == null) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        }
        if (!otp.getOtp().equals(user.getOtp())) {
            throw new ClientRequestException(AccountErrorCode.OTPWRONG, AccountErrorCode.OTPWRONG.getReason());
        }

        Account account = repository.findById(user.getUsername()).orElse(null);

        if (account == null)
            throw new ClientRequestException(AccountErrorCode.PHONENOTEXIST, "Can't find account with phone " + user.getPhone());

        User client = new User();
        client.setUsername(account.getName());
        client.setPassword(user.getPasswd());

        userAuthService.resetUserPassword(new com.edgec.browserbackend.auth.domain.User(client));

    }

    @Override
    public Page<UserPayment> getUserPayment(Pageable pageable, String username) {
        Page<UserPayment> userPayments = userPaymentRepository.findByUsernameAndSucceedOrderByPaymentDateDesc(pageable, username, true);

        return userPayments;
    }

    @Override
    public void sendSmsOtp(String phone) {
        String code = com.edgec.browserbackend.account.service.SmsUtils.sendSmsOTP(phone);
        Otp otp = new Otp();
        otp.setPhone(phone);
        otp.setOtp(code);
        otp.setCreatedAt(Instant.now().toEpochMilli());
        otpRepository.save(otp);
    }

    @Override
    public AccountDto getAccountByCellphone(String cellphone) {
        Account account = repository.findByPhoneNumber(cellphone);
        if (account == null) {
            return null;
        }
        return new AccountDto(account);
    }

    @Override
    public Page<UserPrePaidBilling> listPrepaid(Pageable pageable, String username, String[] chargeType, int year, int month, int day, String zoneId) {

        Page<UserPrePaidBilling> userPrePaidBillings = prePaidBillingRepository.findByUsernameAndChargeTypes(pageable, username, chargeType, year, month, day, zoneId);

        return userPrePaidBillings;
    }

    @Override
    public Page<UserPrePaidBilling> listBills(Pageable pageable, String username, BillQueryCriteriaDto billQueryCriteriaDto, String dateFrom, String dateTo, String zoneId) {
        zoneId = zoneId.replace(" ", "+");
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        List<Integer> dayBegin = listYearMonthDayByStrDate(dateFrom);
        ZonedDateTime zonedDateTimeDayBegin = ZonedDateTime.of(dayBegin.get(0), dayBegin.get(1), dayBegin.get(2), 0, 0, 0, 0, zoneOffset);
        List<Integer> dayEnd = listYearMonthDayByStrDate(dateTo);
        ZonedDateTime zonedDateTimeDayEnd = ZonedDateTime.of(dayEnd.get(0), dayEnd.get(1), dayEnd.get(2), 0, 0, 0, 0, zoneOffset);
        long dayBeginTime = zonedDateTimeDayBegin.withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
        long dayEndTime = zonedDateTimeDayEnd.plusDays(1).withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;

        return prePaidBillingRepository.findBillsByCondition(pageable, username, billQueryCriteriaDto, dayBeginTime, dayEndTime);
    }


    public static void main(String[] args) {
        String zoneId = "+08:00";
        ZoneOffset zoneOffset = ZoneOffset.of(zoneId);
        String dataFrom = "2019-11-20";
        // 获取北京时间 2019年11月20日 0点
        ZonedDateTime zonedDateTime = ZonedDateTime.of(2019, 11, 20, 0, 0, 0, 0, zoneOffset);
        long dayBeginTime = zonedDateTime.withZoneSameInstant(ZoneOffset.UTC).toEpochSecond() * 1000;
        System.out.println(dayBeginTime);
    }


    //根据传入的格式为 yyyy-mm-dd来获取年月日，
    private List<Integer> listYearMonthDayByStrDate(String strDate) {
        List<Integer> nums = new ArrayList<>();
        String[] dates = strDate.split("-");
        int year = 0, month = 0, day = 0;
        if (dates.length > 2) {
            year = Integer.parseInt(dates[0]);
            month = Integer.parseInt(dates[1]);
            day = Integer.parseInt(dates[2]);
        }
        nums.add(year);
        nums.add(month);
        nums.add(day);
        return nums;
    }

    @Override
    public Page<Account> listAccountBySingupDate(Pageable pageable, Date beginDate, Date endDate) {
        return repository.findAllBySignupDateBetween(pageable, beginDate, endDate);
    }

    @Override
    public List<UserBillList> findOverviewByYearAndMonth(String username, int year, int month, String zoneId, Services service) {
        List<UserBillList> userBillLists = new ArrayList<>();
        List<UserPrePaidBilling> userPrePaidBilling = prePaidBillingRepository
                .findOverviewByYearAndMonth(username, year, month, zoneId, service);
        userPrePaidBilling.forEach(x -> {
            UserBillList userBillList = new UserBillList();
            userBillList.setPeriod(year + "-" + (month < 10 ? "0" + month : month));
            userBillList.setBill(x);
            userBillList.setStatus(x.getStatus());
            userBillLists.add(userBillList);
        });
        return userBillLists;
    }

    @Override
    public List<Invoice> findByUsername(String username) {
        List<Invoice> invoices = invoiceRepository.findByUsernameOrderByDateDesc(username);
        invoices.forEach(x -> {
            x.setExpressInformation(x.getUsername() + " " + x.getPhone() + " " + x.getExpressInformation());
        });
        return invoices;
    }

    @Override
    public boolean writeInvoice(Invoice invoice) {
        int num = invoiceSize(invoice.getUsername());
        if (invoice.getAmount() > num || invoice.getAmount() < 100) {
            return false;
        }
        invoiceRepository.save(invoice);
        return true;
    }

    @Override
    public int invoiceSize(String username) {
        List<UserPayment> amounts = userPaymentRepository.findByUsernameAndSucceedAndTradeNoGreaterThan(username, true, "201911290000000000");
        int amount = amounts.stream().map(UserPayment::getAmount).mapToInt((x) -> x).sum();
        List<Invoice> invoices = invoiceRepository.findByUsername(username);
        float invoiceAmount = (float) invoices.stream().map(Invoice::getAmount).mapToDouble((x) -> x).sum();
        return (int) (amount - invoiceAmount);
    }

    @Override
    public void updateUserToken(String username, String token) {
        Account byName = repository.findByName(username);
        byName.setToken(token);
        repository.save(byName);
    }

}
