package com.edgec.browserbackend.browser.controller;

import com.edgec.browserbackend.account.dto.ResultDto;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.Shop;
import com.edgec.browserbackend.browser.dto.ShopPageResultDto;
import com.edgec.browserbackend.browser.dto.ShopResultDto;
import com.edgec.browserbackend.browser.dto.ShopRequestDto;
import com.edgec.browserbackend.browser.dto.ShopStringResultDto;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.utils.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/shop")
public class ShopController {

    private final Logger logger = LoggerFactory.getLogger(ShopController.class);

    @Autowired
    private ShopService shopService;

    @RequestMapping(value = "/add", method = RequestMethod.POST)
    public ResultDto addShop(Principal principal, @RequestBody ShopResultDto shopResultDto) {
        ResultDto resultDto = new ResultDto();
        try {
            ShopStringResultDto shopStringResultDto = new ShopStringResultDto();
            shopStringResultDto.setId(shopService.addShop(principal.getName(), shopResultDto));
            resultDto.setData(shopStringResultDto);
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/multiadd", method = RequestMethod.POST)
    public ResultDto addShops(Principal principal, @RequestParam("file") MultipartFile file) {
        ResultDto resultDto = new ResultDto();
        String name = file.getOriginalFilename();
        if(name.length()<6 || !name.substring(name.length()-5).equals(".xlsx")){
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", BrowserErrorCode.INFORMATIONNOTCOMPELETE.getCode());
            statusInfo.put("message", "文件格式错误");
            resultDto.setStatusInfo(statusInfo);
        }
        try {
            shopService.addShops(principal.getName(), file);
            resultDto.setStatus(0);
        } catch (ClientRequestException | IOException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", 40102);
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }


        return resultDto;

    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public ResultDto updateShop(Principal principal, @RequestBody ShopResultDto shopResultDto) {
        ResultDto resultDto = new ResultDto();
        try {
            ShopStringResultDto shopStringResultDto = new ShopStringResultDto();
            shopStringResultDto.setId(shopService.updateShop(principal.getName(), shopResultDto));
            resultDto.setData(shopStringResultDto);
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    public ResultDto deleteShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            shopService.deleteShop(principal.getName(), shopRequestDto.getShopId());
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/bind", method = RequestMethod.POST)
    public ResultDto bindShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            shopService.bindShop(principal.getName(), shopRequestDto.getShopId(), shopRequestDto.getIpAddr());
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/unbind", method = RequestMethod.POST)
    public ResultDto unBindShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            shopService.unBindShop(principal.getName(), shopRequestDto.getShopId(), shopRequestDto.getIpAddr());
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/transfer", method = RequestMethod.POST)
    public ResultDto transferShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            shopService.transferShop(principal.getName(), shopRequestDto.getShopId(), shopRequestDto.getToGroup());
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/assign", method = RequestMethod.POST)
    public ResultDto assignShop(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            shopService.assignShops(principal.getName(), shopRequestDto.getShopIds(), shopRequestDto.getToUsers());
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }

    @RequestMapping(value = "/list",method = RequestMethod.POST)
    public ResultDto getShopList(Principal principal, @RequestBody ShopRequestDto shopRequestDto) {
        ResultDto resultDto = new ResultDto();
        try {
            ShopPageResultDto shopDtos = shopService.getShopList(principal.getName(), shopRequestDto.getGroup(), shopRequestDto.getPage(),
                    shopRequestDto.getAmount(), shopRequestDto.getFilter());
            resultDto.setData(shopDtos);
            resultDto.setStatus(0);
        }catch (ClientRequestException e) {
            resultDto.setStatus(-1);
            Map<String, Object> statusInfo = new HashMap<>();
            statusInfo.put("code", e.getErrorCode());
            statusInfo.put("message", e.getMessage());
            resultDto.setStatusInfo(statusInfo);
        }
        return resultDto;
    }
}
