package com.edgec.browserbackend.browser.service.Impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.domain.IpChargeRequestDto;
import com.edgec.browserbackend.account.domain.IpChargeResultDto;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.dto.*;
import com.edgec.browserbackend.browser.repository.*;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.browser.service.IpResourceService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.common.commons.utils.NotifyUtils;
import com.edgec.browserbackend.common.utils.FileUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

@Service
public class IpResourceServiceImpl implements IpResourceService {

    private final Logger logger = LoggerFactory.getLogger(IpResourceServiceImpl.class);

    private static String CLOUDAMURL = "https://www.cloudam.cn";
    private static String TESTURL = "http://112.74.13.2";
    private static String USERNAME = "fangguanlianbrowser";

    public static List<String> region = Arrays.asList(
            "asiapa", "hongkong", "japan", "s-korea", "us", "malaysia",
            "yajiada", "singapore", "australia", "germany", "uk", "brazil",
            "moscow", "canada", "france", "sweden", "s-korea", "india", "meast",
            "brazil", "virginia", "ohio", "california", "oregon", "ireland", "london", "ireland"
    );

    public static String genRandom(int srcFlag, int length) {
        String retStr = "";
        String strTable = "";
        switch (srcFlag) {
            case 1:
                strTable = "1234567890";
                break;
            case 2:
                strTable = "1234567890abcdefghijklmnopqrstuvwxyz";
                break;
            case 3:
                strTable = "12345678901234567890abcdefghijkmnpqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
                break;
            case 4:
                strTable = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
                break;
            case 5:
                strTable = "abcdefghijklmnopqrstuvwxyz";
                break;
            default:
                strTable = "1234567890abcdefghijkmnpqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
                break;
        }

        int len = strTable.length() - 3;
        boolean bDone = true;
        do {
            retStr = "";
            int count = 0;
            for (int i = 0; i < length; i++) {
                double dblR = Math.random() * len;
                int intR = (int) Math.floor(dblR);
                char c = strTable.charAt(intR);
                if (('0' <= c) && (c <= '9')) {
                    count++;
                }
                retStr += strTable.charAt(intR);
            }
            if (count >= 2) {
                bDone = false;
            }
        } while (bDone);
        retStr += "a";
        retStr += "A";
        retStr += "1";
        return retStr;
    }

    private static List<String> port = Arrays.asList("20000", "20001");

    private static List<String> protocol = Arrays.asList("socks5", "proxy");

    @Value("${spring.profiles.active}")
    private String profiles;

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Autowired
    private UserShopRepository userShopRepository;

    @Autowired
    private ShopRepository shopRepository;

    @Autowired
    private AccountService accountService;

    @Autowired
    private IpOptionsRepository ipOptionsRepository;

    @Autowired
    private PlatformOptionsRepository platformOptionsRepository;

    @Autowired
    private IpAndShopService ipAndShopService;

    @Autowired
    private SpecialLineRepository specialLineRepository;

    public HttpHeaders buildPostHeader() {
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        if (profiles.equals("dev"))
            header.setBearerAuth("oq5tg3gMsflHcK5iZ2741G5R30XYd9blyOqH9qeBItKtrzfTsGIoy8AsxqqNXdcm");
        else if (profiles.equals("prod"))
            header.setBearerAuth("tKWsuHzcngf0RQPMss70f9jgymDIwgQ9zbLfESJdcou3pZSNWl7lNTzto8VQgwaO");
        return header;
    }

    public HttpHeaders buildGetHeader() {
        HttpHeaders headers = new HttpHeaders();
        if (profiles.equals("dev"))
            headers.setBearerAuth("oq5tg3gMsflHcK5iZ2741G5R30XYd9blyOqH9qeBItKtrzfTsGIoy8AsxqqNXdcm");
        else if (profiles.equals("prod"))
            headers.setBearerAuth("tKWsuHzcngf0RQPMss70f9jgymDIwgQ9zbLfESJdcou3pZSNWl7lNTzto8VQgwaO");
        return headers;
    }

    @Override
    public List<String> buyIp(String username, IpResourceRequestDto ipResourceRequestDto) throws Exception {
        // 1. 数据校验
        if (ipResourceRequestDto.getRegion() == null || ipResourceRequestDto.getRegionCn() == null) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }


        // 2. 计算 ip 单价信息
        double newprice = 0;
        if (!"own".equals(ipResourceRequestDto.getVendor())) {
            Map<String, List<String>> priceList = ipOptionsRepository.findAll().get(0).getIpPlatForm();
            List<String> vendorPrices = priceList.get(ipResourceRequestDto.getRegionCn());
            String price = vendorPrices.stream()
                                       .filter(x -> Vendor.valueOf(ipResourceRequestDto.getVendor()).getValue().equals(x.substring(0, x.indexOf("-"))))
                                       .map(x -> x.substring(x.lastIndexOf("-") + 1))
                                       .collect(Collectors.joining());
            newprice = "week".equals(ipResourceRequestDto.getUnit()) ? (Integer.valueOf(price) / 3) : Integer.valueOf(price);
        }

        // 3. 计算总共需要花费多少钱 并 校验账户余额 够不够扣费
        IpChargeResultDto ipChargeResultDto = accountService.preChargeByMoney(username, newprice * ipResourceRequestDto.getAmount() * ipResourceRequestDto.getPeriod());
        if (!ipChargeResultDto.isSuccess()) {
            throw new ClientRequestException(AccountErrorCode.NOTENOUGHBALANCE);
        }

        // 生成随机密码
        String password = StringUtils.isNotBlank(ipResourceRequestDto.getPassword()) ? ipResourceRequestDto.getPassword() : genRandom(3, 12);

        List<IpResourceDto> ipResourceDtos = new ArrayList<>();
        List<String> ipIds = new ArrayList<>();

        for (int i = 0; i < ipResourceRequestDto.getAmount(); i++) {
            IpResource ipResource = new IpResource();
            ipResource.setPeriod(ipResourceRequestDto.getPeriod());
            //充6送1
            if (ipResourceRequestDto.getUnit().equals("month")) {
                if (ipResourceRequestDto.getPeriod() == 6) {
                    ipResource.setPeriod(7);
                }
                if (ipResourceRequestDto.getPeriod() == 12) {
                    ipResource.setPeriod(14);
                }
            }

            // 根据 ip 的 vendor 来封装部分的 ipResource 信息
            if (ipResourceRequestDto.getVendor().equals("local")) {
                ipResource.setAddr("本地Ip未使用");
                ipResource.setIpType(IpType.LOCAL);
                ipResource.setVendor(Vendor.valueOf(ipResourceRequestDto.getVendor()));
                ipResource.setVendorCn("本地");
                ipResource.setStatus(4);

                long validTime = 0;
                if (ipResourceRequestDto.getUnit().equals("week")) {
                    validTime = Instant.now().atZone(ZoneOffset.UTC).plusWeeks(ipResource.getPeriod()).toInstant().toEpochMilli();
                } else {
                    validTime = Instant.now().atZone(ZoneOffset.UTC).plusMonths(ipResource.getPeriod()).toInstant().toEpochMilli();
                }
                ipResource.setValidTime(validTime);
                ipResource.setUsername(USERNAME);
                ipResource.setPort(port);
            } else if (ipResourceRequestDto.getVendor().equals("own")) {
                if (ipResourceRequestDto.getAddr() == null || ipResourceRequestDto.getAddr().isEmpty() || StringUtils.isBlank(ipResourceRequestDto.getAddr().get(i))) {
                    throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
                }

                ipResource.setSpecialLine(ipResourceRequestDto.isSpecialLine());
                ipResource.setAddr(ipResourceRequestDto.getAddr().get(i));
                ipResource.setIpType(IpType.OWN);
                ipResource.setVendor(Vendor.valueOf(ipResourceRequestDto.getVendor()));
                ipResource.setVendorCn("自有");
                ipResource.setStatus(4);
                ipResource.setUsername(ipResourceRequestDto.getUsername());
                ipResource.setValidTime(Instant.now().atZone(ZoneOffset.UTC).toInstant().toEpochMilli());
                ipResource.setPort(ipResourceRequestDto.getPorts());
            } else {
                ipResource.setAddr("");
                ipResource.setIpType(IpType.VENDOR);
                ipResource.setVendor(Vendor.valueOf(ipResourceRequestDto.getVendor()));
                switch (ipResourceRequestDto.getVendor()) {
                    case "aliyun":
                        ipResource.setVendorCn("阿里云");
                        break;
                    case "tencent":
                        ipResource.setVendorCn("腾讯云");
                        break;
                    case "aws":
                        ipResource.setVendorCn("亚马逊云");
                        break;
                }
                ipResource.setStatus(6);

                long validTime = 0;
                if (ipResourceRequestDto.getUnit().equals("week")) {
                    validTime = Instant.now().atZone(ZoneOffset.UTC).plusWeeks(ipResource.getPeriod()).toInstant().toEpochMilli();
                } else {
                    validTime = Instant.now().atZone(ZoneOffset.UTC).plusMonths(ipResource.getPeriod()).toInstant().toEpochMilli();
                }
                ipResource.setValidTime(validTime);
                ipResource.setUsername(USERNAME);
                ipResource.setPort(port);
            }

            if (account.getParent() != null) {
                ipResource.setUserParent(account.getParent());
            }

            ipResource.setRegion(ipResourceRequestDto.getRegion());
            // 只要 申请的 ip 地域不是在 大陆，则一定会执行下面 if 中的代码块
            boolean condition1 = StringUtils.isNotBlank(ipResource.getRegion()) && region.contains(ipResource.getRegion());
            boolean condition2 = ipResourceRequestDto.getVendor().equals("own") && ipResource.isSpecialLine();
            if (condition1 || condition2) {
                ipResource.setProxyUsername(ipResource.getAddr());
                ipResource.setProxyPassword(genRandom(3, 12));
                ipResource.setSpecialLine(true);
                ipResource.setHealthLockTimestamp(Instant.now().minusSeconds(60 * 20).toEpochMilli());
            }

            ipResource.setPurchasedTime(Instant.now().toEpochMilli());
            ipResource.setRegionCn(ipResourceRequestDto.getRegionCn());
            ipResource.setProtocol(protocol);
            ipResource.setPassword(password);
            ipResource.setOwner(username);
            ipResource.setLocked(false);
            ipResource.setLockTimestamp(Instant.now().toEpochMilli());
            ipResource.setUnit(ipResourceRequestDto.getUnit());
            ipResource.setPrice(newprice);

            IpResource ipResource1 = ipResourceRepository.save(ipResource);

            ipResourceDtos.add(new IpResourceDto(ipResource1, null, false));

            ipIds.add(ipResource1.getId());
        }

        if (!ipResourceRequestDto.getVendor().equals("own")) {
            // 封装购买 ip 的费用 信息
            IpChargeRequestDto ipChargeRequestDto = buildIpChargeRequestDto(ipResourceRequestDto, 1, ipResourceRequestDto.getPayMethod());
            // 用来记录购买与消费的操作  userbalance   与 费用明细  userprepaidbilling
            accountService.chargeByMoney(username, newprice * ipChargeRequestDto.getAmount() * ipChargeRequestDto.getPeriod(), ipChargeRequestDto);
        }

        // 如果ip需要绑定店铺，则将店铺与ip绑定 todo 感觉这个地方代码用不到，
        if (ipResourceRequestDto.getShopId() != null) {
            ShopRequestDto shopRequestDto = new ShopRequestDto();
            shopRequestDto.setIpId(ipResourceDtos.get(0).getId());
            shopRequestDto.setShopId(ipResourceRequestDto.getShopId());
            ipAndShopService.bindShop(username, shopRequestDto);
        }

        return ipIds;
    }

    @Override
    public IpOperationResultDto renewIp(String username, IpResourceRequestDto ipResourceRequestDto) {
        // 1. 校验当前用户是否存在
        accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        // 2. 获取 ip 资源 地域 - 云厂商 - 价格 信息
        Map<String, List<String>> ipPlatForm = ipOptionsRepository.findAll().get(0).getIpPlatForm();

        // 3. 计算如果是按月续费ip一个月一共需要花费多少钱，如果是按周学费，则在原来的基础上除以3。
        AtomicReference<Double> totalprice = getDoubleAtomicReference(ipResourceRequestDto, ipPlatForm);
        String unit = ipResourceRequestDto.getUnit();
        double newprice = unit.equals("week") ? (int) (totalprice.get() / 3) : totalprice.get().intValue();

        // 4. 计算总共需要花费多少钱 并 校验账户余额 够不够扣费， todo 这个地方 * ipResourceRequestDto.getAmount() 应该是多余的，因为在续费的时候，这个值始终为1
        IpChargeResultDto ipChargeResultDto = accountService.preChargeByMoney(username, newprice * ipResourceRequestDto.getPeriod() * ipResourceRequestDto.getAmount());
        if (!ipChargeResultDto.isSuccess()) {
            throw new ClientRequestException(AccountErrorCode.NOTENOUGHBALANCE);
        }

        int period = ipResourceRequestDto.getPeriod();

        // 5. 按月充值充6送1
        period = setAndGetPeriod(unit, period);

        // 6. 实际业务逻辑
        IpOperationResultDto ipOperationResultDto = new IpOperationResultDto();
        List<String> failedList = ipResourceRequestDto.getAddr();
        if (ipResourceRequestDto.getAddr() != null && ipResourceRequestDto.getAddr().size() > 0) {
            // 6.1 远程调用 云E 续费 传入的所有的 ip，校验是否续费成功
            RenewIpResultDto renewIpResultDto = getRenewIpResultDto(ipResourceRequestDto, period);
            if (StringUtils.isNotBlank(renewIpResultDto.getErrorCode())) {
                logger.error(renewIpResultDto.getErrorCode());
                throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
            }

            // 6.2 远程续费成功后的业务
            renewIpResultDto.getIplist().forEach(
                    x -> {
                        IpResource ipResource = ipResourceRepository.findByAddrAndIsDeleted(x.getIp(), false);
                        List<String> vendorPrices = ipPlatForm.get(ipResource.getRegionCn());
                        // 计算 ip 价格
                        String price = vendorPrices.stream()
                                                   .filter(vendorprice -> ipResource.getVendor().getValue().equals(vendorprice.substring(0, vendorprice.indexOf("-"))))
                                                   .map(vendorprice -> vendorprice.substring(vendorprice.lastIndexOf("-") + 1))
                                                   .collect(Collectors.joining());
                        double newprice1 = ipResourceRequestDto.getUnit().equals("week") ? (Integer.valueOf(price) / 3) : Integer.valueOf(price);

                        // 封装 ip 的续费 信息
                        IpChargeRequestDto ipChargeRequestDto = buildIpChargeRequestDto(ipResourceRequestDto, 2, ipResourceRequestDto.getPayMethod());
                        ipChargeRequestDto.setChargeIp(x.getIp());

                        // 用来记录购买与消费的操作  userbalance   与 费用明细  userprepaidbilling
                        accountService.chargeByMoney(username, newprice1 * ipResourceRequestDto.getPeriod() * ipResourceRequestDto.getAmount(), ipChargeRequestDto);

                        // 更新ip资源的到期时间
                        ipResource.setValidTime(Instant.parse(x.getValidTill()).toEpochMilli());
                        ipResourceRepository.save(ipResource);

                        // 封装 ipOperationResultDto
                        ipOperationResultDto.getSuccessList().add(x.getIp());
                        failedList.remove(x.getIp());
                    }
            );

            // 封装 ipOperationResultDto
            ipOperationResultDto.setFailList(failedList);
        }
        if (ipResourceRequestDto.getIpId() != null && ipResourceRequestDto.getIpId().size() > 0) {
            int finalPeriod = period;
            ipResourceRequestDto.getIpId().forEach(
                    x -> {
                        IpResource ipResource = ipResourceRepository.findByIdAndIsDeleted(x, false);
                        List<String> vendorPrices;
                        if (!ipResourceRequestDto.getVendor().equals("own"))
                            vendorPrices = ipPlatForm.get(ipResource.getRegionCn());
                        else {
                            if (ipResource.isSpecialLine())
                                vendorPrices = ipPlatForm.get("自有专线");
                            else
                                vendorPrices = ipPlatForm.get("自有");
                        }

                        String price = vendorPrices.stream()
                                                   .filter(vendorprice -> ipResource.getVendor().getValue().equals(vendorprice.substring(0, vendorprice.indexOf("-"))))
                                                   .map(vendorprice -> vendorprice.substring(vendorprice.lastIndexOf("-") + 1))
                                                   .collect(Collectors.joining());

                        double newprice1 = ipResourceRequestDto.getUnit().equals("week") ? (Integer.valueOf(price) / 3) : Integer.valueOf(price);
                        IpChargeRequestDto ipChargeRequestDto = buildIpChargeRequestDto(ipResourceRequestDto, 2, ipResourceRequestDto.getPayMethod());
                        ipChargeRequestDto.setChargeIp(x);
                        accountService.chargeByMoney(username, newprice1 * ipResourceRequestDto.getPeriod() * ipResourceRequestDto.getAmount(), ipChargeRequestDto);

                        if (ipResourceRequestDto.getUnit().equals("week")) {
                            ipResource.setValidTime(Instant.ofEpochMilli(ipResource.getValidTime()).atZone(ZoneOffset.UTC)
                                                           .plusWeeks(finalPeriod).toInstant().toEpochMilli());
                        } else {
                            ipResource.setValidTime(Instant.ofEpochMilli(ipResource.getValidTime()).atZone(ZoneOffset.UTC)
                                                           .plusMonths(finalPeriod).toInstant().toEpochMilli());
                        }
                        ipResourceRepository.save(ipResource);
                        ipOperationResultDto.getSuccessList().add(x);

                        failedList.remove(x);
                    }
            );
        }

        return ipOperationResultDto;
    }

    @Override
    public IpOperationResultDto deleteExpiredIp(String username, IpResourceRequestDto ipResourceRequestDto) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders headers = buildGetHeader();
        Map<String, String> params = new HashMap<String, String>();
        HttpEntity<Map<String, String>> httpEntity = new HttpEntity<>(params, headers);

        IpOperationResultDto ipOperationResultDto = new IpOperationResultDto();

        if (ipResourceRequestDto.getAddr() != null && ipResourceRequestDto.getAddr().size() > 0) {
            ipResourceRequestDto.getAddr().forEach(
                    ipAddr -> {
                        // 1. 查找需要删除的 ip 资源 是否被占用、 是否是当前用户的，若占用或不是当前用户的，则删除失败
                        IpResource ipResource = ipResourceRepository.findByAddrAndIsDeletedAndIsLocked(ipAddr, false, false);
                        if (ipResource == null) {
                            ipOperationResultDto.getFailList().add(ipAddr);
                            // 在 foreach 中 return 的作用 相当于 普通 for循环中 continue 的作用
                            return;
                        }


                        try {
                            // 调用 uri删除远程的 IP资源
                            ResponseEntity<String> result = restTemplate.exchange(URL + "/intelligroup/ipresources?accountId=browser&ip={ip}", HttpMethod.DELETE, httpEntity, String.class, ipAddr);
                            DeleteIpResultDto deleteIpResultDto = JSON.parseObject(result.getBody(), DeleteIpResultDto.class);
                            if (deleteIpResultDto.getResult().equals("failed") || StringUtils.isNotBlank(deleteIpResultDto.getErrorCode())) {
                                NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 删除失败", NotifyUtils.MsgType.WEBHOOK);
                                logger.error("ip " + ipResource.getAddr() + " 删除失败");
                                logger.error("ErrorCode: " + deleteIpResultDto.getErrorCode() + " " + deleteIpResultDto.getResult());
                            }

                        } catch (Exception e) {
                            logger.error("fail to delete ip", e);
                            ipOperationResultDto.getFailList().add(ipAddr);
                        }

                        try {
                            // ip资源为未绑定，则直接删除
                            if (ipResource.getStatus() == 6) {
                                ipResourceRepository.delete(ipResource);

                                // 其他情况（主要是 ip资源还未创建成功的状态），则将 ip资源的状态改为删除
                            } else {
                                ipResource.setShopIds(null);
                                ipResource.setDeleted(true);
                                ipResourceRepository.save(ipResource);
                            }
                            ipOperationResultDto.getSuccessList().add(ipAddr);
                        } catch (Exception e) {
                            logger.error("fail to delete ip", e);
                            ipOperationResultDto.getFailList().add(ipAddr);
                            return;
                        }
                    }
            );
        }
        if (ipResourceRequestDto.getIpId() != null && ipResourceRequestDto.getIpId().size() > 0) {
            ipResourceRequestDto.getIpId().forEach(
                    ipId -> {
                        IpResource ipResource = ipResourceRepository.findByIdAndIsDeletedAndIsLocked(ipId, false, false);
                        if (ipResource == null) {
                            ipOperationResultDto.getFailList().add(ipId);
                            return;
                        }
                        if (ipResource.getStatus() == 6) {
                            ipResourceRepository.delete(ipResource);
                        } else if (StringUtils.isNotBlank(ipResource.getAddr()) && ipResource.getIpType() == IpType.VENDOR) {
                            try {
                                ResponseEntity<String> result = restTemplate.exchange(URL + "/intelligroup/ipresources?accountId=browser&ip={ip}",
                                        HttpMethod.DELETE, httpEntity, String.class, ipResource.getAddr());
                                DeleteIpResultDto deleteIpResultDto = JSON.parseObject(result.getBody(), DeleteIpResultDto.class);
                                if (deleteIpResultDto.getResult().equals("failed") || StringUtils.isNotBlank(deleteIpResultDto.getErrorCode())) {
                                    NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 删除失败", NotifyUtils.MsgType.WEBHOOK);
                                    logger.error("ip " + ipResource.getAddr() + " 删除失败");
                                }
                            } catch (Exception t) {
                                logger.error("ip " + ipResource.getAddr() + " 删除失败", t);
                            }
                        }


                        ipResource.setDeleted(true);
                        ipResource.setShopIds(null);
                        ipResourceRepository.save(ipResource);
                        ipOperationResultDto.getSuccessList().add(ipId);
                    }
            );
        }

        return ipOperationResultDto;

    }

    @Override
    public IpOperationResultDto deleteIp(String username, IpResourceRequestDto ipResourceRequestDto) {

        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders headers = buildGetHeader();
        Map<String, String> params = new HashMap<String, String>();
        HttpEntity<Map<String, String>> httpEntity = new HttpEntity<>(params, headers);

        IpOperationResultDto ipOperationResultDto = new IpOperationResultDto();

        if (ipResourceRequestDto.getAddr() != null && ipResourceRequestDto.getAddr().size() > 0) {
            ipResourceRequestDto.getAddr().forEach(
                    ipAddr -> {
                        // 1. 查找需要删除的 ip 资源 是否被占用、 是否是当前用户的，若占用或不是当前用户的，则删除失败
                        IpResource ipResource = ipResourceRepository.findByAddrAndIsDeletedAndIsLocked(ipAddr, false, false);
                        if (ipResource == null) {
                            ipOperationResultDto.getFailList().add(ipAddr);
                            // 在 foreach 中 return 的作用 相当于 普通 for循环中 continue 的作用
                            return;
                        }

                        // 2. 如果根据 shopId 与 用户名 查不到 用户的商铺，则删除失败
                        if (ipResource.getShopIds() != null) {
                            List<Shop> shops = shopRepository.findByShopIdIn(ipResource.getShopIds());
                            for (Shop shop : shops) {
                                UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shop.getShopId());
                                if (userShop == null) {
                                    ipOperationResultDto.getFailList().add(ipAddr);
                                    return;
                                }
                            }
                        }

                        try {
                            // 调用 uri删除远程的 IP资源
                            ResponseEntity<String> result = restTemplate.exchange(URL + "/intelligroup/ipresources?accountId=browser&ip={ip}", HttpMethod.DELETE, httpEntity, String.class, ipAddr);
                            DeleteIpResultDto deleteIpResultDto = JSON.parseObject(result.getBody(), DeleteIpResultDto.class);
                            if (deleteIpResultDto.getResult().equals("failed") || StringUtils.isNotBlank(deleteIpResultDto.getErrorCode())) {
                                NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 删除失败", NotifyUtils.MsgType.WEBHOOK);
                                logger.error("ip " + ipResource.getAddr() + " 删除失败");
                                logger.error("ErrorCode: " + deleteIpResultDto.getErrorCode() + " " + deleteIpResultDto.getResult());
                            }

                        } catch (Exception e) {
                            logger.error("fail to delete ip", e);
                            ipOperationResultDto.getFailList().add(ipAddr);
                        }

                        try {
                            // ip资源为未绑定，则直接删除
                            if (ipResource.getStatus() == 6) {
                                ipResourceRepository.delete(ipResource);

                                // 其他情况（主要是 ip资源还未创建成功的状态），则将 ip资源的状态改为删除
                            } else {
                                ipResource.setShopIds(null);
                                ipResource.setDeleted(true);
                                ipResourceRepository.save(ipResource);
                            }
                            ipOperationResultDto.getSuccessList().add(ipAddr);
                        } catch (Exception e) {
                            logger.error("fail to delete ip", e);
                            ipOperationResultDto.getFailList().add(ipAddr);
                            return;
                        }
                    }
            );
        }
        if (ipResourceRequestDto.getIpId() != null && ipResourceRequestDto.getIpId().size() > 0) {
            ipResourceRequestDto.getIpId().forEach(
                    ipId -> {
                        IpResource ipResource = ipResourceRepository.findByIdAndIsDeletedAndIsLocked(ipId, false, false);
                        if (ipResource == null) {
                            ipOperationResultDto.getFailList().add(ipId);
                            return;
                        }
                        if (ipResource.getStatus() == 6) {
                            ipResourceRepository.delete(ipResource);
                        } else if (StringUtils.isNotBlank(ipResource.getAddr()) && ipResource.getIpType() == IpType.VENDOR) {
                            try {
                                ResponseEntity<String> result = restTemplate.exchange(URL + "/intelligroup/ipresources?accountId=browser&ip={ip}",
                                        HttpMethod.DELETE, httpEntity, String.class, ipResource.getAddr());
                                DeleteIpResultDto deleteIpResultDto = JSON.parseObject(result.getBody(), DeleteIpResultDto.class);
                                if (deleteIpResultDto.getResult().equals("failed") || StringUtils.isNotBlank(deleteIpResultDto.getErrorCode())) {
                                    NotifyUtils.sendMessage("防关联浏览器 ip " + ipResource.getAddr() + " 删除失败", NotifyUtils.MsgType.WEBHOOK);
                                    logger.error("ip " + ipResource.getAddr() + " 删除失败");
                                }
                            } catch (Exception t) {
                                logger.error("ip " + ipResource.getAddr() + " 删除失败", t);
                            }
                        }

                        if (ipResource.getShopIds() != null) {
                            List<Shop> shops = shopRepository.findByShopIdIn(ipResource.getShopIds());
                            if (shops != null && shops.size() > 0) {
                                for (Shop shop : shops) {
                                    UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shop.getShopId());
                                    if (userShop == null) {
                                        ipOperationResultDto.getFailList().add(ipId);
                                        return;
                                    }
                                }
                            }
                        }
                        ipResource.setDeleted(true);
                        ipResource.setShopIds(null);
                        ipResourceRepository.save(ipResource);
                        ipOperationResultDto.getSuccessList().add(ipId);
                    }
            );
        }

        return ipOperationResultDto;
    }

    @Override
    public IpPageResultDto getIpList(String username, int groupType, int page, int amount, IpFilterDto ipFilterDto) {
        // 获取当前用户的账户
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        // 获取当前用户对应的 groupType 的 ip 资及未使用的 ip资源
        List<IpResource> ipResources = new ArrayList<>();
        List<IpResource> notUsed = new ArrayList<>();
        dealIpResourceByGroup(ipResources, notUsed, account, username, groupType);

        if (CollectionUtils.isNotEmpty(notUsed)) {
            ipResources.addAll(notUsed);
        }

        // 去掉重复 ip
        List<String> allIpIds = ipResources.stream().distinct().map(IpResource::getId).collect(Collectors.toList());


        // 根据过滤条件来分页获取 ip 资源
        amount = amount > 100 ? 100 : amount;
        Pageable pageable = PageRequest.of(page, amount);
        Page<IpResource> ipResources1 = getIpResourcesByFilter(ipFilterDto, allIpIds, pageable);

        List<IpResourceDto> ipResourceDtos = new ArrayList<>();
        if (ipResources1 != null) {
            ipResources1.getContent().forEach(
                    x -> {
                        // 1. 如果 ip 资源的 addr 非空
                        if (StringUtils.isNotBlank(x.getAddr())) {
                            // 1.1 锁定时间在 2分钟内 且 IpType 为 VENDOR 且 （IP资源状态 为 0：正常 或 2：即将过期）
                            if (x.getLockTimestamp() >= Instant.now().minusSeconds(2 * 60).toEpochMilli() && x.getIpType() == IpType.VENDOR && (x.getStatus() == 0 || x.getStatus() == 2)) {
                                x.setStatus(3);

                                // 1.2 ip资源在未来七天内到期,且 ip 资源的状态不是 3（正在分配）、5（已失效）、6（未分配），则将 ip 资源设置为 2（即将过期）
                            } else if (x.getValidTime() <= Instant.now().plusSeconds(60 * 60 * 24 * 7).toEpochMilli() && x.getValidTime() > Instant.now().toEpochMilli()) {
                                if (x.getStatus() != 5 && x.getStatus() != 3 && x.getStatus() != 6) {
                                    ipResourceRepository.updateStatus(x.getId(), 2);
                                }

                                // 1.3 ip资源在七天前到期，且 ip 资源的状态不是 3（正在分配）、6（未分配），则删除 ip 资源
                            } else if (x.getValidTime() <= Instant.now().minusSeconds(60 * 60 * 24 * 7).toEpochMilli() && x.getStatus() != 3 && x.getStatus() != 6) {
                                if (x.getIpType() == IpType.VENDOR) {
                                    IpResourceRequestDto ipResourceRequestDto = new IpResourceRequestDto();
                                    ipResourceRequestDto.setAddr(Arrays.asList(x.getAddr()));
                                    deleteExpiredIp(username, ipResourceRequestDto);
                                } else {
                                    IpResourceRequestDto ipResourceRequestDto = new IpResourceRequestDto();
                                    ipResourceRequestDto.setIpId(Arrays.asList(x.getId()));
                                    deleteExpiredIp(username, ipResourceRequestDto);
                                }
                                return;

                                // 1.4 ip资源到期，且 ip 资源的状态不是 3（正在分配）、6（未分配），则设置 ip 资源的状态为 1（已过期）
                            } else if (x.getValidTime() <= Instant.now().toEpochMilli() && x.getStatus() != 3 && x.getStatus() != 6) {
                                ipResourceRepository.updateStatus(x.getId(), 1);

                                // 1.5 其他
                            } else {
                                //
                                if ((x.getStatus() == 0 || x.getStatus() == 1 || x.getStatus() == 2) && !x.getVendor().equals(Vendor.local)) {
                                    ipResourceRepository.updateStatus(x.getId(), 0);
                                } else if (x.getIpType().equals(IpType.LOCAL) && (x.getStatus() == 1 || x.getStatus() == 2) && x.getAddr().equals("本地Ip未使用")) {
                                    ipResourceRepository.updateStatus(x.getId(), 4);
                                }
                            }
                        }

                        // 2. 如果 ip 资源的 status 为 6（未分配） 或者  （ ip资源为专线 且 purchasedTime（购买IP的时间） 在 14 分钟内）
                        if (x.getStatus() == 6 || (x.isSpecialLine() && x.getPurchasedTime() > (Instant.now().minusSeconds(14 * 60).toEpochMilli()))) {
                            x.setStatus(3);
                        }

                        // 3. 如果 ip 资源的 status 为 3（正在分配）
                        if (x.getStatus() == 3) {
                            x.setAddr("");
                        }

                        List<ShopDto> shopDtos = getShopDtos(x);
                        SpecialLine specialLine = specialLineRepository.findAll().get(0);
                        IpResourceDto ipResourceDto = x.isSpecialLine() ? new IpResourceDto(x, shopDtos, false, specialLine) : new IpResourceDto(x, shopDtos, false);
                        ipResourceDtos.add(ipResourceDto);
                    }
            );
        }

        // 将当前页的数据封装到 ipPageResultDto 中
        IpPageResultDto ipPageResultDto = new IpPageResultDto();

        int totalSize = allIpIds.size();
        if (ipResourceDtos == null || ipResourceDtos.isEmpty()) {
            totalSize = 0;
        }

        Page<IpResourceDto> ipResourceDtoPage = new PageImpl<>(ipResourceDtos, pageable, totalSize);

        ipPageResultDto.setIpList(ipResourceDtoPage.getContent());
        PageInfo pageInfo = new PageInfo(ipResourceDtoPage.getPageable().getPageNumber(), ipResourceDtoPage.getTotalPages(), totalSize);
        ipPageResultDto.setIpPage(pageInfo);

        return ipPageResultDto;
    }

    @Override
    public void setIpOptions() {
        String path = ClassLoader.getSystemResource("ipOptions.json").getPath();
        IpOptions jsonObject = JSONObject.parseObject(FileUtil.read(new File(path), "UTF-8"), IpOptions.class);
        ipOptionsRepository.save(jsonObject);
    }

    @Override
    public IpOptions getIpOptions() {
        try {
            IpOptions jsonObject = ipOptionsRepository.findAll().get(0);
            return jsonObject;
        } catch (Exception e) {
            logger.error(e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public List<PlatformOptions> getPlatformOptions() {
        try {
            List<PlatformOptions> platformOptions = platformOptionsRepository.findAll();
            return platformOptions;
        } catch (Exception e) {
            logger.error(e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public IpSummary getIpSummary(String username) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));


        IpSummary ipSummary = new IpSummary();
        List<String> shopIds = userShopRepository.findByUsername(username).stream().map(x -> x.getShopId()).collect(Collectors.toList());
        List<String> used = ipResourceRepository.findShopIdInList(shopIds, false).stream().map(x -> x.getId()).collect(Collectors.toList());
        List<String> unbind = ipResourceRepository.findByOwnerAndIsDeletedAndBind(username, false, false).stream().map(x -> x.getId()).collect(Collectors.toList());
        ipSummary.setUnbind(unbind.size());
        unbind.addAll(used);
        int expired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(1, unbind, false);
        ipSummary.setExpired(expired);
        int willexpired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(2, unbind, false);
        ipSummary.setWillExpire(willexpired);
        ipSummary.setTotal(unbind.size());
        return ipSummary;
    }

    @Override
    public void updateIp(String username, IpResourceUpdateDto ipResourceUpdateDto) {
        // 1. 根据id 来获取账户
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));

        // 只有管理员可以 更新 ip，管理员的权限是 8
        if (account.getPermission() < 8) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }

        // 2.根据传入的id查询是否有指定的 isDelete 为 false的 ip 资源
        IpResource ipResource = ipResourceRepository.findByIdAndIsDeleted(ipResourceUpdateDto.getIpId(), false);
        if (ipResource == null) {
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        }

        // 封装 ipResource 部分信息
        dealIpResource(ipResourceUpdateDto, ipResource);

        // 3.更新ip资源
        ipResourceRepository.save(ipResource);
    }

    @Override
    public boolean queryIpExist(String username, IpResourceUpdateDto ipResourceUpdateDto) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        IpResource ipResource = ipResourceRepository.findByAddrAndIsDeleted(ipResourceUpdateDto.getAddr(), false);
        if (ipResource != null) {
            return true;
        }
        return false;
    }

    @Override
    public IpResourceDto queryIp(String username, IpResourceRequestDto ipResourceRequestDto) {
        Account account = accountRepository.findByName(username).orElseThrow(() -> new ClientRequestException(AccountErrorCode.NAMENOTEXIST));
        IpResource ipResource = null;
        if (ipResourceRequestDto.getAddr() != null && ipResourceRequestDto.getAddr().size() > 0)
            ipResource = ipResourceRepository.findByAddrAndIsDeleted(ipResourceRequestDto.getAddr().get(0), false);
        else if (ipResourceRequestDto.getIpId() != null && ipResourceRequestDto.getIpId().size() > 0)
            ipResource = ipResourceRepository.findByIdAndIsDeleted(ipResourceRequestDto.getIpId().get(0), false);
        else {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }

        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);

        List<ShopDto> shopDtos = new ArrayList<>();
        List<Shop> shops = null;
        if (ipResource.getShopIds() != null)
            shops = shopRepository.findByShopIdIn(ipResource.getShopIds());
        if (shops == null || shops.size() > 0) {
            shopDtos.add(new ShopDto());
        } else {
            for (Shop shop : shops) {
                shopDtos.add(new ShopDto(shop));
            }
        }
        if (StringUtils.isNotBlank(ipResource.getAddr())) {
            if (ipResource.getValidTime() <= Instant.now().plusSeconds(60 * 60 * 24 * 7).toEpochMilli() && ipResource.getValidTime() > Instant.now().toEpochMilli()) {
                ipResourceRepository.updateStatus(ipResource.getId(), 2);
            } else if (ipResource.getValidTime() <= Instant.now().minusSeconds(60 * 60 * 24 * 7).toEpochMilli()) {
                deleteExpiredIp(username, ipResourceRequestDto);
            } else if (ipResource.getValidTime() <= Instant.now().toEpochMilli()) {
                ipResourceRepository.updateStatus(ipResource.getId(), 1);
            } else {
                if (ipResource.getStatus() != 4 && ipResource.getStatus() != 5) {
                    ipResourceRepository.updateStatus(ipResource.getId(), 0);
                }
            }
        }
        IpResourceDto ipResourceDto = new IpResourceDto(ipResource, shopDtos, true);

        return ipResourceDto;
    }

    @Override
    public void setSpecialLine() {
        List<IpResource> ipResources = ipResourceRepository.findByRegionInAndIsDeleted(region, false);
        for (IpResource ipResource : ipResources) {
            ipResource.setProxyUsername(ipResource.getAddr());
            ipResource.setProxyPassword(genRandom(3, 12));
            ipResource.setSpecialLine(true);
            ipResourceRepository.save(ipResource);
        }
    }

    @Override
    public void releaseDeletedIp() {
        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        RestTemplate restTemplate = new RestTemplate();
        HttpHeaders headers = buildGetHeader();
        Map<String, String> params = new HashMap<String, String>();
        HttpEntity<Map<String, String>> httpEntity = new HttpEntity<>(params, headers);
        List<IpResource> ipResources = ipResourceRepository.findByIsDeleted(true);
        ipResources.forEach(x -> {
            ResponseEntity<String> result = restTemplate.exchange(URL + "/intelligroup/ipresources?accountId=browser&ip={ip}", HttpMethod.DELETE, httpEntity, String.class, x.getAddr());
            DeleteIpResultDto deleteIpResultDto = JSON.parseObject(result.getBody(), DeleteIpResultDto.class);
        });
    }

    @Override
    public void transferBindShops() {
        List<IpResource> ipResources = ipResourceRepository.findByIsDeleted(false);
        for (IpResource ipResource : ipResources) {
            ipResourceRepository.addShopId(ipResource.getId(), ipResource.getShopId());
            ipResourceRepository.updateBind(ipResource.getId(), true);
        }
    }


    private IpChargeRequestDto buildIpChargeRequestDto(IpResourceRequestDto request, int chargeType, int payMethod) {
        IpChargeRequestDto ipChargeRequestDto = new IpChargeRequestDto();
        ipChargeRequestDto.setAmount(request.getAmount());
        ipChargeRequestDto.setRegion(request.getRegion());
        ipChargeRequestDto.setPeriod(request.getPeriod());
        ipChargeRequestDto.setUnit(request.getUnit());
        ipChargeRequestDto.setChargeType(chargeType);
        ipChargeRequestDto.setPayMethod(payMethod);
        return ipChargeRequestDto;
    }

    private void dealIpResource(IpResourceUpdateDto ipResourceUpdateDto, IpResource ipResource) {
        if (StringUtils.isNotBlank(ipResourceUpdateDto.getAddr())) {
            ipResource.setAddr(ipResourceUpdateDto.getAddr());
        }

        if (StringUtils.isNotBlank(ipResource.getRegion()) && region.contains(ipResource.getRegion())) {
            ipResource.setProxyUsername(ipResource.getAddr());
            ipResource.setProxyPassword(genRandom(3, 12));
            ipResource.setSpecialLine(true);
        }
        ipResource.setStatus(ipResourceUpdateDto.getStatus());
    }

    private void dealIpResourceByGroup(List<IpResource> ipResourceList, List<IpResource> notUsedList, Account account, String username, int groupType) {
        // 获取当前用户所有的商铺id
        List<String> shopIds = userShopRepository.findByUsername(username).stream().map(UserShop::getShopId).collect(Collectors.toList());

        List<String> accountNameList = new ArrayList<>();
        accountNameList.add(account.getName());
        boolean isParent = account.getParent() == null;
        if (isParent) {
            // 获取当前用户的 子账户名
            List<String> children = accountRepository.findByParent(account.getName()).stream().map(Account::getName).collect(Collectors.toList());
            accountNameList.addAll(children);
        }


        List<IpResource> ipResources = null;
        List<IpResource> notUsed = null;
        switch (groupType) {
            case 1:
                ipResources = ipResourceRepository.findShopIdInList(shopIds, false);
                if (!isParent) {
                    notUsed = ipResourceRepository.findByOwnerAndIsDeletedAndBind(username, false, false);
                } else {
                    notUsed = ipResourceRepository.findByOwnerInAndIsDeletedAndBind(accountNameList, false, false);
                }
                break;
            case 2:
                ipResources = ipResourceRepository.findShopIdInListAndStatus(shopIds, false, 2);
                if (!isParent) {
                    notUsed = ipResourceRepository.findByOwnerAndStatusAndIsDeletedAndBind(username, 2, false, false);
                } else {
                    notUsed = ipResourceRepository.findByOwnerInAndStatusAndIsDeletedAndBind(accountNameList, 2, false, false);
                }
                break;
            case 3:
                ipResources = ipResourceRepository.findShopIdInListAndStatus(shopIds, false, 1);
                if (!isParent) {
                    notUsed = ipResourceRepository.findByOwnerAndStatusAndIsDeletedAndBind(username, 1, false, false);
                } else {
                    notUsed = ipResourceRepository.findByOwnerInAndStatusAndIsDeletedAndBind(accountNameList, 1, false, false);
                }
                break;
            case 4:
                if (!isParent) {
                    notUsed = ipResourceRepository.findByOwnerAndStatusIsNotInAndIsDeletedAndBind(username, Arrays.asList(3, 5, 6), false, false);
                } else {
                    notUsed = ipResourceRepository.findByOwnerInAndStatusIsNotInAndIsDeletedAndBind(accountNameList, Arrays.asList(3, 5, 6), false, false);
                }
                break;
            case 5:
                // 已分配
                ipResources = ipResourceRepository.findShopIdInList(shopIds, false);
                if (!isParent)
                    notUsed = ipResourceRepository.findByOwnerAndStatusInAndIsDeletedAndBind(username, Arrays.asList(0, 2, 4, 8), false, false);
                else
                    notUsed = ipResourceRepository.findByOwnerInAndStatusInAndIsDeletedAndBind(accountNameList, Arrays.asList(0, 2, 4, 8), false, false);
                break;
        }
        if (CollectionUtils.isNotEmpty(ipResources))
            ipResourceList.addAll(ipResources);
        if (CollectionUtils.isNotEmpty(notUsed))
            notUsedList.addAll(notUsed);
    }

    @NotNull
    private List<ShopDto> getShopDtos(IpResource x) {
        List<ShopDto> shopDtos = new ArrayList<>();
        List<Shop> shops = new ArrayList<>();
        if (x.getShopIds() != null) {
            shops = shopRepository.findByShopIdIn(x.getShopIds());
        }

        if (shops.isEmpty()) {
            shopDtos.add(new ShopDto());
        } else {
            for (Shop shop : shops) {
                shopDtos.add(new ShopDto(shop));
            }
        }
        return shopDtos;
    }

    private Page<IpResource> getIpResourcesByFilter(IpFilterDto ipFilterDto, List<String> allIpIds, Pageable pageable) {
        Page<IpResource> ipResources1 = null;
        if (StringUtils.isNotBlank(ipFilterDto.getRegion())) {
            ipResources1 = ipResourceRepository.findByIsDeletedAndIdInAndRegionCnLikeOrderByPurchasedTimeDesc(false, allIpIds, ipFilterDto.getRegion(), pageable);
        }
        if (StringUtils.isNotBlank(ipFilterDto.getAddr())) {
            ipResources1 = ipResourceRepository.findByIsDeletedAndIdInAndAddrLikeOrderByPurchasedTimeDesc(false, allIpIds, ipFilterDto.getAddr(), pageable);
        }
        if (StringUtils.isNotBlank(ipFilterDto.getVendor())) {
            ipResources1 = ipResourceRepository.findByIsDeletedAndIdInAndVendorCnLikeOrderByPurchasedTimeDesc(false, allIpIds, ipFilterDto.getVendor(), pageable);
        }
        if (ipFilterDto.isEmpty()) {
            ipResources1 = ipResourceRepository.findByIdInAndIsDeletedOrderByPurchasedTimeDesc(allIpIds, false, pageable);
        }
        return ipResources1;
    }

    private int setAndGetPeriod(String unit, int period) {
        if (unit.equals("month")) {
            if (period == 6) {
                period = 7;
            }
            if (period == 12) {
                period = 14;
            }
        }
        return period;
    }

    @NotNull
    private AtomicReference<Double> getDoubleAtomicReference(IpResourceRequestDto ipResourceRequestDto, Map<String, List<String>> ipPlatForm) {
        AtomicReference<Double> totalprice = new AtomicReference<>((double) 0);
        ipResourceRequestDto.getAddr().stream()
                            .map(x -> ipResourceRepository.findByAddrAndIsDeleted(x, false))
                            .forEach(
                                    ipResource -> {
                                        List<String> regionAndPriceInfos;
                                        if (!ipResourceRequestDto.getVendor().equals("own")) {
                                            regionAndPriceInfos = ipPlatForm.get(ipResource.getRegionCn());
                                        } else {
                                            regionAndPriceInfos = ipResource.isSpecialLine() ? ipPlatForm.get("自有专线") : ipPlatForm.get("自有");
                                        }

                                        for (String vendorprice : regionAndPriceInfos) {
                                            if (ipResource.getVendor().getValue().equals(vendorprice.substring(0, vendorprice.indexOf("-"))))
                                                totalprice.updateAndGet(
                                                        v -> new Double(v + Double.valueOf(vendorprice.substring(vendorprice.lastIndexOf("-") + 1)))
                                                );
                                        }
                                    }
                            );
        return totalprice;
    }

    @Nullable
    private RenewIpResultDto getRenewIpResultDto(IpResourceRequestDto ipResourceRequestDto, int period) {
        RestTemplate restTemplate = new RestTemplate();

        HashMap<String, Object> map = new HashMap<>();
        map.put("iplist", ipResourceRequestDto.getAddr());
        map.put("period", period);
        map.put("unit", ipResourceRequestDto.getUnit());
        HttpHeaders headers = buildPostHeader();
        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(map, headers);

        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        ResponseEntity<String> result = restTemplate.exchange(URL + "/intelligroup/renewip?accountId=browser", HttpMethod.PUT, entity, String.class);
        return JSON.parseObject(result.getBody(), RenewIpResultDto.class);
    }
}
