package com.edgec.browserbackend.account.controller;


import com.edgec.browserbackend.account.service.*;
import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.common.auth.Securitys;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;


@RestController
public class AdministratorController {

    @Autowired
    private AdministratorService administratorService;
    @Autowired
    private UserRateService userRateService;

    @Autowired
    private MonthReportService monthReportService;

    @Autowired
    private PaymentService paymentService;

    @Autowired
    private UserLackMoneyService userLackMoneyService;

    @Autowired
    private UserPrePaidBillingService userPrePaidBillingService;

    //--------------------------------ROOT用户操作--------------------------------------//
    //管理员注册tested
    @PreAuthorize(Securitys.ROOT_EL)
    @RequestMapping(path = "/0xadministrator",method = RequestMethod.POST)
    public Administrator createNewAdministrator(@Valid @RequestBody Administrator administrator){
        return administratorService.createAdministrator(administrator);
    }

    //删除管理员testing
    @PreAuthorize(Securitys.ROOT_EL)
    @RequestMapping(path = "/0xadministrator/{name}", method = RequestMethod.DELETE)
    public void deleteAdmin( @PathVariable String name) {
        administratorService.deleteAdministrator(name);
    }


    //  查询所有的管理员
    @PreAuthorize(Securitys.ROOT_EL)
    @RequestMapping(path = "/0xadministrator",method = RequestMethod.GET)
    public List<Administrator> getALLAdministrator(){
        return administratorService.getAllAdministrator();
    }


    //修改管理员权限tested
    @PreAuthorize(Securitys.ROOT_EL)
    @RequestMapping(path = "/0xadministrator/{name}", method = RequestMethod.PUT)
    public Administrator updateAgencyInfo(@PathVariable String name, @Valid @RequestBody String string){
        return administratorService.updateAdministrator(name,string);
    }

    //用户通过银行转账，手动插入余额记录
    @PreAuthorize(Securitys.ROOT_EL)
    @RequestMapping(path = "/0xadministrator/banktransfer", method = RequestMethod.POST)
    public UserBalance bankTransferInsert(@RequestBody UserBalance userBalance) {
        return paymentService.bankTransferInsertion(userBalance.getUsername(), (int) userBalance.getBalanced());
    }

    //--------------------------------用户--------------------------------------//
    //根据用户的性名，或者电话，或者邮件，查询用户的信息tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/searchuser/{target}",method = RequestMethod.GET)
    public Account getAccountByTarget(@RequestParam("by") String by, @PathVariable String target) {
        Account searchAccount = new Account();

        if("name".equals(by)){
            searchAccount  =  administratorService.getAccountByName(target);
        }
        else if("phoneNumber".equals(by)){
            searchAccount = administratorService.getAccountByPhoneNumber(target);
        }
        else if("email".equals(by)){
            searchAccount = administratorService.getAccountByEmail(target);
        }

        return searchAccount;

    }

    //根据用户名查询用户账单tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/searchuserbill/{name}",method = RequestMethod.GET)
    public List<UserPrePaidBilling> getUserBillByName(@PathVariable String name){
        return administratorService.getUserBillingByName(name);
    }



    //给用户激活未激活或者被锁住的账户tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/unlockaccount/{name}",method = RequestMethod.PUT)
    public Account unLockAccount(@PathVariable String name , @Valid @RequestBody Account account){
        return administratorService.unLockLockedAccount(name,account);
    }


    //锁住异常用户tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/lockaccount/{name}",method = RequestMethod.PUT)
    public Account lockAccount(@PathVariable String name , @Valid @RequestBody Account account){
        return administratorService.lockAbnormalAccount(name,account);
    }

    //查询用户余额
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/finduserbalance", method = RequestMethod.GET)
    public UserBalance findUserBlanceByName(@RequestParam(value = "username") String username) {
        return administratorService.findUserBalance(username);
    }

    // 查询所有已欠费用户
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/findlackmoney", method = RequestMethod.GET)
    public List<UserLackMoney> findAllUserLackMoney() {
        return userLackMoneyService.findAll();
    }

    // 手动刷新用户未支付订单，欠费状态
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/refreshbilling", method = RequestMethod.POST)
    public void refreshUserBillingAndCost(@RequestParam(value = "username") String username) {
        userPrePaidBillingService.refreshUserBillingStatus(username);
    }

    // 查询用户当前所有欠费账单
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/unpaidbilling/{username}", method = RequestMethod.GET)
    public List<UserPrePaidBilling> findUnpaidUserBilling(@PathVariable String username) {
        return userPrePaidBillingService.findUserBillingByUnpaid(username, BillStatus.UNPAID);
    }

    //--------------------------------系统--------------------------------------//
    //查询某时段内用户的账单，tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path = "/0xadministrator/searchusers", method = RequestMethod.GET)
    public Page<UserPrePaidBilling> searchAllUserBilling(@RequestParam(value = "page") int page, @RequestParam(value = "year1") int year1, @RequestParam(value = "month1") int month1, @RequestParam(value = "year2") int year2, @RequestParam(value = "month2") int month2) {
        return administratorService.searchAllUserBillingPage(page, year1, year2, month1, month2);
    }

    //查看所有的用户<分页查询> tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/searchalluser",method = RequestMethod.GET)
    public Page<Account> searchAllUser(@RequestParam(value="page")int page , @RequestParam(value="size")int size ){//(@PathVariable int  page){
        Pageable pageable = PageRequest.of(page,size);
        return administratorService.searchAllUserPage(pageable);
    }

    //查询某时段内增加的用户
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/searchperioduser",method = RequestMethod.GET)
    public Page<Account> searchCreateAccount(@RequestParam(value="page") int page, @RequestParam(value="size") int size, @RequestParam(value="strDate1") String  strDate1, @RequestParam(value="strDate2")  String  strDate2 ){
        Pageable pageable = PageRequest.of(page, size);
        return administratorService.searchCreateAccountBetween(pageable,strDate1,strDate2);
    }

    //search company earnings detail by month  tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path ="/0xadministrator/companyearning/{stringdate}",method = RequestMethod.GET)
    public List<CompanyEarningsDto> getCompanyEarning(@PathVariable String stringdate ){
        return administratorService.getCompanyEarningDetail(stringdate);
    }


    //search information of service-fee owed users   tested
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path = "/0xadministrator/oweduser", method = RequestMethod.GET)
    public List<UserPrePaidBilling> getOwedUser() {
        return administratorService.getServiceFeeOwedUserInfo();
    }


    //查找某个用户或者代理有没有被锁住
    @PreAuthorize("hasRole('ADMIN')")
    @RequestMapping(path = "/0xadministrator/lockstate/{name}",method = RequestMethod.GET)
    public boolean getLockState(@PathVariable String name){
        return administratorService.getUserLockState(name);
    }



    //对欠费用户强制停机


    /**  用户费率  */

    // 设置费率
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/updateuserrate/{id}/{money}", method = RequestMethod.PUT)
    public boolean setUserRate(@PathVariable String id, @PathVariable float money) {
        userRateService.updateUserRate(id, money);
        return true;
    }

    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/adduserrate", method = RequestMethod.POST)
    public void insertUserRate(@RequestBody UserRate userRate){
        userRateService.addUserRate(userRate);
    }

    // 查询用户费率
    @RequestMapping(path = "/0xadministrator/searchcurrentuserrate", method = RequestMethod.GET)
    public List<UserRate> getUserRate(@RequestParam String username) {
        return userRateService.getUserRate(username);
    }

    // 分页查询用户费率
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/searchalluserrate", method = RequestMethod.GET)
    public Page<UserRate> getUserRatePage(@RequestParam(value = "pageNo", defaultValue = "0") Integer pageNo,
                                          @RequestParam(value = "pageSize", defaultValue = "10") Integer pageSize) {
        PageRequest page = PageRequest.of(pageNo, pageSize);
        return userRateService.getUserRatePage(page);
    }

    // 删除用户费率
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/deleteuserrate/{id}", method = RequestMethod.DELETE)
    public boolean deleteUserRate(@PathVariable String id) {
        return userRateService.deleteUserRate(id);
    }


    // 注销用户
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/deleteUser", method = RequestMethod.DELETE)
    public void deleteUser(@RequestParam String username) {
        administratorService.deleteUser(username);
    }


    //每月预付费统计分页展示
    @PreAuthorize(Securitys.ADMIN_EL)
    @RequestMapping(path = "/0xadministrator/listReport", method = RequestMethod.GET)
    public Page<CompanyMonthReport> listMonthReport(@RequestParam(value="page", defaultValue = "0") int page,
                                                    @RequestParam(value="size", defaultValue = "10") int size){

        Pageable pageable = PageRequest.of(page,size);
        return monthReportService.listMonthReport(pageable);
    }

    // 迁移userbilling到userprepaidbilling
    @PreAuthorize(Securitys.ROOT_EL)
    @RequestMapping(path = "/0xadministrator/userbillingtransfer", method = RequestMethod.PUT)
    public boolean userbillingTransfer() {
        return administratorService.userbillingTransfer();
    }
}












