package com.edgec.browserbackend.browser.service.Impl;

import com.edgec.browserbackend.account.domain.Account;
import com.edgec.browserbackend.account.domain.User;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.AccountRepository;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.dto.ShopFilterDto;
import com.edgec.browserbackend.browser.dto.ShopResultDto;
import com.edgec.browserbackend.browser.dto.PageInfo;
import com.edgec.browserbackend.browser.dto.ShopPageResultDto;
import com.edgec.browserbackend.browser.repository.GroupRepository;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.browser.repository.ShopRepository;
import com.edgec.browserbackend.browser.repository.UserShopRepository;
import com.edgec.browserbackend.browser.service.ShopService;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Service
@Transactional
public class ShopServiceImpl implements ShopService {

    private final Logger logger = LoggerFactory.getLogger(ShopServiceImpl.class);

    @Autowired
    ShopRepository shopRepository;

    @Autowired
    AccountRepository accountRepository;

    @Autowired
    UserShopRepository userShopRepository;

    @Autowired
    GroupRepository groupRepository;

    @Autowired
    IpResourceRepository ipResourceRepository;

    @Override
    public String addShop(String username, ShopResultDto shopResultDto) {
        Account account = accountRepository.findByName(username);
        String id = null;
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (account.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        if (account.getShopCount() >= 10000) {
            throw new ClientRequestException(AccountErrorCode.SHOPMAX);
        }
        UserShop us = null;
        if (shopResultDto.getShopId() != null)
            us = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
        if (shopResultDto.getGroup() == null)
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        Group group = groupRepository.findById(shopResultDto.getGroup()).orElse(null);
        if (group == null) {
            throw new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST);
        }
        try {
            shopResultDto.setOwner(username);
            Shop shop = new Shop();
            shop.of(shopResultDto);
            id = shopRepository.save(shop).getShopId();
            UserShop userShop = new UserShop();
            userShop.setUsername(username);
            userShop.setShopId(id);
            if (shopResultDto.getGroup() != null && us == null) {
                userShop.setGroupId(shopResultDto.getGroup());
            }
            userShopRepository.save(userShop);
            //可以优化
            account.setShopCount(account.getShopCount() + 1);
            accountRepository.save(account);
        }catch (Exception e) {
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        return id;
    }

    @Override
    public List<String> addShops(String username, List<Shop> shops) {
        return null;
    }

    @Override
    public String updateShop(String username, ShopResultDto shopResultDto) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (shopResultDto == null || StringUtils.isBlank(shopResultDto.getShopId())) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopResultDto.getShopId());
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop_old = shopRepository.findById(shopResultDto.getShopId()).orElseGet(null);
        if (shop_old == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        try {
            shop_old = shop_old.of(shopResultDto);
            shopRepository.save(shop_old);
        }catch (Exception e) {
            logger.error("fail to update", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.INFORMATIONNOTCOMPELETE);
        }
        return shop_old.getShopId();
    }

    @Override
    public void deleteShop(String username, String shopId) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        if (shopId == null) {
            throw new ClientRequestException(AccountErrorCode.OTHERS);
        }
        logger.error(username, shopId);
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        if (shop.getIpId() != null && shop.getIp() != null) {
            unBindShop(username, shopId, shop.getIp());
        }
        boolean result = userShopRepository.deleteByUsernameAndShopId(username, shopId);
        if (result) {
            shopRepository.deleteById(shopId);
            account.setShopCount(account.getShopCount() - 1);
            accountRepository.save(account);
        } else throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
    }

    @Override
    public void bindShop(String username, String shopId, String ipAddr) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        if (shop.getIpId() != null && shop.getIp() != null) {
            unBindShop(username, shopId, shop.getIp());
        }
        IpResource ipResource = ipResourceRepository.findByAddrAndIsDeleted(ipAddr, false);
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        try {
            shop.setIp(ipAddr);
            shop.setIpId(ipResource.getId());
            shop.setIpRegion(ipResource.getRegion());
            shopRepository.save(shop);
            List<String> history = ipResource.getBindHistory();
            history.add(shop.getShopId());
            ipResource.setShopId(shopId);
            if (shop.getShopName()!=null)
                ipResource.setShopName(shop.getShopName());
            ipResourceRepository.save(ipResource);
            userShop.setIpId(ipResource.getId());
            userShopRepository.save(userShop);
        } catch (Exception e) {
            logger.error("fail to bind shop and ip", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void unBindShop(String username, String shopId, String ipAddr) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        if (account.getPermission() < 4 || userShop == null) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        IpResource ipResource = ipResourceRepository.findByAddrAndIsDeleted(ipAddr, false);
        if (ipResource == null)
            throw new ClientRequestException(BrowserErrorCode.IPNOTEXIST);
        if (!ipAddr.equals(shop.getIp()))
            throw new ClientRequestException(BrowserErrorCode.IPNOTBINDTOSHOP);
        try {
            shop.setIp(null);
            shop.setIpId(null);
            shop.setIpRegion(null);
            ipResource.setShopId(null);
            ipResource.setShopName(null);
            ipResourceRepository.save(ipResource);
            shopRepository.save(shop);
            userShop.setIpId(null);
            userShopRepository.save(userShop);
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }
    }

    @Override
    public void transferShop(String username, String shopId, String groupId) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        UserShop userShop = userShopRepository.findByUsernameAndShopId(username, shopId);
        Group group = groupRepository.findById(groupId).orElse(null);
        if (account.getPermission() < 4 || userShop == null || !group.getOwner().equals(username)) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Shop shop = shopRepository.findById(shopId).orElse(null);
        if (shop == null)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        try {
            userShop.setGroupId(groupId);
            userShopRepository.save(userShop);
        } catch (Exception e) {
            logger.error("fail to unbind", e.getMessage());
            throw new ClientRequestException(BrowserErrorCode.UNKNOWN);
        }

    }

    @Override
    public void assignShops(String username, List<String> shopIds, List<String> users) {
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        List<UserShop> userShops = userShopRepository.findByUsernameAndShopIdIn(username, shopIds);
        if (account.getPermission() < 8 || userShops == null || userShops.size() < 1) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        Pageable pageable = PageRequest.of(0, 100);
        List<Shop> shops = shopRepository.findByShopIdIn(shopIds, pageable).getContent();
        if (shops == null || shops.size() < 1)
            throw new ClientRequestException(BrowserErrorCode.SHOPNOTEXIST);
        List<Account> accounts = accountRepository.findByNameIn(users);
        if (accounts == null || accounts.size() != users.size())
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        shops.stream().forEach(shop -> {
            try {
                userShopRepository.deleteByShopIdExceptOwner(shop.getShopId(), shop.getOwner());
                for (Account account1 : accounts) {
                    UserShop userShop1 = userShopRepository.findByUsernameAndShopId(account1.getName(), shop.getShopId());
                    if (userShop1 != null)
                        return;
                    userShop1 = new UserShop(account1.getName(), shop.getShopId(), "-1");
                    if (shop.getIpId() != null)
                        userShop1.setIpId(shop.getIpId());
                    userShopRepository.save(userShop1);
                }
            } catch (Exception e) {
                logger.error("fail to assign", e.getMessage());
                throw  new ClientRequestException(BrowserErrorCode.UNKNOWN);
            }
        });
    }

    @Override
    public ShopPageResultDto getShopList(String username, String groupId, int page, int amount, ShopFilterDto shopFilterDto) {
        if (amount > 100)
            amount = 100;
        Pageable pageable = PageRequest.of(page, amount);
        Account account = accountRepository.findByName(username);
        if (account == null) {
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST);
        }
        Group group = null;
        if (groupId != null)
            group = groupRepository.findById(groupId).orElse(null);
        if (group == null)
            throw new ClientRequestException(BrowserErrorCode.GROUPNOTEXIST);
        if (group != null && group.getOwner() != null && !group.getOwner().equals(username)) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION);
        }
        List<String> shopIds = new ArrayList<>();
        if (groupId.equals("-1")) {
            shopIds = userShopRepository.findByUsername(username).stream().
                    map(x -> x.getShopId()).collect(Collectors.toList());
        } else {
            shopIds = userShopRepository.findByUsernameAndGroupId(username, groupId).stream().
                    map(x -> x.getShopId()).collect(Collectors.toList());
        }
        Page<Shop> shops;
        if (shopFilterDto != null && StringUtils.isNotBlank(shopFilterDto.getIpRegion()))
            shops = shopRepository.findByShopIdInAndIpRegionLike(shopIds, shopFilterDto.getIpRegion(), pageable);
        else if (shopFilterDto != null && StringUtils.isNotBlank(shopFilterDto.getShopAccount()))
            shops = shopRepository.findByShopIdInAndShopAccountLike(shopIds, shopFilterDto.getShopAccount(), pageable);
        else if (shopFilterDto != null && StringUtils.isNotBlank(shopFilterDto.getShopName()))
            shops = shopRepository.findByShopIdInAndShopNameLike(shopIds, shopFilterDto.getShopName(), pageable);
        else
            shops = shopRepository.findByShopIdIn(shopIds, pageable);
        if (shops == null || shops.getNumberOfElements() < 1)
            return new ShopPageResultDto();
        List<ShopResultDto> shopResultDtos = new ArrayList<>();
        shops.getContent().stream().forEach(x -> {
            IpResource ipResource = ipResourceRepository.findByAddrAndIsDeleted(x.getIp(), false);
            if (ipResource == null)
                ipResource = new IpResource();
            String group1 = userShopRepository.findByUsernameAndShopId(username, x.getShopId()).getGroupId();
            shopResultDtos.add(ShopResultDto.of(x, group1, ipResource));
        });
        Page<ShopResultDto> shopDtoPage = new PageImpl<>(shopResultDtos, pageable, shopIds.size());
        ShopPageResultDto shopPageResultDto = new ShopPageResultDto();
        shopPageResultDto.setShopList(shopDtoPage.getContent());
        PageInfo pageInfo = new PageInfo();
        pageInfo.setCurrentPage(shopDtoPage.getPageable().getPageNumber());
        pageInfo.setTotalPages(shopDtoPage.getTotalPages());
        pageInfo.setTotalItems(shopIds.size());
        shopPageResultDto.setShopPage(pageInfo);
        return shopPageResultDto;
    }

    @Override
    public ShopSummary getShopSummary(String username) {
        ShopSummary shopSummary = new ShopSummary();
        int unbind = userShopRepository.countByUsernameAndIpIdIsNull(username);
        shopSummary.setUnbind(unbind);
        List<String> bind = userShopRepository.findByUsernameAndIpIdIsNotNull(username).stream().map(x -> x.getIpId()).collect(Collectors.toList());
        int expired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(1, bind, false);
        int willexpired = ipResourceRepository.countByStatusAndIdInAndIsDeleted(2, bind, false);
        shopSummary.setExpired(expired);
        shopSummary.setWillExpire(willexpired);
        shopSummary.setTotal(userShopRepository.countByUsername(username));
        return shopSummary;
    }
}
