package com.edgec.browserbackend.common.lock;

import com.edgec.browserbackend.common.commons.core.ApplicationContextProvider;
import org.springframework.data.mongodb.core.MongoTemplate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * MongoDB分布式锁工具类
 *
 * @author JMW
 */
public class MongoDistributedLock {

    static MongoLockRepository mongoLockRepository;
    private static final String VALUE = "value";

    static {
        mongoLockRepository = new MongoLockRepository(ApplicationContextProvider.getBean(MongoTemplate.class));
    }

    /**
     * 获得锁的步骤：
     * 1、首先判断锁是否被其他请求获得；如果没被其他请求获得则往下进行；
     * 2、判断锁资源是否过期，如果过期则释放锁资源；
     * 3.1、尝试获得锁资源，如果value=1，那么获得锁资源正常;（在当前请求已经获得锁的前提下，还可能有其他请求尝试去获得锁，此时会导致当前锁的过期时间被延长，由于延长时间在毫秒级，可以忽略。）
     * 3.2、value>1,则表示当前请求在尝试获取锁资源过程中，其他请求已经获取了锁资源，即当前请求没有获得锁；
     * ！！！注意，不需要锁资源时，及时释放锁资源！！！。
     *
     * @param lockKey 加锁字段
     * @param lockKeyValue 加锁字段
     * @param expire 加锁时间
     * @return
     */
    public static boolean getLock(MongoDistributedLockKey lockKey, String lockKeyValue, long expire) {
        String key = lockKey.getKey(lockKeyValue);
        List<MongoLock> mongoLocks = mongoLockRepository.getByKey(key);
        //判断该锁是否被获得,锁已经被其他请求获得，直接返回
        if (mongoLocks.size() > 0 && mongoLocks.get(0).getExpire() >= System.currentTimeMillis()) {
            return false;
        }
        //释放过期的锁
        if (mongoLocks.size() > 0 && mongoLocks.get(0).getExpire() < System.currentTimeMillis()) {
            releaseLockExpire(key, System.currentTimeMillis());
        }
        //！！(在高并发前提下)在当前请求已经获得锁的前提下，还可能有其他请求尝试去获得锁，此时会导致当前锁的过期时间被延长，由于延长时间在毫秒级，可以忽略。
        Map<String, Object> mapResult = mongoLockRepository.incrByWithExpire(key, 1, System.currentTimeMillis() + expire);
        //如果结果是1，代表当前请求获得锁
        if ((Integer) mapResult.get(VALUE) == 1) {
            return true;
            //如果结果>1，表示当前请求在获取锁的过程中，锁已被其他请求获得。
        } else if ((Integer) mapResult.get(VALUE) > 1) {
            return false;
        }
        return false;
    }

    /**
     * 释放锁
     *
     * @param lockKey 释放字段
     * @param lockKeyValue 释放字段
     */
    public static void releaseLock(MongoDistributedLockKey lockKey, String lockKeyValue) {
        String key = lockKey.getKey(lockKeyValue);
        Map<String, Object> condition = new HashMap<>(1);
        condition.put("key", key);
        mongoLockRepository.remove(condition);
    }

    /**
     * 释放过期锁
     *
     * @param key 释放的字段
     * @param expireTime 释放的时间限制
     */
    private static void releaseLockExpire(String key, long expireTime) {
        mongoLockRepository.removeExpire(key, expireTime);
    }
}
