package com.edgec.browserbackend.browser.task;

import com.edgec.browserbackend.account.domain.IpChargeRequestDto;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.browser.domain.*;
import com.edgec.browserbackend.browser.dto.*;
import com.edgec.browserbackend.browser.repository.IpResourceRepository;
import com.edgec.browserbackend.browser.service.IpAndShopService;
import com.edgec.browserbackend.common.utils.ThreadPoolUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.*;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

import static com.edgec.browserbackend.browser.service.Impl.IpResourceServiceImpl.genRandom;
import static com.edgec.browserbackend.browser.service.Impl.IpResourceServiceImpl.region;

@Component
public class BrowserTask {

    private static final Logger log = LoggerFactory.getLogger(BrowserTask.class);

    private static String CLOUDAMURL = "https://www.cloudam.cn";
    private static String TESTURL = "http://112.74.13.2";

    @Autowired
    private IpResourceRepository ipResourceRepository;

    @Autowired
    private AccountService accountService;

    @Autowired
    private IpAndShopService ipAndShopService;

    @Value("${spring.profiles.active}")
    private String profiles;

    public HttpHeaders buildGetHeader() {
        HttpHeaders headers = new HttpHeaders();
        headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON));
        if (profiles.equals("dev") || profiles.equals("staging"))
            headers.setBearerAuth("oq5tg3gMsflHcK5iZ2741G5R30XYd9blyOqH9qeBItKtrzfTsGIoy8AsxqqNXdcm");
        else if (profiles.equals("prod"))
            headers.setBearerAuth("tKWsuHzcngf0RQPMss70f9jgymDIwgQ9zbLfESJdcou3pZSNWl7lNTzto8VQgwaO");
        return headers;
    }

    public HttpHeaders buildPostHeader() {
        HttpHeaders header = new HttpHeaders();
        header.setContentType(MediaType.APPLICATION_JSON);
        if (profiles.equals("dev") || profiles.equals("staging")) {
            header.setBearerAuth("oq5tg3gMsflHcK5iZ2741G5R30XYd9blyOqH9qeBItKtrzfTsGIoy8AsxqqNXdcm");
        } else if (profiles.equals("prod")) {
            header.setBearerAuth("tKWsuHzcngf0RQPMss70f9jgymDIwgQ9zbLfESJdcou3pZSNWl7lNTzto8VQgwaO");
        }
        return header;
    }

    private IpChargeRequestDto buildIpChargeRequestDto(IpResource request, int chargeType, int payMethod) {
        IpChargeRequestDto ipChargeRequestDto = new IpChargeRequestDto();
        ipChargeRequestDto.setAmount(1);
        ipChargeRequestDto.setChargeType(chargeType);
        ipChargeRequestDto.setRegion(request.getRegion());
        ipChargeRequestDto.setPeriod(request.getPeriod());
        ipChargeRequestDto.setUnit(request.getUnit());
        ipChargeRequestDto.setPayMethod(0);
        return ipChargeRequestDto;
    }


    @Scheduled(cron = "0 0/1 * * * ?")
    public void buyIpTasks() {
        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        long time = Instant.now().minusSeconds(300).toEpochMilli();
        List<IpResource> ipResources = ipResourceRepository.sampleTasks(6, time);
        for (IpResource ipResource : ipResources) {
            long start = System.currentTimeMillis();
            CompletableFuture.runAsync(() -> {
                if (ipResourceRepository.lockTask(ipResource)) {
                    try {
                        boolean result = false;
                        RestTemplate restTemplate = new RestTemplate();
                        HttpHeaders header = buildPostHeader();
                        HashMap<String, Object> map = new HashMap<>();
                        map.put("name", ipResource.getUsername());
                        map.put("region", ipResource.getRegion());
                        int period = 0;
                        period = ipResource.getPeriod();
                        if (ipResource.getUnit().equals("month") && ipResource.getPeriod() == 6)
                            period = 7;
                        else if (ipResource.getUnit().equals("month") && ipResource.getPeriod() == 12)
                            period = 14;
                        map.put("period", String.valueOf(period));
                        map.put("provider", ipResource.getVendor());
                        map.put("unit", ipResource.getUnit());
                        map.put("amount", 1);
                        map.put("loginPassword", ipResource.getPassword());
                        map.put("startscript", "");
                        map.put("ipkeptperiod", 7);
                        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(map, header);
                        IpBuyResultDto ipBuyResultDto = null;
                        try {
                            ipBuyResultDto = restTemplate.postForObject(URL + "/intelligroup/ipresources?accountId=browser", httpEntity, IpBuyResultDto.class);
                            if (StringUtils.isNotBlank(ipBuyResultDto.getErrorCode())) {
                                log.error("fail to buy ip");
                                log.error(ipBuyResultDto.getErrorCode());
                            }
                            if (ipBuyResultDto != null && ipBuyResultDto.getIplist() != null && ipBuyResultDto.getIplist().size() >= 1) {
                                AtomicInteger index = new AtomicInteger();
                                ipBuyResultDto.getIplist().forEach(x -> {
                                    if (ipResource != null) {
                                        ipResource.setAddr(x.getIp());
                                        ipResource.setStatus(3);
                                        ipResource.setValidTime(Instant.parse(x.getValidTill()).toEpochMilli());
                                        if (StringUtils.isNotBlank(ipResource.getRegion()) && region.contains(ipResource.getRegion())) {
                                            ipResource.setProxyUsername(ipResource.getAddr());
                                            ipResource.setProxyPassword(genRandom(3, 12));
                                            ipResource.setSpecialLine(true);
                                        }
                                        ipResourceRepository.save(ipResource);
                                    } else {
                                        log.error("no ipResource");
                                    }
                                    index.getAndIncrement();
                                });
                                result = true;
                            }
                        } catch (Exception e) {
                            log.error(e.getMessage());
                            result = false;
                        }
                        if (result == false && (ipResource.getPurchasedTime() < Instant.now().minusSeconds(7200).toEpochMilli())) {
                            IpChargeRequestDto ipChargeRequestDto = buildIpChargeRequestDto(ipResource, 3, 0);
                            accountService.chargeByMoney(ipResource.getUsername(), -ipResource.getPrice(), ipChargeRequestDto);
                            if (ipResource.getShopId() != null) {
                                ShopRequestDto shopRequestDto = new ShopRequestDto();
                                shopRequestDto.setIpId(ipResource.getId());
                                shopRequestDto.setShopId(ipResource.getShopId());
                                ipAndShopService.unBindShop(ipResource.getUsername(), shopRequestDto);
                            }
                            ipResourceRepository.deleteById(ipResource.getId());
                        }
                    } finally {
                        long end = System.currentTimeMillis();
                        log.debug("buyIpTask {} execution time is: " + (end - start) / 1000 + "s", ipResource.getId());
                        try {
                            ipResourceRepository.unLockTask(ipResource.getId());
                        } catch (Throwable th) {
                            log.error("unlock failed", th);
                            //try again
                            ipResourceRepository.unLockTask(ipResource.getId());
                        }
                    }
                }
            }, ThreadPoolUtils.buyIpTasksPool);
        }
    }

    @Scheduled(cron = "0 0/1 * * * ?")
    public void queryIpTasks() {
        String URL = (profiles.equals("dev") || profiles.equals("staging")) ? TESTURL : CLOUDAMURL;
        long time = Instant.now().minusSeconds(300).toEpochMilli();
        List<IpResource> ipResources = ipResourceRepository.sampleTasks(3, time);
        for (IpResource ipResource : ipResources) {
            long start = System.currentTimeMillis();
            CompletableFuture.runAsync(() -> {
                if (ipResourceRepository.lockTask(ipResource)) {
                    try {
                        SimpleClientHttpRequestFactory requestFactory = new SimpleClientHttpRequestFactory();
                        requestFactory.setConnectTimeout(10000);// 设置超时
                        requestFactory.setReadTimeout(10000);
                        RestTemplate restTemplate = new RestTemplate(requestFactory);
                        Map<String, Object> params = new HashMap<>();
                        HttpHeaders headers = buildGetHeader();
                        HttpEntity<Map<String, Object>> entity = new HttpEntity<>(params, headers);
                        ResponseEntity<IpInfoResultDto> result = restTemplate.exchange(URL + "/ecc/ipinfo?accountId={accountId}&ip={ip}",
                                HttpMethod.GET, entity, IpInfoResultDto.class, "browser", ipResource.getAddr());
                        IpInfoResultDto ipInfoResultDto = result.getBody();
                        if (ipInfoResultDto != null && StringUtils.isBlank(ipInfoResultDto.getErrorCode())) {
                            if ( StringUtils.isNotEmpty(ipInfoResultDto.getStatus())) {
                                ipResource.setStatus(0);
                                ipResourceRepository.save(ipResource);
                            }
                        }
                    } catch (Exception e) {
                        log.error(e.getMessage(), e);
                    } finally {
                        long end = System.currentTimeMillis();
                        log.debug("queryIpTask {} execution time is: " + (end - start) / 1000 + "s", ipResource.getId());
                        try {
                            ipResourceRepository.unLockTask(ipResource.getId());
                        } catch (Throwable th) {
                            log.error("unlock failed", th);
                            //try again
                            ipResourceRepository.unLockTask(ipResource.getId());
                        }
                    }
                }
            }, ThreadPoolUtils.queryIpTasksPool);
        }
    }


}
