package com.edgec.browserbackend.account.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.request.AlipayFundTransToaccountTransferRequest;
import com.alipay.api.request.AlipayTradePagePayRequest;
import com.alipay.api.request.AlipayTradeQueryRequest;
import com.alipay.api.response.AlipayFundTransToaccountTransferResponse;
import com.alipay.api.response.AlipayTradePagePayResponse;
import com.alipay.api.response.AlipayTradeQueryResponse;
import com.edgec.browserbackend.account.controller.AccountController;
import com.edgec.browserbackend.account.domain.*;
import com.edgec.browserbackend.account.exception.AccountErrorCode;
import com.edgec.browserbackend.account.repository.*;
import com.edgec.browserbackend.account.service.AccountService;
import com.edgec.browserbackend.account.service.PaymentService;
import com.edgec.browserbackend.account.service.SmsUtils;
import com.edgec.browserbackend.alipay.AlipayConfig;
import com.edgec.browserbackend.alipay.CloudamAlipayConfig;
import com.edgec.browserbackend.alipay.VpsAlipayConfig;
import com.edgec.browserbackend.browser.ErrorCode.BrowserErrorCode;
import com.edgec.browserbackend.browser.domain.PayBack;
import com.edgec.browserbackend.browser.repository.PayBackRepository;
import com.edgec.browserbackend.common.commons.error.ClientRequestException;
import com.edgec.browserbackend.wxpay.*;
import com.google.gson.Gson;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.YearMonth;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

@Service
public class PaymentServiceImpl implements PaymentService {

    private final Logger log = LoggerFactory.getLogger(getClass());

    @Autowired
    private UserPaymentRepository userPaymentRepository;

    @Autowired
    private UserBalanceRepository userBalanceRepository;

    @Autowired
    private PayBackRepository payBackRepository;

    @Autowired
    private AccountService accountService;

    @Autowired
    private UserPrePaidBillingRepository userPrePaidBillingRepository;

    @Autowired
    private AccountRepository accountRepository;

    @Autowired
    private UserWithdrawRepository userWithdrawRepository;

    @Override
    public String wechatPayCallback(String tradno) {
        // 当微信回调该接口时，说明微信那边已经充值成功，这个时候需要更新系统中 userpayment 充值状态
        UserPayment byTradeNo = userPaymentRepository.findByTradeNo(tradno);
        if (byTradeNo != null && !byTradeNo.isSucceed()) {
            log.info("微信支付回调确认订单" + byTradeNo.getTradeNo() + "支付状态,此时chargeType为 0");
            log.error("wechat order：step-5 callback {}", byTradeNo.getTradeNo());
            wxCheckOrderStatus(byTradeNo.getTradeNo(), 0);
        }
        return "<xml>\n" +
                "\n" +
                "  <return_code><![CDATA[SUCCESS]]></return_code>\n" +
                "  <return_msg><![CDATA[OK]]></return_msg>\n" +
                "</xml>";
    }

    @Override
    public void alipaCallback(String tradno) {
        UserPayment byTradeNo = userPaymentRepository.findByTradeNo(tradno);
        if (byTradeNo != null && !byTradeNo.isSucceed()) {
            log.info("支付宝支付回调确认订单" + byTradeNo.getTradeNo() + "支付状态,此时chargeType为 0");
            log.error("alipay order：step-2 callback {}", tradno);
            aliCheckOrderStatus(byTradeNo.getTradeNo(), 0);
        }
    }

    @Override
    public UserPaymentDto wxCheckOrderStatus(String tradeno, int chargeType) {
        // 校验数据
        UserPaymentDto result = new UserPaymentDto();
        result.setPaid(false);
        if (StringUtils.isBlank(tradeno)) {
            return result;
        }

        UserPayment byTradeNo = userPaymentRepository.findByTradeNo(tradeno);
        if (byTradeNo == null) {
            return result;
        }
        if (byTradeNo.isSucceed()) {
            result.setPaid(true);
            return result;
        }
        Account byName = accountService.findByName(byTradeNo.getUsername());

        // 如果订单是用支付宝支付的，则调用支付宝相关的代码
        if (PaymentMethod.ALIPAY.equals(byTradeNo.getPaymentMethod())) {
            return aliCheckOrderStatus(tradeno, chargeType);
        }

        try {
            UserBalance balance = getUserBalance(result, byTradeNo);

            // 1. 构建微信支付订单查询数据，并获取微信支付订单信息
            //    获取微信支付订单存在失败的情况，所以针对失败情况去使用定时任务来重新调用本方法
            boolean isVpsClient = true;
            WXPayConfig ourWxPayConfig = isVpsClient ? new FGLWxConfig() : new CloudamWxConfig();
            WXPay wxPay = new WXPay(ourWxPayConfig);
            Map<String, String> data = encapsulateWxOrderDataMap(tradeno, ourWxPayConfig);
            Map<String, String> respData = wxPay.orderQuery(data);

            if (respData.get("return_code").equals("SUCCESS") && respData.get("return_msg").equals("OK") && "SUCCESS".equals(respData.get("result_code"))) {
                /*
                 * SUCCESS—支付成功
                 * REFUND—转入退款
                 * NOTPAY—未支付
                 * CLOSED—已关闭
                 * REVOKED—已撤销（付款码支付）
                 * USERPAYING--用户支付中（付款码支付）
                 * PAYERROR--支付失败(其他原因，如银行返回失败)
                 */
                result.setStatus(respData.get("trade_state"));
                // 2. 如果订单支付成功(具体业务逻辑)
                if ("SUCCESS".equals(respData.get("trade_state"))) {
                    dealPaySuccess(chargeType, result, byTradeNo, byName, balance);
                    return result;
                }
            }
            result.setPaid(false);
        } catch (Exception e) {
            log.error("Wechat payment order generation fails", e);
            result.setPaid(false);
        }
        return result;
    }

    private String convertAlipayStatus(String src) {
        switch (src) {
            case "WAIT_BUYER_PAY":
                return "USERPAYING";
            case "TRADE_CLOSED":
                return "CLOSED";
            case "TRADE_SUCCESS":
                return "SUCCESS";
            case "TRADE_FINISHED":
                return "SUCCESS";
        }
        return src;
    }

    @Override
    public UserPaymentDto aliCheckOrderStatus(String tradno, int chargeType) {
        UserPaymentDto result = new UserPaymentDto();
        result.setPaid(false);
        if (StringUtils.isBlank(tradno)) {
            return result;
        }

        UserPayment byTradeNo = userPaymentRepository.findByTradeNo(tradno);
        if (byTradeNo == null) {
            return result;
        }
        if (byTradeNo.isSucceed()) {
            result.setPaid(true);
            return result;
        }

        Account byName = accountService.findByName(byTradeNo.getUsername());

        try {
            UserBalance balance = getUserBalance(result, byTradeNo);

            boolean isVpsClient = true;
            AlipayConfig alipayConfig = new VpsAlipayConfig();
            AlipayClient alipayClient = new DefaultAlipayClient(alipayConfig.getURL(), alipayConfig.getAPPID(),
                    alipayConfig.getAPP_PRIVATE_KEY(), "json", alipayConfig.getCHARSET(),
                    alipayConfig.getALIPAY_PUBLIC_KEY(), alipayConfig.getSIGN_TYPE());
            AlipayTradeQueryRequest alipayRequest = buildAlipayTradeQueryRequest(byTradeNo);

            AlipayTradeQueryResponse response = alipayClient.execute(alipayRequest);

            log.info("ali order status {}, {}", JSONObject.toJSONString(response), tradno);
            if (response.getTradeStatus() == null) {
                result.setPaid(false);
                result.setStatus(convertAlipayStatus("WAIT_BUYER_PAY"));
                return result;
            }
            result.setStatus(convertAlipayStatus(response.getTradeStatus()));

            if ("TRADE_SUCCESS".equals(response.getTradeStatus()) || "TRADE_FINISHED".equals(response.getTradeStatus())) {
                // 2. 如果订单支付成功(具体业务逻辑)
                dealPaySuccess(chargeType, result, byTradeNo, byName, balance);
                return result;
            }

            result.setPaid(false);
        } catch (Exception e) {
            log.error("Alipay payment order generation fails {},{}", e, tradno);
            result.setPaid(false);
        }
        return result;
    }

    @NotNull
    private UserBalance getUserBalance(UserPaymentDto result, UserPayment byTradeNo) {
        UserBalance balance = userBalanceRepository.findById(byTradeNo.getUsername()).orElse(null);
        if (balance != null) {
            result.setBalance(Math.round(balance.getBalanced()));
        }
        if (balance == null) {
            balance = new UserBalance();
            balance.setUsername(byTradeNo.getUsername());
        }
        return balance;
    }

    @Override
    public String alipayPutPayOrder(String username, int amount, String by) {

        Account byName = accountService.findByName(username);
        if (byName.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION, "account does not have permission: " + username);
        }

        try {
            // 创建并封装 UserPayment 信息
            UserPayment internalOrder = getUserPayment(username, amount, PaymentMethod.ALIPAY);

            // 获取支付宝支付的配置信息
            boolean isVpsClient = true;
            AlipayConfig alipayConfig = isVpsClient ? new VpsAlipayConfig() : new CloudamAlipayConfig();
            // 获取支付宝的 client
            AlipayClient alipayClient = new DefaultAlipayClient(alipayConfig.getURL(), alipayConfig.getAPPID(),
                    alipayConfig.getAPP_PRIVATE_KEY(), "json", alipayConfig.getCHARSET(),
                    alipayConfig.getALIPAY_PUBLIC_KEY(), alipayConfig.getSIGN_TYPE());
            // 构建请求信息 并设置请求成功的回调 url
            AlipayTradePagePayRequest alipayRequest = buildAlipayTradePagePayRequest(amount, by, internalOrder, isVpsClient, alipayConfig);

            // 调用 支付宝支付接口
            AlipayTradePagePayResponse response = alipayClient.pageExecute(alipayRequest);

            // 将支付信息保存到 userPayment
            internalOrder.setTransId(response.getTradeNo());
            userPaymentRepository.save(internalOrder);
            log.error("alipay order：step-1 create pay order {}", internalOrder.getTradeNo());

            String form = response.getBody();
            return form;

        } catch (Exception e) {
            log.error("Alypay payment order generation fails", e);
            throw new ClientRequestException(AccountErrorCode.ALIPAYERROR, "Alypay payment order error");
        }
    }

    @Override
    public UserPaymentDto wxPutPayOrder(String username, int amount) {
        Account byName = accountService.findByName(username);
        if (byName.getPermission() < 4) {
            throw new ClientRequestException(AccountErrorCode.NOPERMISSION, "account does not have permission: " + username);
        }

        // 创建并封装 UserPayment 信息
        UserPayment internalOrder = getUserPayment(username, amount, PaymentMethod.WECHAT);
        log.error("wechat order：step-1 create order {}", internalOrder.getTradeNo());

        try {

            // 回调的url
            String notifyUrl = AccountController.WECHAT_PAY_CALLBACK_URL + internalOrder.getTradeNo();

            // 获取微信支付的配置信息
            boolean isVpsClient = true;
            WXPayConfig ourWxPayConfig = isVpsClient ? new FGLWxConfig() : new CloudamWxConfig();

            // 封装请求 微信支付 的数据
            Map<String, String> data = encapsulateDataMap(amount, internalOrder, notifyUrl, ourWxPayConfig);

            // 调用微信支付接口，支付成功后会回调 notifyUrl， 并返回支付结果
            WXPay wxPay = new WXPay(ourWxPayConfig);
            Map<String, String> respData = wxPay.unifiedOrder(data);

            if (respData.get("return_code").equals("SUCCESS") && respData.get("result_code").equals("SUCCESS")) {
                log.error("wechat order：step-2 pay status {},{}", respData.get("result_code"), internalOrder.getTradeNo());
                // 若微信支付成功，则将支付信息保存到 userPayment
                userPaymentRepository.save(internalOrder);
                log.error("wechat order：step-3.1 save pay info success {}", internalOrder.getTradeNo());
                UserPaymentDto result = new UserPaymentDto();
                result.setTradeNo(internalOrder.getTradeNo());
                result.setUrl(respData.get("code_url"));
                return result;
            } else {
                log.error("wechat order：step-3.2 Fail to create ex order {},{}" + JSONObject.toJSONString(respData), internalOrder.getTradeNo());
                throw new ClientRequestException(AccountErrorCode.WECHATERROR, "Wechat payment order error");
            }
        } catch (Exception e) {
            log.error("wechat order：step-4 Wechat payment order generation fails {}, {}", e, internalOrder.getTradeNo());
            throw new ClientRequestException(AccountErrorCode.WECHATERROR, "Wechat payment order error");
        }
    }

    @Override
    public UserBalance bankTransferInsertion(String username, int amount, int payMethod) {

        UserPayment bankOrder = new UserPayment();
        bankOrder.setUsername(username);
        bankOrder.setAmount(amount);
        bankOrder.setPaymentMethod(PaymentMethod.BANK);
        bankOrder.setSucceed(true);
        userPaymentRepository.save(bankOrder);

        UserBalance userBalance = userBalanceRepository.findById(username).orElse(null);

        UserPrePaidBilling bill = new UserPrePaidBilling();
        Account account = accountRepository.findByName(username).get();
        if (account != null && account.getParent() != null)
            bill.setAdministrator(account.getParent());
        else
            bill.setAdministrator(account.getName());
        bill.setTradeNo(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + SmsUtils.createRandom(true, 4));
        bill.setChargeType(0);
        bill.setAmount(0);
        bill.setUnit(null);
        bill.setPeriod(0);
        bill.setPayMethod(payMethod);
        bill.setUsername(username);
        bill.setTotal((float) amount);
        bill.setStatus(BillStatus.PAID);
        bill.setPrepaid(true);
        bill.setTimestamp(Instant.now().toEpochMilli());

        final YearMonth lastmonth = YearMonth.now();

        int monthValue = lastmonth.getMonthValue();
        int year = lastmonth.getYear();
        bill.setYear(year);
        bill.setMonth(monthValue);

        if (userBalance == null) {
            userBalance = new UserBalance();
            userBalance.setUsername(username);
            userBalance.setUsed(0);
            userBalance.setBalanced(amount);
            bill.setBalanced(amount);
            userPrePaidBillingRepository.save(bill);
            return userBalanceRepository.save(userBalance);
        } else {
            bill.setBalanced(userBalance.getBalanced() + amount);
            userPrePaidBillingRepository.save(bill);
            userBalanceRepository.incrementBalance(userBalance, amount, 0);
        }

        if (account.getPromotionCode() != null) {
            Account account1 = accountRepository.findByPromotion(account.getPromotionCode());
            if (account1 != null) {
                account1.getPromotion().setCommission(account1.getPromotion().getCommission() + amount);
                accountRepository.save(account1);
            }
        }

        return userBalanceRepository.findById(username).orElse(null);
    }


    @Override
    public UserPaymentDto h5wxPayOrder(String ip, String username, int amount) {

        Account byName = accountService.findByName(username);
        if (byName == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + username);

        boolean isVpsClient = true;

        UserPayment internalOrder = new UserPayment();
        internalOrder.setAmount(amount);
        internalOrder.setPaymentMethod(PaymentMethod.WECHAT);
        internalOrder.setSucceed(false);
        internalOrder.setUsername(username);

        try {
            String notifyUrl = AccountController.WECHAT_PAY_CALLBACK_URL + internalOrder.getTradeNo();
//            String redirectUrl = "https://www.cloudam.cn";
//            String redirectUrlEncode = URLEncoder.encode(redirectUrl, "utf-8");

            WXPayConfig ourWxPayConfig = isVpsClient ? new FGLWxConfig() : new CloudamWxConfig();
            WXPay wxPay = new WXPay(ourWxPayConfig);

            Map<String, String> data = new HashMap<>();
            data.put("appid", ourWxPayConfig.getAppID());
            data.put("mch_id", ourWxPayConfig.getMchID());         //商户号
            data.put("body", ourWxPayConfig.getBody());
            data.put("trade_type", "MWEB");
            data.put("notify_url", notifyUrl);                     //回调地址
            data.put("spbill_create_ip", ip);             //终端ip
            data.put("total_fee", "" + amount * 100);       //订单总金额
            data.put("fee_type", "CNY");                           //默认人民币
            data.put("out_trade_no", internalOrder.getTradeNo());   //交易号
            data.put("nonce_str", internalOrder.getTradeNo());   // 随机字符串小于32位

            JSONObject h5info = new JSONObject();
            h5info.put("type", "wap");
            h5info.put("wap_url", ourWxPayConfig.getWebUrl());
            h5info.put("wap_name", ourWxPayConfig.getBody());
            JSONObject scene_info = new JSONObject();
            scene_info.put("h5_info", h5info);
            data.put("scene_info", scene_info.toJSONString());
            String s = WXPayUtil.generateSignature(data, ourWxPayConfig.getKey());  //签名
            data.put("sign", s);

            Map<String, String> respData = wxPay.unifiedOrder(data);
            if (respData.get("return_code").equals("SUCCESS") && respData.get("result_code").equals("SUCCESS")) {
                userPaymentRepository.save(internalOrder);
                UserPaymentDto result = new UserPaymentDto();
                result.setTradeNo(internalOrder.getTradeNo());
                result.setUrl(respData.get("mweb_url"));
                return result;
            } else
                throw new ClientRequestException(AccountErrorCode.WECHATERROR, "Wechat payment order error");
        } catch (Exception e) {
            log.error("Wechat payment order generation fails", e);
            throw new ClientRequestException(AccountErrorCode.WECHATERROR, "Wechat payment order error");
        }
    }

    @Override
    public boolean alipayWithdraw(String username, String account, String realName, double amount) {
        java.text.DecimalFormat df = new java.text.DecimalFormat("#.00");
        Account byName = accountService.findByName(username);
        if (byName == null)
            throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + username);

        boolean isVpsClient = true;

        if (amount > byName.getPromotion().getAllGift())
            throw new ClientRequestException(AccountErrorCode.NOTENOUGHBALANCE);

        UserWithdraw internalOrder = new UserWithdraw();
        internalOrder.setAmount(byName.getPromotion().getCommission());
        internalOrder.setWithdrawMethod(PaymentMethod.ALIPAY);
        internalOrder.setSucceed(false);
        internalOrder.setUsername(username);

        try {

            byName.getPromotion().setAllGift(byName.getPromotion().getAllGift() - amount);
            byName.getPromotion().setWithdrawn(byName.getPromotion().getWithdrawn() + amount);
            accountRepository.save(byName);

            AlipayConfig alipayConfig = isVpsClient ? new VpsAlipayConfig() : new CloudamAlipayConfig();
            DefaultAlipayClient alipayClient = new DefaultAlipayClient(alipayConfig.getURL(), alipayConfig.getAPPID(), alipayConfig.getAPP_PRIVATE_KEY(),
                    "json", alipayConfig.getCHARSET(), alipayConfig.getALIPAY_PUBLIC_KEY(), alipayConfig.getSIGN_TYPE());

            AlipayFundTransToaccountTransferRequest alipayRequest = new AlipayFundTransToaccountTransferRequest();

            String out_biz_no = internalOrder.getTradeNo();
            String trans_amount = "" + df.format(amount);
            String identity_type = "ALIPAY_LOGONID";
            String subject = "礼金提现" + out_biz_no;

            AliTransfer aliTransfer = new AliTransfer();
            aliTransfer.setOut_biz_no(internalOrder.getTradeNo());
            aliTransfer.setPayee_type(identity_type);
            aliTransfer.setAmount(trans_amount);
            aliTransfer.setPayer_show_name("防关联浏览器");
            aliTransfer.setPayee_account(account);
            aliTransfer.setPayee_real_name(realName);
            aliTransfer.setRemark(subject);

            String json = new Gson().toJson(aliTransfer);
            alipayRequest.setBizContent(json);

            AlipayFundTransToaccountTransferResponse response = alipayClient.execute(alipayRequest);

            if (response == null)
                throw new ClientRequestException(BrowserErrorCode.UNKNOWN);

            if (response.isSuccess()) {

                internalOrder.setSucceed(true);
                userWithdrawRepository.save(internalOrder);
                UserPrePaidBilling bill = new UserPrePaidBilling();
                if (byName != null && byName.getParent() != null)
                    bill.setAdministrator(byName.getParent());
                else
                    bill.setAdministrator(byName.getName());
                bill.setTradeNo(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + SmsUtils.createRandom(true, 4));
                bill.setChargeType(4);
                bill.setAmount(0);
                bill.setUnit(null);
                bill.setPeriod(0);
                bill.setPayMethod(1);
                bill.setUsername(username);
                bill.setTotal((float) amount);
                bill.setStatus(BillStatus.PAID);
                bill.setPrepaid(true);
                bill.setTimestamp(Instant.now().toEpochMilli());

                final YearMonth lastmonth = YearMonth.now();

                int monthValue = lastmonth.getMonthValue();
                int year = lastmonth.getYear();
                bill.setYear(year);
                bill.setMonth(monthValue);

                userPrePaidBillingRepository.save(bill);
                return true;
            } else {
                byName.getPromotion().setAllGift(byName.getPromotion().getAllGift() + amount);
                byName.getPromotion().setWithdrawn(byName.getPromotion().getWithdrawn() - amount);
                accountRepository.save(byName);
                internalOrder.setSucceed(false);
                userWithdrawRepository.save(internalOrder);
                log.info(response.getBody());
                return false;
            }
        } catch (Exception e) {
            log.error("Alipay payment order generation fails", e);
            throw new ClientRequestException(AccountErrorCode.ALIPAYERROR, "Alipay payment order error");
        }
    }


    @Deprecated
    @Override
    public UserPaymentDto wxCheckOrderStatus(String tradeno) {
        UserPaymentDto result = new UserPaymentDto();
        result.setPaid(false);

        UserPayment byTradeNo = userPaymentRepository.findByTradeNo(tradeno);
        if (byTradeNo == null)
            return result;
        if (byTradeNo.isSucceed()) {
            result.setPaid(true);
            return result;
        }

        Account byName = accountService.findByName(byTradeNo.getUsername());
        if (PaymentMethod.ALIPAY.equals(byTradeNo.getPaymentMethod())) {
            return aliCheckOrderStatus(tradeno);
        }

        try {
            UserBalance balance = userBalanceRepository.findById(byTradeNo.getUsername()).orElse(null);

            if (balance != null) {
                result.setBalance(Math.round(balance.getBalanced()));
            }

            boolean isVpsClient = true;
            WXPayConfig ourWxPayConfig = isVpsClient ? new FGLWxConfig() : new CloudamWxConfig();
            WXPay wxPay = new WXPay(ourWxPayConfig);
            Map<String, String> data = encapsulateWxOrderDataMap(tradeno, ourWxPayConfig);
            Map<String, String> respData = wxPay.orderQuery(data);

            if (respData.get("return_code").equals("SUCCESS") && respData.get("return_msg").equals("OK") && "SUCCESS".equals(respData.get("result_code"))) {
                /*
                 * SUCCESS—支付成功
                 * REFUND—转入退款
                 * NOTPAY—未支付
                 * CLOSED—已关闭
                 * REVOKED—已撤销（付款码支付）
                 * USERPAYING--用户支付中（付款码支付）
                 * PAYERROR--支付失败(其他原因，如银行返回失败)
                 */
                result.setStatus(respData.get("trade_state"));
                if ("SUCCESS".equals(respData.get("trade_state"))) {
                    byTradeNo.setSucceed(true);
                    boolean b = userPaymentRepository.updatePaymentResult(byTradeNo, true, new Date());

                    if (b) {
                        if (balance == null) {
                            balance = new UserBalance();
                            balance.setUsername(byTradeNo.getUsername());
                        }
                        // 充值返现
                        PayBack payBack = payBackRepository.findByPay(byTradeNo.getAmount());
                        if (payBack == null) {
                            payBack = new PayBack();
                        }

                        balance.setBalanced(balance.getBalanced() + byTradeNo.getAmount() + payBack.getBack());
                        balance = userBalanceRepository.save(balance);
                    }

                    result.setBalance(Math.round(balance.getBalanced()));
                    result.setPaid(true);
                    return result;
                }
            }
            result.setPaid(false);
        } catch (Exception e) {
            log.error("Wechat payment order generation fails{},{}", e, tradeno);
            result.setPaid(false);
        }
        return result;
    }

    @Deprecated
    @Override
    public UserPaymentDto aliCheckOrderStatus(String tradno) {
        UserPaymentDto result = new UserPaymentDto();
        result.setPaid(false);
        try {

            UserPayment byTradeNo = userPaymentRepository.findByTradeNo(tradno);
            UserBalance balance = userBalanceRepository.findById(byTradeNo.getUsername()).orElse(null);

            if (balance != null)
                result.setBalance(Math.round(balance.getBalanced()));

            if (byTradeNo == null)
                return result;
            if (byTradeNo.isSucceed()) {
                result.setPaid(true);
                return result;
            }

            Account byName = accountService.findByName(byTradeNo.getUsername());
            if (byName == null)
                throw new ClientRequestException(AccountErrorCode.NAMENOTEXIST, "account does not exist: " + byTradeNo.getUsername());

            boolean isVpsClient = true;

            AlipayConfig alipayConfig = new VpsAlipayConfig();
            AlipayClient alipayClient = new DefaultAlipayClient(alipayConfig.getURL(), alipayConfig.getAPPID(), alipayConfig.getAPP_PRIVATE_KEY(),
                    "json", alipayConfig.getCHARSET(), alipayConfig.getALIPAY_PUBLIC_KEY(), alipayConfig.getSIGN_TYPE());

            AlipayTradeQueryRequest alipayRequest = buildAlipayTradeQueryRequest(byTradeNo);
            AlipayTradeQueryResponse response = alipayClient.execute(alipayRequest);
            log.info("ali order status :" + JSONObject.toJSONString(response));
            result.setStatus(convertAlipayStatus(response.getTradeStatus()));

            if ("TRADE_SUCCESS".equals(response.getTradeStatus()) || "TRADE_FINISHED".equals(response.getTradeStatus())) {
                byTradeNo.setSucceed(true);
                boolean b = userPaymentRepository.updatePaymentResult(byTradeNo, true, new Date());

                if (b) {

                    if (balance == null) {
                        balance = new UserBalance();
                        balance.setUsername(byTradeNo.getUsername());
                    }
                    // 充值返现
                    PayBack payBack = payBackRepository.findByPay(byTradeNo.getAmount());
                    if (payBack == null)
                        payBack = new PayBack();
                    balance.setBalanced(balance.getBalanced() + byTradeNo.getAmount() + payBack.getBack());
                    balance = userBalanceRepository.save(balance);
                }

                result.setBalance(Math.round(balance.getBalanced()));
                result.setPaid(true);
                return result;
            }

            result.setPaid(false);
        } catch (Exception e) {
            log.error("Alipay payment order generation fails {},{}", e, tradno);
            result.setPaid(false);
        }
        return result;
    }

    @NotNull
    private UserPayment getUserPayment(String username, int amount, PaymentMethod paymentMethod) {
        UserPayment internalOrder = new UserPayment();
        internalOrder.setUsername(username);
        internalOrder.setAmount(amount);
        internalOrder.setPaymentMethod(paymentMethod);
        // 此时充值尚未成功，需要等 微信或者支付宝 回调才能确认是否成功
        internalOrder.setSucceed(false);
        return internalOrder;
    }

    @NotNull
    private Map<String, String> encapsulateDataMap(int amount, UserPayment internalOrder, String notifyUrl, WXPayConfig ourWxPayConfig) throws Exception {
        Map<String, String> data = new HashMap<>();
        data.put("appid", ourWxPayConfig.getAppID());
        data.put("mch_id", ourWxPayConfig.getMchID());         //商户号
        data.put("body", ourWxPayConfig.getBody());
        data.put("trade_type", "NATIVE");                         //NATIVE 扫码支付
        data.put("notify_url", notifyUrl);                     //回调地址
        data.put("spbill_create_ip", "127.0.0.1");             //终端ip
        data.put("total_fee", "" + amount * 100);       //订单总金额
        data.put("fee_type", "CNY");                           //默认人民币
        data.put("out_trade_no", internalOrder.getTradeNo());   //交易号
        data.put("nonce_str", internalOrder.getTradeNo());   // 随机字符串小于32位
        String s = WXPayUtil.generateSignature(data, ourWxPayConfig.getKey());  //签名
        data.put("sign", s);
        return data;
    }

    @NotNull
    private Map<String, String> encapsulateWxOrderDataMap(String tradeno, WXPayConfig ourWxPayConfig) throws Exception {
        Map<String, String> data = new HashMap<>();
        data.put("appid", ourWxPayConfig.getAppID());
        data.put("mch_id", ourWxPayConfig.getMchID());         //商户号
        data.put("out_trade_no", tradeno);   //交易号
        data.put("nonce_str", SmsUtils.createRandom(false, 24));   // 随机字符串小于32位
        String s = WXPayUtil.generateSignature(data, ourWxPayConfig.getKey());  //签名
        data.put("sign", s);
        return data;
    }

    @NotNull
    private UserPrePaidBilling getUserPrePaidBilling(UserPayment byTradeNo, Account byName, UserBalance balance, PayBack payBack) {
        UserPrePaidBilling bill = new UserPrePaidBilling();
        if (byName != null && byName.getParent() != null) {
            bill.setAdministrator(byName.getParent());
        } else {
            bill.setAdministrator(byName.getName());
        }

        bill.setTradeNo(new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + SmsUtils.createRandom(true, 4));
        bill.setChargeType(0);
        bill.setAmount(0);
        bill.setUnit(null);
        bill.setPeriod(0);

        if (byTradeNo.getPaymentMethod().equals(PaymentMethod.ALIPAY)) {
            bill.setPayMethod(1);
        }
        if (byTradeNo.getPaymentMethod().equals(PaymentMethod.WECHAT)) {
            bill.setPayMethod(2);
        }

        bill.setUsername(byTradeNo.getUsername());
        bill.setTotal((float) byTradeNo.getAmount());
        bill.setStatus(BillStatus.PAID);
        bill.setPrepaid(true);
        bill.setTimestamp(Instant.now().toEpochMilli());
        // 设置到账金额为 账户余额 + 充值金额 + 充值优惠赠送
        bill.setBalanced(balance.getBalanced() + byTradeNo.getAmount() + payBack.getBack());
        final YearMonth lastmonth = YearMonth.now();
        int monthValue = lastmonth.getMonthValue();
        int year = lastmonth.getYear();
        bill.setYear(year);
        bill.setMonth(monthValue);
        return bill;
    }

    @NotNull
    private AlipayTradePagePayRequest buildAlipayTradePagePayRequest(int amount, String by, UserPayment internalOrder, boolean isVpsClient, AlipayConfig alipayConfig) {
        AlipayTradePagePayRequest alipayRequest = new AlipayTradePagePayRequest();
        alipayRequest.setReturnUrl(alipayConfig.getReturnUrl(isVpsClient, by) + internalOrder.getTradeNo());
        // 调用支付宝支付接口，支付成功后会回调 notifyUrl
        alipayRequest.setNotifyUrl(alipayConfig.getNOTIFY_URL() + internalOrder.getTradeNo());

        String out_trade_no = internalOrder.getTradeNo();
        String product_code = "FAST_INSTANT_TRADE_PAY";
        String total_amount = "" + amount + ".00";
        String subject = "订单" + out_trade_no;
        String body = "";
        alipayRequest.setBizContent(""
                + "{"
                + "\"out_trade_no\":\"" + out_trade_no + "\","
                + "\"product_code\":\"" + product_code + "\","
                + "\"total_amount\":\"" + total_amount + "\","
                + "\"subject\":\"" + subject + "\","
                + "\"body\":\"" + body + "\""
                + "}"
        );
        return alipayRequest;
    }

    @NotNull
    private AlipayTradeQueryRequest buildAlipayTradeQueryRequest(UserPayment byTradeNo) {
        AlipayTradeQueryRequest alipayRequest = new AlipayTradeQueryRequest();

        String out_trade_no = byTradeNo.getTradeNo();
        alipayRequest.setBizContent(""
                + "{"
                + "\"out_trade_no\":\"" + out_trade_no + "\""
                + "}"
        );
        return alipayRequest;
    }

    private void dealPaySuccess(int chargeType, UserPaymentDto result, UserPayment byTradeNo, Account byName, UserBalance balance) {
        byTradeNo.setSucceed(true);
        // 2.1 更新 userPayment 的支付状态为成功
        boolean b = userPaymentRepository.updatePaymentResult(byTradeNo, true, new Date());
        log.error("wechat or ali pay success：step-1 update order status {},{}", b, byTradeNo.getTradeNo());

        if (b) {
            log.info("订单 " + byTradeNo.getTradeNo() + " 的支付状态更新为成功");
            // 2.2 获取充值优惠赠送
            PayBack payBack = payBackRepository.findByPay(byTradeNo.getAmount());
            if (payBack == null) {
                payBack = new PayBack();
            }

            // chargeType 为 0 代表 充值
            if (chargeType == 0) {
                // 2.3 构建用户充值信息并保存
                UserPrePaidBilling bill = getUserPrePaidBilling(byTradeNo, byName, balance, payBack);
                userPrePaidBillingRepository.save(bill);
                log.info("构建订单 " + byTradeNo.getTradeNo() + " 的充值信息成功");
                log.error("wechat or ali pay success：step-2 create paidbilling info {},{}", bill.getId(), byTradeNo.getTradeNo());
            }

            // 2.4 更新账户余额 userBalance
            balance.setBalanced(balance.getBalanced() + byTradeNo.getAmount() + payBack.getBack());
            userBalanceRepository.save(balance);
            log.info("更新充值订单 " + byTradeNo.getTradeNo() + " 所对应的账户 " + balance.getUsername() + " 的余额成功");
            log.error("wechat or ali pay success：step-3 create balance info {},{}", balance.getUsername(), byTradeNo.getTradeNo());

            // 2.5 如果当前用户通过邀请码注册的，则更新邀请人的佣金
            if (byName.getPromotionCode() != null) {
                Account account = accountRepository.findByPromotion(byName.getPromotionCode());
                if (account != null) {
                    account.getPromotion().setCommission(account.getPromotion().getCommission() + byTradeNo.getAmount());
                    accountRepository.save(account);
                    log.info("更新邀请人的佣金成功");
                }
            }
        }

        result.setBalance(Math.round(balance.getBalanced()));
        result.setPaid(true);
    }
}
